/**
 * This file is a part of VideoCut package.
 * ---------------------------------------------------------------------- 
 * Copyright (C) 2007-2008 troorl
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ---------------------------------------------------------------------- 
 *
 * @author troorl <troorl@gmail.com>
 */

#include "TConfigdialog.h"

TConfigDialog::TConfigDialog()
{
    // List of pages
    contentsWidget = new QListWidget;
	contentsWidget->setFlow(QListView::LeftToRight);
	contentsWidget->setViewMode(QListView::IconMode);
    contentsWidget->setIconSize(QSize(32, 32));
	contentsWidget->setMaximumHeight(64);

    // Pages of options
    pagesWidget = new QStackedWidget;
    page1 = new TMainPage(this);
    page2 = new TFontsPage(this);
	page3 = new TFileInfoPage(this);
	page4 = new TColorPage(this);
    pagesWidget->addWidget(page1);
    pagesWidget->addWidget(page2);
	pagesWidget->addWidget(page3);
	pagesWidget->addWidget(page4);

    // Bottom buttons
    closeButton = new QPushButton(tr("Close"));
    applyButton = new QPushButton(tr("Apply"));
    okButton = new QPushButton(tr("Ok"));

    createIcons();
    contentsWidget->setCurrentRow(0);

    connect(closeButton, SIGNAL(clicked()), this, SLOT(close()));

    QVBoxLayout *verticalLayout = new QVBoxLayout;
    verticalLayout->addWidget(contentsWidget);
	verticalLayout->addWidget(pagesWidget);

    QHBoxLayout *buttonsLayout = new QHBoxLayout;
    buttonsLayout->addStretch(1);
    buttonsLayout->addWidget(okButton);
    buttonsLayout->addWidget(applyButton);
    buttonsLayout->addWidget(closeButton);
	buttonsLayout->setContentsMargins(0, 0, 0, 11);
	verticalLayout->addLayout(buttonsLayout);
	
    QHBoxLayout *mainLayout = new QHBoxLayout;
	mainLayout->addLayout(verticalLayout);
    mainLayout->addSpacing(0);
	mainLayout->setContentsMargins(0, 0, 0, 0);
	mainLayout->setSizeConstraint(QLayout::SetMinimumSize);
	
    setLayout(mainLayout);

    setWindowTitle(tr("Config Dialog"));
    this->setModal(true);

    // Read options
    settings = new TUserSettings();

    this->ReadSettings();

    connect(applyButton, SIGNAL(clicked()), this, SLOT(S_WriteSettings()));
    connect(okButton, SIGNAL(clicked()), this, SLOT(S_WriteSettingsOk()));
}

void TConfigDialog::createIcons()
{
	QListWidgetItem *configButton = new QListWidgetItem(contentsWidget);
	configButton->setIcon(QIcon(":/img/options.png"));
	configButton->setText(tr("Main options"));
	configButton->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);

	QListWidgetItem *updateButton = new QListWidgetItem(contentsWidget);
	updateButton->setIcon(QIcon(":/img/fonts.png"));
	updateButton->setText(tr("Fonts"));
	updateButton->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);
	
	QListWidgetItem *fileInfoButton = new QListWidgetItem(contentsWidget);
	fileInfoButton->setIcon(QIcon(":/img/about.png"));
	fileInfoButton->setText(tr("File information"));
	fileInfoButton->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);
	
	QListWidgetItem *fileColorButton = new QListWidgetItem(contentsWidget);
	fileColorButton->setIcon(QIcon(":/img/colors.png"));
	fileColorButton->setText(tr("Colors"));
	fileColorButton->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);

	connect(contentsWidget,
			SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
				   this, SLOT(changePage(QListWidgetItem *, QListWidgetItem*)));
}

void TConfigDialog::ReadSettings()
{
    settings->sync();
    settings->readSettings();
	
	page1->countFrames->setValue(settings->getCountFrames());
	page1->countHor->setValue(settings->getCountHor());
	page1->indentBorder->setValue(settings->getIndentBorder());
	page1->indentItems->setValue(settings->getIndentItems());
	page1->scale->setValue((int)settings->getScale());
	page1->quality->setValue(settings->getQuality());
	page1->formatImg->setCurrentIndex(settings->getFormatImg());
	page1->logo->setChecked(settings->getLogo());
	page1->styleBox->setCurrentIndex(settings->getStyle());
	
	page1->indentTimeStart->setTime(settings->getIndentTimeStart());
	page1->indentTimeFinish->setTime(settings->getIndentTimeFinish());
	
    page2->f1->setFont(settings->getFontMetaInfo());
    page2->f2->setFont(settings->getFontTimeInfo());
	
	page3->infoName->setChecked(settings->getInfoName());
	page3->infoSize->setChecked(settings->getInfoSize());
	page3->infoDuration->setChecked(settings->getInfoDuration());
	page3->infoResolution->setChecked(settings->getInfoResolution());
	page3->infoVideo->setChecked(settings->getInfoVideo());
	page3->infoAudio->setChecked(settings->getInfoAudio());
	
	page4->colorInfoText->setColor(settings->getColorInfoText());
	page4->colorInfoInfo->setColor(settings->getColorInfoInfo());
	page4->colorBackground->setColor(settings->getColorBackground());
}

void TConfigDialog::S_WriteSettings() // write settings to the config-file
{
    settings->sync();
	
    // write settings to the structure
	settings->setCountFrames(page1->countFrames->value());
	settings->setCountHor(page1->countHor->value());
	settings->setIndentBorder(page1->indentBorder->value());
	settings->setIndentItems(page1->indentItems->value());
	settings->setScale(page1->scale->value());
	settings->setQuality(page1->quality->value());
	settings->setFormatImg(page1->formatImg->currentIndex());
	settings->setLogo(page1->logo->isChecked());
	settings->setStyle(page1->styleBox->currentIndex());
	
	settings->setIndentTimeStart(page1->indentTimeStart->time());
	settings->setIndentTimeFinish(page1->indentTimeFinish->time());

    settings->setFontMetaInfo(page2->f1->getFont());
    settings->setFontTimeInfo(page2->f2->getFont());
	
	settings->setInfoName(page3->infoName->isChecked());
	settings->setInfoSize(page3->infoSize->isChecked());
	settings->setInfoDuration(page3->infoDuration->isChecked());
	settings->setInfoResolution(page3->infoResolution->isChecked());
	settings->setInfoVideo(page3->infoVideo->isChecked());
	settings->setInfoAudio(page3->infoAudio->isChecked());
	
	settings->setColorInfoText(page4->colorInfoText->getColor());
	settings->setColorInfoInfo(page4->colorInfoInfo->getColor());
	settings->setColorBackground(page4->colorBackground->getColor());

    settings->writeSettings(); // write settings to the config

    emit applySettings(); // generating signal about new settings
}

void TConfigDialog::S_WriteSettingsOk()
{
    S_WriteSettings();
    close();
}

void TConfigDialog::changePage(QListWidgetItem *current, QListWidgetItem *previous)
{
    if (!current)
        current = previous;

    pagesWidget->setCurrentIndex(contentsWidget->row(current));
}
