# $Id: user_mood.tcl 1447 2008-06-04 19:29:26Z sergei $
# Implementation of XEP-0107 "User mood"
# Based on Version 1.1 (2007-06-04).

namespace eval mood {
    variable node http://jabber.org/protocol/mood
    variable substatus
    variable mood

    variable options

    custom::defvar options(auto-subscribe) 0 \
	[::msgcat::mc "Auto-subscribe to other's user mood notifications."] \
	-command [namespace current]::register_in_disco \
	-group PEP -type boolean

    variable m2d
    variable d2m

    array set m2d [list \
	afraid       [::msgcat::mc "afraid"] \
	amazed       [::msgcat::mc "amazed"] \
	angry        [::msgcat::mc "angry"] \
	annoyed      [::msgcat::mc "annoyed"] \
	anxious      [::msgcat::mc "anxious"] \
	aroused      [::msgcat::mc "aroused"] \
	ashamed      [::msgcat::mc "ashamed"] \
	bored        [::msgcat::mc "bored"] \
	brave        [::msgcat::mc "brave"] \
	calm         [::msgcat::mc "calm"] \
	cold         [::msgcat::mc "cold"] \
	confused     [::msgcat::mc "confused"] \
	contented    [::msgcat::mc "contented"] \
	cranky       [::msgcat::mc "cranky"] \
	curious      [::msgcat::mc "curious"] \
	depressed    [::msgcat::mc "depressed"] \
	disappointed [::msgcat::mc "disappointed"] \
	disgusted    [::msgcat::mc "disgusted"] \
	distracted   [::msgcat::mc "distracted"] \
	embarrassed  [::msgcat::mc "embarrassed"] \
	excited      [::msgcat::mc "excited"] \
	flirtatious  [::msgcat::mc "flirtatious"] \
	frustrated   [::msgcat::mc "frustrated"] \
	grumpy       [::msgcat::mc "grumpy"] \
	guilty       [::msgcat::mc "guilty"] \
	happy        [::msgcat::mc "happy"] \
	hot          [::msgcat::mc "hot"] \
	humbled      [::msgcat::mc "humbled"] \
	humiliated   [::msgcat::mc "humiliated"] \
	hungry       [::msgcat::mc "hungry"] \
	hurt         [::msgcat::mc "hurt"] \
	impressed    [::msgcat::mc "impressed"] \
	in_awe       [::msgcat::mc "in_awe"] \
	in_love      [::msgcat::mc "in_love"] \
	indignant    [::msgcat::mc "indignant"] \
	interested   [::msgcat::mc "interested"] \
	intoxicated  [::msgcat::mc "intoxicated"] \
	invincible   [::msgcat::mc "invincible"] \
	jealous      [::msgcat::mc "jealous"] \
	lonely       [::msgcat::mc "lonely"] \
	mean         [::msgcat::mc "mean"] \
	moody        [::msgcat::mc "moody"] \
	nervous      [::msgcat::mc "nervous"] \
	neutral      [::msgcat::mc "neutral"] \
	offended     [::msgcat::mc "offended"] \
	playful      [::msgcat::mc "playful"] \
	proud        [::msgcat::mc "proud"] \
	relieved     [::msgcat::mc "relieved"] \
	remorseful   [::msgcat::mc "remorseful"] \
	restless     [::msgcat::mc "restless"] \
	sad          [::msgcat::mc "sad"] \
	sarcastic    [::msgcat::mc "sarcastic"] \
	serious      [::msgcat::mc "serious"] \
	shocked      [::msgcat::mc "shocked"] \
	shy          [::msgcat::mc "shy"] \
	sick         [::msgcat::mc "sick"] \
	sleepy       [::msgcat::mc "sleepy"] \
	stressed     [::msgcat::mc "stressed"] \
	surprised    [::msgcat::mc "surprised"] \
	thirsty      [::msgcat::mc "thirsty"] \
	worried      [::msgcat::mc "worried"] \
    ]
    foreach m [array names m2d] {
	set d2m($m2d($m)) $m
    }
    unset m

    pubsub::register_event_notification_handler $node \
	    [namespace current]::process_mood_notification
    hook::add user_mood_notification_hook \
	    [namespace current]::notify_via_status_message

    hook::add finload_hook \
	    [namespace current]::on_init 60
    hook::add connected_hook \
	    [namespace current]::on_connect_disconnect
    hook::add disconnected_hook \
	    [namespace current]::on_connect_disconnect
    hook::add roster_jid_popup_menu_hook \
	    [namespace current]::add_roster_pep_menu_item
    hook::add roster_user_popup_info_hook \
	    [namespace current]::provide_roster_popup_info
    hook::add userinfo_hook \
	    [namespace current]::provide_userinfo

    disco::register_feature $node
}

proc mood::register_in_disco {args} {
    variable options
    variable node

    if {$options(auto-subscribe)} {
	disco::register_feature $node+notify
    } else {
	disco::unregister_feature $node+notify
    }
}

proc mood::add_roster_pep_menu_item {m connid jid} {
    set rjid [roster::find_jid $connid $jid]

    if {$rjid == ""} {
 	set rjid [node_and_server_from_jid $jid]
    }

    set pm [pep::get_roster_menu_pep_submenu $m $connid $rjid]

    set mm [menu $pm.mood -tearoff no]
    $pm add cascade -menu $mm \
	    -label [::msgcat::mc "User mood"]

    $mm add command \
	    -label [::msgcat::mc "Subscribe"] \
	    -command [list [namespace current]::subscribe $connid $rjid]
    $mm add command \
	    -label [::msgcat::mc "Unsubscribe"] \
	    -command [list [namespace current]::unsubscribe $connid $rjid]

    hook::run roster_pep_user_mood_menu_hook $mm $connid $rjid
}

proc mood::subscribe {connid jid args} {
    variable node
    variable substatus

    set to [node_and_server_from_jid $jid]
    set cmd [linsert $args 0 [namespace current]::subscribe_result $connid $to]
    pep::subscribe $to $node \
	    -connection $connid \
	    -command $cmd
    set substatus($connid,$to) sent-subscribe
}

proc mood::unsubscribe {connid jid args} {
    variable node
    variable substatus

    set to [node_and_server_from_jid $jid]
    set cmd [linsert $args 0 [namespace current]::unsubscribe_result $connid $to]
    pep::unsubscribe $to $node \
	    -connection $connid \
	    -command $cmd
    set substatus($connid,$to) sent-unsubscribe
}

# Err may be one of: OK, ERR and DISCONNECT
proc mood::subscribe_result {connid jid res child args} {
    variable substatus

    set cmd ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command {
		set cmd $val
	    }
	    default {
		return -code error "unknown option: $opt"
	    }
	}
    }

    switch -- $res {
	OK {
	    set substatus($connid,$jid) from
	}
	ERR {
	    set substatus($connid,$jid) error
	}
	default {
	    return
	}
    }

    if {$cmd != ""} {
	lappend cmd $jid $res $child
	eval $cmd
    }
}

proc mood::unsubscribe_result {connid jid res child args} {
    variable substatus
    variable mood

    set cmd ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command {
		set cmd $val
	    }
	    default {
		return -code error "unknown option: $opt"
	    }
	}
    }

    if {[string equal $res OK]} {
	set substatus($connid,$jid) none
	catch {unset mood(mood,$connid,$jid)}
	catch {unset mood(text,$connid,$jid)}
    }

    if {$cmd != ""} {
	lappend cmd $jid $res $child
	eval $cmd
    }
}

proc mood::provide_roster_popup_info {var connid user} {
    variable substatus
    variable mood
    variable m2d

    upvar 0 $var info

    set jid [node_and_server_from_jid $user]

    if {[info exists mood(mood,$connid,$jid)]} {
	set m $mood(mood,$connid,$jid)
	if {[info exists m2d($m)]} {
	    set status $m2d($m)
	} else {
	    set status $m
	    debugmsg pubsub "Failed to found description for user mood \"$m\"\
			     -- discrepancies with XEP-0107?"
	}
	if {[info exists mood(text,$connid,$jid)] && $mood(text,$connid,$jid) != ""} {
	    append status ": " $mood(text,$connid,$jid)
	}
	append info [::msgcat::mc "\n\tMood: %s" $status]
    } elseif {[info exists substatus($connid,$jid)]} {
	append info [::msgcat::mc "\n\tUser mood subscription: %s" \
			    $substatus($connid,$jid)]
    } else {
	return
    }

}

proc mood::process_mood_notification {connid jid items} {
    variable node
    variable mood

    set newmood ""
    set newtext ""
    set retract false
    set parsed  false

    foreach item $items {
	jlib::wrapper:splitxml $item tag vars isempty chdata children

	switch -- $tag {
	    retract {
		set retract true
	    }
	    default {
		foreach imood $children {
		    jlib::wrapper:splitxml $imood tag1 vars1 isempty1 chdata1 children1

		    if {![string equal $tag1 mood]} continue
		    set xmlns [jlib::wrapper:getattr $vars1 xmlns]
		    if {![string equal $xmlns $node]} continue

		    set parsed true

		    foreach i $children1 {
			jlib::wrapper:splitxml $i tag2 vars2 isempty2 chdata2 children2

			switch -- $tag2 {
			    text {
				set newtext $chdata2
			    }
			    default {
				set newmood $tag2
			    }
			}
		    }
		}
	    }
	}
    }

    if {$parsed} {
	set mood(mood,$connid,$jid) $newmood
	set mood(text,$connid,$jid) $newtext

	hook::run user_mood_notification_hook $connid $jid $newmood $newtext
    } elseif {$retract} {
	catch {unset mood(mood,$connid,$jid)}
	catch {unset mood(text,$connid,$jid)}

	hook::run user_mood_notification_hook $connid $jid "" ""
    }
}

proc mood::notify_via_status_message {connid jid mood text} {
    variable m2d

    set contact [::roster::itemconfig $connid $jid -name]
    if {$contact == ""} {
	set contact $jid
    }

    if {$mood == ""} {
	set msg [::msgcat::mc "%s's mood is unset" $contact]
    } elseif {[info exists m2d($mood)]} {
	set msg [::msgcat::mc "%s's mood changed to %s" $contact $m2d($mood)]
	if {$text != ""} {
	    append msg ": $text"
	}
    } else {
	set msg [::msgcat::mc "%s's mood changed to %s" $contact $mood]
	if {$text != ""} {
	    append msg ": $text"
	}
    }

    set_status $msg
}

proc mood::publish {connid mood args} {
    variable node

    set text ""
    set callback ""
    foreach {opt val} $args {
	switch -- $opt {
	    -reason  { set text $val }
	    -command { set callback $val }
	}
    }

    set content [list [jlib::wrapper:createtag $mood]]
    if {$text != ""} {
	lappend content [jlib::wrapper:createtag text -chdata $text]
    }

    set cmd [list pep::publish_item $node mood \
		  -connection $connid \
		  -payload [list [jlib::wrapper:createtag mood \
				      -vars [list xmlns $node] \
				      -subtags $content]]]

    if {$callback != ""} {
	lappend cmd -command $callback
    }

    eval $cmd
}

proc mood::unpublish {connid args} {
    variable node

    set callback ""
    foreach {opt val} $args {
	switch -- $opt {
	    -command { set callback $val }
	}
    }

    set cmd [list pep::delete_item $node mood \
		  -notify true \
		  -connection $connid]

    if {$callback != ""} {
	lappend cmd -command $callback
    }

    eval $cmd
}

proc mood::on_init {} {
    set m [pep::get_main_menu_pep_submenu]
    set mm [menu $m.mood -tearoff $::ifacetk::options(show_tearoffs)]
    $m add cascade -menu $mm \
	   -label [::msgcat::mc "User mood"]
    $mm add command -label [::msgcat::mc "Publish user mood..."] \
	    -state disabled \
	    -command [namespace current]::show_publish_dialog
    $mm add command -label [::msgcat::mc "Unpublish user mood"] \
	    -state disabled \
	    -command [namespace current]::show_unpublish_dialog
    $mm add checkbutton -label [::msgcat::mc "Auto-subscribe to other's user mood"] \
	    -variable [namespace current]::options(auto-subscribe)
}

proc mood::on_connect_disconnect {args} {
    set mm [pep::get_main_menu_pep_submenu].mood
    set idx [expr {$::ifacetk::options(show_tearoffs) ? 1 : 0}]

    switch -- [llength [jlib::connections]] {
	0 {
	    $mm entryconfigure $idx -state disabled
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user mood"] \
		-state disabled
	}
	1 {
	    $mm entryconfigure $idx -state normal
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user mood"] \
		-state normal
	}
	default {
	    $mm entryconfigure $idx -state normal
	    $mm entryconfigure [incr idx] \
		-label [::msgcat::mc "Unpublish user mood..."] \
		-state normal
	}
    }
}

proc mood::show_publish_dialog {} {
    variable d2m
    variable moodvalue
    variable moodreason
    variable myjid

    set w .user_mood
    if {[winfo exists $w]} {
	destroy $w
    }

    set connids [jlib::connections]
    if {[llength $connids] == 0} {
	NonmodalMessageDlg [epath] \
		-aspect 50000 \
		-icon error \
		-title [::msgcat::mc "Error"] \
		-message [::msgcat::mc "Publishing is only possible\
					while being online"]
	return
    }

    Dialog $w -title [::msgcat::mc "User mood"] \
	    -modal none -separator 1 -anchor e -default 0 -cancel 1 -parent .
    $w add -text [::msgcat::mc "Publish"] \
	   -command [list [namespace current]::do_publish $w]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    set f [$w getframe]

    set connjids [list [::msgcat::mc "All"]]
    foreach connid $connids {
	lappend connjids [jlib::connection_jid $connid]
    }
    set myjid [lindex $connjids 0]

    label $f.ccap -text [::msgcat::mc "Use connection:"]
    ComboBox $f.conn -editable false \
	    -values $connjids \
	    -textvariable [namespace current]::myjid
    label $f.mcap -text [::msgcat::mc "Mood:"]
    ComboBox $f.mood -editable false \
	    -values [lsort [array names d2m]] \
	    -textvariable [namespace current]::moodvalue
    label $f.rcap -text [::msgcat::mc "Reason:"]
    entry $f.reason -textvariable [namespace current]::moodreason

    if {[llength $connjids] > 1} {
	grid $f.ccap   -row 0 -column 0 -sticky e
	grid $f.conn   -row 0 -column 1 -sticky ew
    }
    grid $f.mcap   -row 1 -column 0 -sticky e
    grid $f.mood   -row 1 -column 1 -sticky ew
    grid $f.rcap   -row 2 -column 0 -sticky e
    grid $f.reason -row 2 -column 1 -sticky ew

    grid columnconfigure $f 1 -weight 1

    $w draw
}

proc mood::do_publish {w} {
    variable d2m
    variable moodvalue
    variable moodreason
    variable myjid

    if {$moodvalue == ""} {
	NonmodalMessageDlg [epath] \
		-aspect 50000 \
		-icon error \
		-title [::msgcat::mc "Error"] \
		-message [::msgcat::mc "Cannot publish empty mood"]
	return
    }

    foreach connid [jlib::connections] {
	if {[string equal $myjid [jlib::connection_jid $connid]] || \
		[string equal $myjid [::msgcat::mc "All"]]} {
	    publish $connid $d2m($moodvalue) \
		    -reason $moodreason \
		    -command [namespace current]::publish_result
	    break
	}
    }

    unset moodvalue moodreason myjid
    destroy $w
}

# $res is one of: OK, ERR, DISCONNECT
proc mood::publish_result {res child} {
    switch -- $res {
	ERR {
	    set error [error_to_string $child]
	}
	default {
	    return
	}
    }

    NonmodalMessageDlg [epath] \
	    -aspect 50000 \
	    -icon error \
	    -title [::msgcat::mc "Error"] \
	    -message [::msgcat::mc "User mood publishing failed: %s" $error]
}

proc mood::show_unpublish_dialog {} {
    variable myjid

    set w .user_mood
    if {[winfo exists $w]} {
	destroy $w
    }

    set connids [jlib::connections]
    if {[llength $connids] == 0} {
	NonmodalMessageDlg [epath] \
		-aspect 50000 \
		-icon error \
		-title [::msgcat::mc "Error"] \
		-message [::msgcat::mc "Unpublishing is only possible\
					while being online"]
	return
    }

    Dialog $w -title [::msgcat::mc "User mood"] \
	    -modal none -separator 1 -anchor e -default 0 -cancel 1 -parent .
    $w add -text [::msgcat::mc "Unpublish"] \
	   -command [list [namespace current]::do_unpublish $w]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    set f [$w getframe]

    set connjids [list [::msgcat::mc "All"]]
    foreach connid $connids {
	lappend connjids [jlib::connection_jid $connid]
    }
    set myjid [lindex $connjids 0]

    label $f.ccap -text [::msgcat::mc "Use connection:"]
    ComboBox $f.conn -editable false \
	    -values $connjids \
	    -textvariable [namespace current]::myjid

    if {[llength $connjids] > 1} {
	grid $f.ccap   -row 0 -column 0 -sticky e
	grid $f.conn   -row 0 -column 1 -sticky ew
    }

    grid columnconfigure $f 1 -weight 1

    if {[llength $connids] == 1} {
	do_unpublish $w
    } else {
	$w draw
    }
}

proc mood::do_unpublish {w} {
    variable myjid

    foreach connid [jlib::connections] {
	if {[string equal $myjid [jlib::connection_jid $connid]] || \
		[string equal $myjid [::msgcat::mc "All"]]} {
	    unpublish $connid \
		    -command [namespace current]::unpublish_result
	    break
	}
    }

    unset myjid
    destroy $w
}

# $res is one of: OK, ERR, DISCONNECT
proc mood::unpublish_result {res child} {
    switch -- $res {
	ERR {
	    if {[lindex [error_type_condition $child] 1] == "item-not-found"} {
		return
	    }
	    set error [error_to_string $child]
	}
	default {
	    return
	}
    }

    NonmodalMessageDlg [epath] \
	    -aspect 50000 \
	    -icon error \
	    -title [::msgcat::mc "Error"] \
	    -message [::msgcat::mc "User mood unpublishing failed: %s" $error]
}

proc mood::provide_userinfo {notebook connid jid editable} {
    variable mood
    variable m2d
    variable ::userinfo::userinfo

    if {$editable} return

    set barejid [node_and_server_from_jid $jid]
    if {![info exists mood(mood,$connid,$barejid)]} return

    set userinfo(mood,$jid) $m2d($mood(mood,$connid,$barejid))
    if {[info exists mood(text,$connid,$barejid)]} {
	set userinfo(moodreason,$jid) $mood(text,$connid,$barejid)
    } else {
	set userinfo(moodreason,$jid) ""
    }

    set f [pep::get_userinfo_dialog_pep_frame $notebook]
    set mf [userinfo::pack_frame $f.mood [::msgcat::mc "User mood"]]

    userinfo::pack_entry $jid $mf 0 mood [::msgcat::mc "Mood"]:
    userinfo::pack_entry $jid $mf 1 moodreason [::msgcat::mc "Reason"]:
}

# vim:ts=8:sw=4:sts=4:noet
