//===--- SortStrings.swift ------------------------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2021 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//
import TestsUtils

let t: [BenchmarkCategory] = [.validation, .api, .algorithm, .String]
// Sort an array of strings using an explicit sort predicate.
public let benchmarks = [
  BenchmarkInfo(name: "SortSortedStrings",
    runFunction: run_SortSortedStrings, tags: t,
    setUpFunction: { blackHole(sortedWords) }),
  BenchmarkInfo(name: "SortStrings",
    runFunction: run_SortStrings, tags: t,
    setUpFunction: { blackHole(words) }),
  BenchmarkInfo(name: "SortStringsUnicode",
    runFunction: run_SortStringsUnicode, tags: t,
    setUpFunction: { blackHole(unicodeWords) }, legacyFactor: 5),
]

let sortedWords = words.sorted()
let words: [String] = [
  "woodshed",
  "lakism",
  "gastroperiodynia",
  "afetal",
  "ramsch",
  "Nickieben",
  "undutifulness",
  "birdglue",
  "ungentlemanize",
  "menacingly",
  "heterophile",
  "leoparde",
  "Casearia",
  "decorticate",
  "neognathic",
  "mentionable",
  "tetraphenol",
  "pseudonymal",
  "dislegitimate",
  "Discoidea",
  "intitule",
  "ionium",
  "Lotuko",
  "timbering",
  "nonliquidating",
  "oarialgia",
  "Saccobranchus",
  "reconnoiter",
  "criminative",
  "disintegratory",
  "executer",
  "Cylindrosporium",
  "complimentation",
  "Ixiama",
  "Araceae",
  "silaginoid",
  "derencephalus",
  "Lamiidae",
  "marrowlike",
  "ninepin",
  "dynastid",
  "lampfly",
  "feint",
  "trihemimer",
  "semibarbarous",
  "heresy",
  "tritanope",
  "indifferentist",
  "confound",
  "hyperbolaeon",
  "planirostral",
  "philosophunculist",
  "existence",
  "fretless",
  "Leptandra",
  "Amiranha",
  "handgravure",
  "gnash",
  "unbelievability",
  "orthotropic",
  "Susumu",
  "teleutospore",
  "sleazy",
  "shapeliness",
  "hepatotomy",
  "exclusivism",
  "stifler",
  "cunning",
  "isocyanuric",
  "pseudepigraphy",
  "carpetbagger",
  "respectiveness",
  "Jussi",
  "vasotomy",
  "proctotomy",
  "ovatotriangular",
  "aesthetic",
  "schizogamy",
  "disengagement",
  "foray",
  "haplocaulescent",
  "noncoherent",
  "astrocyte",
  "unreverberated",
  "presenile",
  "lanson",
  "enkraal",
  "contemplative",
  "Syun",
  "sartage",
  "unforgot",
  "wyde",
  "homeotransplant",
  "implicational",
  "forerunnership",
  "calcaneum",
  "stomatodeum",
  "pharmacopedia",
  "preconcessive",
  "trypanosomatic",
  "intracollegiate",
  "rampacious",
  "secundipara",
  "isomeric",
  "treehair",
  "pulmonal",
  "uvate",
  "dugway",
  "glucofrangulin",
  "unglory",
  "Amandus",
  "icterogenetic",
  "quadrireme",
  "Lagostomus",
  "brakeroot",
  "anthracemia",
  "fluted",
  "protoelastose",
  "thro",
  "pined",
  "Saxicolinae",
  "holidaymaking",
  "strigil",
  "uncurbed",
  "starling",
  "redeemeress",
  "Liliaceae",
  "imparsonee",
  "obtusish",
  "brushed",
  "mesally",
  "probosciformed",
  "Bourbonesque",
  "histological",
  "caroba",
  "digestion",
  "Vindemiatrix",
  "triactinal",
  "tattling",
  "arthrobacterium",
  "unended",
  "suspectfulness",
  "movelessness",
  "chartist",
  "Corynebacterium",
  "tercer",
  "oversaturation",
  "Congoleum",
  "antiskeptical",
  "sacral",
  "equiradiate",
  "whiskerage",
  "panidiomorphic",
  "unplanned",
  "anilopyrine",
  "Queres",
  "tartronyl",
  "Ing",
  "notehead",
  "finestiller",
  "weekender",
  "kittenhood",
  "competitrix",
  "premillenarian",
  "convergescence",
  "microcoleoptera",
  "slirt",
  "asteatosis",
  "Gruidae",
  "metastome",
  "ambuscader",
  "untugged",
  "uneducated",
  "redistill",
  "rushlight",
  "freakish",
  "dosology",
  "papyrine",
  "iconologist",
  "Bidpai",
  "prophethood",
  "pneumotropic",
  "chloroformize",
  "intemperance",
  "spongiform",
  "superindignant",
  "divider",
  "starlit",
  "merchantish",
  "indexless",
  "unidentifiably",
  "coumarone",
  "nomism",
  "diaphanous",
  "salve",
  "option",
  "anallantoic",
  "paint",
  "thiofurfuran",
  "baddeleyite",
  "Donne",
  "heterogenicity",
  "decess",
  "eschynite",
  "mamma",
  "unmonarchical",
  "Archiplata",
  "widdifow",
  "apathic",
  "overline",
  "chaetophoraceous",
  "creaky",
  "trichosporange",
  "uninterlined",
  "cometwise",
  "hermeneut",
  "unbedraggled",
  "tagged",
  "Sminthurus",
  "somniloquacious",
  "aphasiac",
  "Inoperculata",
  "photoactivity",
  "mobship",
  "unblightedly",
  "lievrite",
  "Khoja",
  "Falerian",
  "milfoil",
  "protectingly",
  "householder",
  "cathedra",
  "calmingly",
  "tordrillite",
  "rearhorse",
  "Leonard",
  "maracock",
  "youngish",
  "kammererite",
  "metanephric",
  "Sageretia",
  "diplococcoid",
  "accelerative",
  "choreal",
  "metalogical",
  "recombination",
  "unimprison",
  "invocation",
  "syndetic",
  "toadback",
  "vaned",
  "cupholder",
  "metropolitanship",
  "paramandelic",
  "dermolysis",
  "Sheriyat",
  "rhabdus",
  "seducee",
  "encrinoid",
  "unsuppliable",
  "cololite",
  "timesaver",
  "preambulate",
  "sampling",
  "roaster",
  "springald",
  "densher",
  "protraditional",
  "naturalesque",
  "Hydrodamalis",
  "cytogenic",
  "shortly",
  "cryptogrammatical",
  "squat",
  "genual",
  "backspier",
  "solubleness",
  "macroanalytical",
  "overcovetousness",
  "Natalie",
  "cuprobismutite",
  "phratriac",
  "Montanize",
  "hymnologist",
  "karyomiton",
  "podger",
  "unofficiousness",
  "antisplasher",
  "supraclavicular",
  "calidity",
  "disembellish",
  "antepredicament",
  "recurvirostral",
  "pulmonifer",
  "coccidial",
  "botonee",
  "protoglobulose",
  "isonym",
  "myeloid",
  "premiership",
  "unmonopolize",
  "unsesquipedalian",
  "unfelicitously",
  "theftbote",
  "undauntable",
  "lob",
  "praenomen",
  "underriver",
  "gorfly",
  "pluckage",
  "radiovision",
  "tyrantship",
  "fraught",
  "doppelkummel",
  "rowan",
  "allosyndetic",
  "kinesiology",
  "psychopath",
  "arrent",
  "amusively",
  "preincorporation",
  "Montargis",
  "pentacron",
  "neomedievalism",
  "sima",
  "lichenicolous",
  "Ecclesiastes",
  "woofed",
  "cardinalist",
  "sandaracin",
  "gymnasial",
  "lithoglyptics",
  "centimeter",
  "quadrupedous",
  "phraseology",
  "tumuli",
  "ankylotomy",
  "myrtol",
  "cohibitive",
  "lepospondylous",
  "silvendy",
  "inequipotential",
  "entangle",
  "raveling",
  "Zeugobranchiata",
  "devastating",
  "grainage",
  "amphisbaenian",
  "blady",
  "cirrose",
  "proclericalism",
  "governmentalist",
  "carcinomorphic",
  "nurtureship",
  "clancular",
  "unsteamed",
  "discernibly",
  "pleurogenic",
  "impalpability",
  "Azotobacterieae",
  "sarcoplasmic",
  "alternant",
  "fitly",
  "acrorrheuma",
  "shrapnel",
  "pastorize",
  "gulflike",
  "foreglow",
  "unrelated",
  "cirriped",
  "cerviconasal",
  "sexuale",
  "pussyfooter",
  "gadolinic",
  "duplicature",
  "codelinquency",
  "trypanolysis",
  "pathophobia",
  "incapsulation",
  "nonaerating",
  "feldspar",
  "diaphonic",
  "epiglottic",
  "depopulator",
  "wisecracker",
  "gravitational",
  "kuba",
  "lactesce",
  "Toxotes",
  "periomphalic",
  "singstress",
  "fannier",
  "counterformula",
  "Acemetae",
  "repugnatorial",
  "collimator",
  "Acinetina",
  "unpeace",
  "drum",
  "tetramorphic",
  "descendentalism",
  "cementer",
  "supraloral",
  "intercostal",
  "Nipponize",
  "negotiator",
  "vacationless",
  "synthol",
  "fissureless",
  "resoap",
  "pachycarpous",
  "reinspiration",
  "misappropriation",
  "disdiazo",
  "unheatable",
  "streng",
  "Detroiter",
  "infandous",
  "loganiaceous",
  "desugar",
  "Matronalia",
  "myxocystoma",
  "Gandhiism",
  "kiddier",
  "relodge",
  "counterreprisal",
  "recentralize",
  "foliously",
  "reprinter",
  "gender",
  "edaciousness",
  "chondriomite",
  "concordant",
  "stockrider",
  "pedary",
  "shikra",
  "blameworthiness",
  "vaccina",
  "Thamnophilinae",
  "wrongwise",
  "unsuperannuated",
  "convalescency",
  "intransmutable",
  "dropcloth",
  "Ceriomyces",
  "ponderal",
  "unstentorian",
  "mem",
  "deceleration",
  "ethionic",
  "untopped",
  "wetback",
  "bebar",
  "undecaying",
  "shoreside",
  "energize",
  "presacral",
  "undismay",
  "agricolite",
  "cowheart",
  "hemibathybian",
  "postexilian",
  "Phacidiaceae",
  "offing",
  "redesignation",
  "skeptically",
  "physicianless",
  "bronchopathy",
  "marabuto",
  "proprietory",
  "unobtruded",
  "funmaker",
  "plateresque",
  "preadventure",
  "beseeching",
  "cowpath",
  "pachycephalia",
  "arthresthesia",
  "supari",
  "lengthily",
  "Nepa",
  "liberation",
  "nigrify",
  "belfry",
  "entoolitic",
  "bazoo",
  "pentachromic",
  "distinguishable",
  "slideable",
  "galvanoscope",
  "remanage",
  "cetene",
  "bocardo",
  "consummation",
  "boycottism",
  "perplexity",
  "astay",
  "Gaetuli",
  "periplastic",
  "consolidator",
  "sluggarding",
  "coracoscapular",
  "anangioid",
  "oxygenizer",
  "Hunanese",
  "seminary",
  "periplast",
  "Corylus",
  "unoriginativeness",
  "persecutee",
  "tweaker",
  "silliness",
  "Dabitis",
  "facetiousness",
  "thymy",
  "nonimperial",
  "mesoblastema",
  "turbiniform",
  "churchway",
  "cooing",
  "frithbot",
  "concomitantly",
  "stalwartize",
  "clingfish",
  "hardmouthed",
  "parallelepipedonal",
  "coracoacromial",
  "factuality",
  "curtilage",
  "arachnoidean",
  "semiaridity",
  "phytobacteriology",
  "premastery",
  "hyperpurist",
  "mobed",
  "opportunistic",
  "acclimature",
  "outdistance",
  "sophister",
  "condonement",
  "oxygenerator",
  "acetonic",
  "emanatory",
  "periphlebitis",
  "nonsociety",
  "spectroradiometric",
  "superaverage",
  "cleanness",
  "posteroventral",
  "unadvised",
  "unmistakedly",
  "pimgenet",
  "auresca",
  "overimitate",
  "dipnoan",
  "chromoxylograph",
  "triakistetrahedron",
  "Suessiones",
  "uncopiable",
  "oligomenorrhea",
  "fribbling",
  "worriable",
  "flot",
  "ornithotrophy",
  "phytoteratology",
  "setup",
  "lanneret",
  "unbraceleted",
  "gudemother",
  "Spica",
  "unconsolatory",
  "recorruption",
  "premenstrual",
  "subretinal",
  "millennialist",
  "subjectibility",
  "rewardproof",
  "counterflight",
  "pilomotor",
  "carpetbaggery",
  "macrodiagonal",
  "slim",
  "indiscernible",
  "cuckoo",
  "moted",
  "controllingly",
  "gynecopathy",
  "porrectus",
  "wanworth",
  "lutfisk",
  "semiprivate",
  "philadelphy",
  "abdominothoracic",
  "coxcomb",
  "dambrod",
  "Metanemertini",
  "balminess",
  "homotypy",
  "waremaker",
  "absurdity",
  "gimcrack",
  "asquat",
  "suitable",
  "perimorphous",
  "kitchenwards",
  "pielum",
  "salloo",
  "paleontologic",
  "Olson",
  "Tellinidae",
  "ferryman",
  "peptonoid",
  "Bopyridae",
  "fallacy",
  "ictuate",
  "aguinaldo",
  "rhyodacite",
  "Ligydidae",
  "galvanometric",
  "acquisitor",
  "muscology",
  "hemikaryon",
  "ethnobotanic",
  "postganglionic",
  "rudimentarily",
  "replenish",
  "phyllorhine",
  "popgunnery",
  "summar",
  "quodlibetary",
  "xanthochromia",
  "autosymbolically",
  "preloreal",
  "extent",
  "strawberry",
  "immortalness",
  "colicwort",
  "frisca",
  "electiveness",
  "heartbroken",
  "affrightingly",
  "reconfiscation",
  "jacchus",
  "imponderably",
  "semantics",
  "beennut",
  "paleometeorological",
  "becost",
  "timberwright",
  "resuppose",
  "syncategorematical",
  "cytolymph",
  "steinbok",
  "explantation",
  "hyperelliptic",
  "antescript",
  "blowdown",
  "antinomical",
  "caravanserai",
  "unweariedly",
  "isonymic",
  "keratoplasty",
  "vipery",
  "parepigastric",
  "endolymphatic",
  "Londonese",
  "necrotomy",
  "angelship",
  "Schizogregarinida",
  "steeplebush",
  "sparaxis",
  "connectedness",
  "tolerance",
  "impingent",
  "agglutinin",
  "reviver",
  "hieroglyphical",
  "dialogize",
  "coestate",
  "declamatory",
  "ventilation",
  "tauromachy",
  "cotransubstantiate",
  "pome",
  "underseas",
  "triquadrantal",
  "preconfinemnt",
  "electroindustrial",
  "selachostomous",
  "nongolfer",
  "mesalike",
  "hamartiology",
  "ganglioblast",
  "unsuccessive",
  "yallow",
  "bacchanalianly",
  "platydactyl",
  "Bucephala",
  "ultraurgent",
  "penalist",
  "catamenial",
  "lynnhaven",
  "unrelevant",
  "lunkhead",
  "metropolitan",
  "hydro",
  "outsoar",
  "vernant",
  "interlanguage",
  "catarrhal",
  "Ionicize",
  "keelless",
  "myomantic",
  "booker",
  "Xanthomonas",
  "unimpeded",
  "overfeminize",
  "speronaro",
  "diaconia",
  "overholiness",
  "liquefacient",
  "Spartium",
  "haggly",
  "albumose",
  "nonnecessary",
  "sulcalization",
  "decapitate",
  "cellated",
  "unguirostral",
  "trichiurid",
  "loveproof",
  "amakebe",
  "screet",
  "arsenoferratin",
  "unfrizz",
  "undiscoverable",
  "procollectivistic",
  "tractile",
  "Winona",
  "dermostosis",
  "eliminant",
  "scomberoid",
  "tensile",
  "typesetting",
  "xylic",
  "dermatopathology",
  "cycloplegic",
  "revocable",
  "fissate",
  "afterplay",
  "screwship",
  "microerg",
  "bentonite",
  "stagecoaching",
  "beglerbeglic",
  "overcharitably",
  "Plotinism",
  "Veddoid",
  "disequalize",
  "cytoproct",
  "trophophore",
  "antidote",
  "allerion",
  "famous",
  "convey",
  "postotic",
  "rapillo",
  "cilectomy",
  "penkeeper",
  "patronym",
  "bravely",
  "ureteropyelitis",
  "Hildebrandine",
  "missileproof",
  "Conularia",
  "deadening",
  "Conrad",
  "pseudochylous",
  "typologically",
  "strummer",
  "luxuriousness",
  "resublimation",
  "glossiness",
  "hydrocauline",
  "anaglyph",
  "personifiable",
  "seniority",
  "formulator",
  "datiscaceous",
  "hydracrylate",
  "Tyranni",
  "Crawthumper",
  "overprove",
  "masher",
  "dissonance",
  "Serpentinian",
  "malachite",
  "interestless",
  "stchi",
  "ogum",
  "polyspermic",
  "archegoniate",
  "precogitation",
  "Alkaphrah",
  "craggily",
  "delightfulness",
  "bioplast",
  "diplocaulescent",
  "neverland",
  "interspheral",
  "chlorhydric",
  "forsakenly",
  "scandium",
  "detubation",
  "telega",
  "Valeriana",
  "centraxonial",
  "anabolite",
  "neger",
  "miscellanea",
  "whalebacker",
  "stylidiaceous",
  "unpropelled",
  "Kennedya",
  "Jacksonite",
  "ghoulish",
  "Dendrocalamus",
  "paynimhood",
  "rappist",
  "unluffed",
  "falling",
  "Lyctus",
  "uncrown",
  "warmly",
  "pneumatism",
  "Morisonian",
  "notate",
  "isoagglutinin",
  "Pelidnota",
  "previsit",
  "contradistinctly",
  "utter",
  "porometer",
  "gie",
  "germanization",
  "betwixt",
  "prenephritic",
  "underpier",
  "Eleutheria",
  "ruthenious",
  "convertor",
  "antisepsin",
  "winterage",
  "tetramethylammonium",
  "Rockaway",
  "Penaea",
  "prelatehood",
  "brisket",
  "unwishful",
  "Minahassa",
  "Briareus",
  "semiaxis",
  "disintegrant",
  "peastick",
  "iatromechanical",
  "fastus",
  "thymectomy",
  "ladyless",
  "unpreened",
  "overflutter",
  "sicker",
  "apsidally",
  "thiazine",
  "guideway",
  "pausation",
  "tellinoid",
  "abrogative",
  "foraminulate",
  "omphalos",
  "Monorhina",
  "polymyarian",
  "unhelpful",
  "newslessness",
  "oryctognosy",
  "octoradial",
  "doxology",
  "arrhythmy",
  "gugal",
  "mesityl",
  "hexaplaric",
  "Cabirian",
  "hordeiform",
  "eddyroot",
  "internarial",
  "deservingness",
  "jawbation",
  "orographically",
  "semiprecious",
  "seasick",
  "thermically",
  "grew",
  "tamability",
  "egotistically",
  "fip",
  "preabsorbent",
  "leptochroa",
  "ethnobotany",
  "podolite",
  "egoistic",
  "semitropical",
  "cero",
  "spinelessness",
  "onshore",
  "omlah",
  "tintinnabulist",
  "machila",
  "entomotomy",
  "nubile",
  "nonscholastic",
  "burnt",
  "Alea",
  "befume",
  "doctorless",
  "Napoleonic",
  "scenting",
  "apokreos",
  "cresylene",
  "paramide",
  "rattery",
  "disinterested",
  "idiopathetic",
  "negatory",
  "fervid",
  "quintato",
  "untricked",
  "Metrosideros",
  "mescaline",
  "midverse",
  "Musophagidae",
  "fictionary",
  "branchiostegous",
  "yoker",
  "residuum",
  "culmigenous",
  "fleam",
  "suffragism",
  "Anacreon",
  "sarcodous",
  "parodistic",
  "writmaking",
  "conversationism",
  "retroposed",
  "tornillo",
  "presuspect",
  "didymous",
  "Saumur",
  "spicing",
  "drawbridge",
  "cantor",
  "incumbrancer",
  "heterospory",
  "Turkeydom",
  "anteprandial",
  "neighborship",
  "thatchless",
  "drepanoid",
  "lusher",
  "paling",
  "ecthlipsis",
  "heredosyphilitic",
  "although",
  "garetta",
  "temporarily",
  "Monotropa",
  "proglottic",
  "calyptro",
  "persiflage",
  "degradable",
  "paraspecific",
  "undecorative",
  "Pholas",
  "myelon",
  "resteal",
  "quadrantly",
  "scrimped",
  "airer",
  "deviless",
  "caliciform",
  "Sefekhet",
  "shastaite",
  "togate",
  "macrostructure",
  "bipyramid",
  "wey",
  "didynamy",
  "knacker",
  "swage",
  "supermanism",
  "epitheton",
  "overpresumptuous"
  ]


@inline(never)
func benchSortStrings(_ words: [String]) {
  // Notice that we _copy_ the array of words before we sort it.
  // Pass an explicit '<' predicate to benchmark reabstraction thunks.
  var tempwords = words
  tempwords.sort(by: <)
}

public func run_SortStrings(_ n: Int) {
  for _ in 1...5*n {
    benchSortStrings(words)
  }
}

public func run_SortSortedStrings(_ n: Int) {
  for _ in 1...5*n {
    benchSortStrings(sortedWords)
  }
}

let unicodeWords: [String] = [
  "❄️woodshed",
  "❄️lakism",
  "❄️gastroperiodynia",
  "❄️afetal",
  "❄️ramsch",
  "❄️Nickieben",
  "❄️undutifulness",
  "❄️birdglue",
  "❄️ungentlemanize",
  "❄️menacingly",
  "❄️heterophile",
  "❄️leoparde",
  "❄️Casearia",
  "❄️decorticate",
  "❄️neognathic",
  "❄️mentionable",
  "❄️tetraphenol",
  "❄️pseudonymal",
  "❄️dislegitimate",
  "❄️Discoidea",
  "❄️intitule",
  "❄️ionium",
  "❄️Lotuko",
  "❄️timbering",
  "❄️nonliquidating",
  "❄️oarialgia",
  "❄️Saccobranchus",
  "❄️reconnoiter",
  "❄️criminative",
  "❄️disintegratory",
  "❄️executer",
  "❄️Cylindrosporium",
  "❄️complimentation",
  "❄️Ixiama",
  "❄️Araceae",
  "❄️silaginoid",
  "❄️derencephalus",
  "❄️Lamiidae",
  "❄️marrowlike",
  "❄️ninepin",
  "❄️dynastid",
  "❄️lampfly",
  "❄️feint",
  "❄️trihemimer",
  "❄️semibarbarous",
  "❄️heresy",
  "❄️tritanope",
  "❄️indifferentist",
  "❄️confound",
  "❄️hyperbolaeon",
  "❄️planirostral",
  "❄️philosophunculist",
  "❄️existence",
  "❄️fretless",
  "❄️Leptandra",
  "❄️Amiranha",
  "❄️handgravure",
  "❄️gnash",
  "❄️unbelievability",
  "❄️orthotropic",
  "❄️Susumu",
  "❄️teleutospore",
  "❄️sleazy",
  "❄️shapeliness",
  "❄️hepatotomy",
  "❄️exclusivism",
  "❄️stifler",
  "❄️cunning",
  "❄️isocyanuric",
  "❄️pseudepigraphy",
  "❄️carpetbagger",
  "❄️respectiveness",
  "❄️Jussi",
  "❄️vasotomy",
  "❄️proctotomy",
  "❄️ovatotriangular",
  "❄️aesthetic",
  "❄️schizogamy",
  "❄️disengagement",
  "❄️foray",
  "❄️haplocaulescent",
  "❄️noncoherent",
  "❄️astrocyte",
  "❄️unreverberated",
  "❄️presenile",
  "❄️lanson",
  "❄️enkraal",
  "❄️contemplative",
  "❄️Syun",
  "❄️sartage",
  "❄️unforgot",
  "❄️wyde",
  "❄️homeotransplant",
  "❄️implicational",
  "❄️forerunnership",
  "❄️calcaneum",
  "❄️stomatodeum",
  "❄️pharmacopedia",
  "❄️preconcessive",
  "❄️trypanosomatic",
  "❄️intracollegiate",
  "❄️rampacious",
  "❄️secundipara",
  "❄️isomeric",
  "❄️treehair",
  "❄️pulmonal",
  "❄️uvate",
  "❄️dugway",
  "❄️glucofrangulin",
  "❄️unglory",
  "❄️Amandus",
  "❄️icterogenetic",
  "❄️quadrireme",
  "❄️Lagostomus",
  "❄️brakeroot",
  "❄️anthracemia",
  "❄️fluted",
  "❄️protoelastose",
  "❄️thro",
  "❄️pined",
  "❄️Saxicolinae",
  "❄️holidaymaking",
  "❄️strigil",
  "❄️uncurbed",
  "❄️starling",
  "❄️redeemeress",
  "❄️Liliaceae",
  "❄️imparsonee",
  "❄️obtusish",
  "❄️brushed",
  "❄️mesally",
  "❄️probosciformed",
  "❄️Bourbonesque",
  "❄️histological",
  "❄️caroba",
  "❄️digestion",
  "❄️Vindemiatrix",
  "❄️triactinal",
  "❄️tattling",
  "❄️arthrobacterium",
  "❄️unended",
  "❄️suspectfulness",
  "❄️movelessness",
  "❄️chartist",
  "❄️Corynebacterium",
  "❄️tercer",
  "❄️oversaturation",
  "❄️Congoleum",
  "❄️antiskeptical",
  "❄️sacral",
  "❄️equiradiate",
  "❄️whiskerage",
  "❄️panidiomorphic",
  "❄️unplanned",
  "❄️anilopyrine",
  "❄️Queres",
  "❄️tartronyl",
  "❄️Ing",
  "❄️notehead",
  "❄️finestiller",
  "❄️weekender",
  "❄️kittenhood",
  "❄️competitrix",
  "❄️premillenarian",
  "❄️convergescence",
  "❄️microcoleoptera",
  "❄️slirt",
  "❄️asteatosis",
  "❄️Gruidae",
  "❄️metastome",
  "❄️ambuscader",
  "❄️untugged",
  "❄️uneducated",
  "❄️redistill",
  "❄️rushlight",
  "❄️freakish",
  "❄️dosology",
  "❄️papyrine",
  "❄️iconologist",
  "❄️Bidpai",
  "❄️prophethood",
  "❄️pneumotropic",
  "❄️chloroformize",
  "❄️intemperance",
  "❄️spongiform",
  "❄️superindignant",
  "❄️divider",
  "❄️starlit",
  "❄️merchantish",
  "❄️indexless",
  "❄️unidentifiably",
  "❄️coumarone",
  "❄️nomism",
  "❄️diaphanous",
  "❄️salve",
  "❄️option",
  "❄️anallantoic",
  "❄️paint",
  "❄️thiofurfuran",
  "❄️baddeleyite",
  "❄️Donne",
  "❄️heterogenicity",
  "❄️decess",
  "❄️eschynite",
  "❄️mamma",
  "❄️unmonarchical",
  "❄️Archiplata",
  "❄️widdifow",
  "❄️apathic",
  "❄️overline",
  "❄️chaetophoraceous",
  "❄️creaky",
  "❄️trichosporange",
  "❄️uninterlined",
  "❄️cometwise",
  "❄️hermeneut",
  "❄️unbedraggled",
  "❄️tagged",
  "❄️Sminthurus",
  "❄️somniloquacious",
  "❄️aphasiac",
  "❄️Inoperculata",
  "❄️photoactivity",
  "❄️mobship",
  "❄️unblightedly",
  "❄️lievrite",
  "❄️Khoja",
  "❄️Falerian",
  "❄️milfoil",
  "❄️protectingly",
  "❄️householder",
  "❄️cathedra",
  "❄️calmingly",
  "❄️tordrillite",
  "❄️rearhorse",
  "❄️Leonard",
  "❄️maracock",
  "❄️youngish",
  "❄️kammererite",
  "❄️metanephric",
  "❄️Sageretia",
  "❄️diplococcoid",
  "❄️accelerative",
  "❄️choreal",
  "❄️metalogical",
  "❄️recombination",
  "❄️unimprison",
  "❄️invocation",
  "❄️syndetic",
  "❄️toadback",
  "❄️vaned",
  "❄️cupholder",
  "❄️metropolitanship",
  "❄️paramandelic",
  "❄️dermolysis",
  "❄️Sheriyat",
  "❄️rhabdus",
  "❄️seducee",
  "❄️encrinoid",
  "❄️unsuppliable",
  "❄️cololite",
  "❄️timesaver",
  "❄️preambulate",
  "❄️sampling",
  "❄️roaster",
  "❄️springald",
  "❄️densher",
  "❄️protraditional",
  "❄️naturalesque",
  "❄️Hydrodamalis",
  "❄️cytogenic",
  "❄️shortly",
  "❄️cryptogrammatical",
  "❄️squat",
  "❄️genual",
  "❄️backspier",
  "❄️solubleness",
  "❄️macroanalytical",
  "❄️overcovetousness",
  "❄️Natalie",
  "❄️cuprobismutite",
  "❄️phratriac",
  "❄️Montanize",
  "❄️hymnologist",
  "❄️karyomiton",
  "❄️podger",
  "❄️unofficiousness",
  "❄️antisplasher",
  "❄️supraclavicular",
  "❄️calidity",
  "❄️disembellish",
  "❄️antepredicament",
  "❄️recurvirostral",
  "❄️pulmonifer",
  "❄️coccidial",
  "❄️botonee",
  "❄️protoglobulose",
  "❄️isonym",
  "❄️myeloid",
  "❄️premiership",
  "❄️unmonopolize",
  "❄️unsesquipedalian",
  "❄️unfelicitously",
  "❄️theftbote",
  "❄️undauntable",
  "❄️lob",
  "❄️praenomen",
  "❄️underriver",
  "❄️gorfly",
  "❄️pluckage",
  "❄️radiovision",
  "❄️tyrantship",
  "❄️fraught",
  "❄️doppelkummel",
  "❄️rowan",
  "❄️allosyndetic",
  "❄️kinesiology",
  "❄️psychopath",
  "❄️arrent",
  "❄️amusively",
  "❄️preincorporation",
  "❄️Montargis",
  "❄️pentacron",
  "❄️neomedievalism",
  "❄️sima",
  "❄️lichenicolous",
  "❄️Ecclesiastes",
  "❄️woofed",
  "❄️cardinalist",
  "❄️sandaracin",
  "❄️gymnasial",
  "❄️lithoglyptics",
  "❄️centimeter",
  "❄️quadrupedous",
  "❄️phraseology",
  "❄️tumuli",
  "❄️ankylotomy",
  "❄️myrtol",
  "❄️cohibitive",
  "❄️lepospondylous",
  "❄️silvendy",
  "❄️inequipotential",
  "❄️entangle",
  "❄️raveling",
  "❄️Zeugobranchiata",
  "❄️devastating",
  "❄️grainage",
  "❄️amphisbaenian",
  "❄️blady",
  "❄️cirrose",
  "❄️proclericalism",
  "❄️governmentalist",
  "❄️carcinomorphic",
  "❄️nurtureship",
  "❄️clancular",
  "❄️unsteamed",
  "❄️discernibly",
  "❄️pleurogenic",
  "❄️impalpability",
  "❄️Azotobacterieae",
  "❄️sarcoplasmic",
  "❄️alternant",
  "❄️fitly",
  "❄️acrorrheuma",
  "❄️shrapnel",
  "❄️pastorize",
  "❄️gulflike",
  "❄️foreglow",
  "❄️unrelated",
  "❄️cirriped",
  "❄️cerviconasal",
  "❄️sexuale",
  "❄️pussyfooter",
  "❄️gadolinic",
  "❄️duplicature",
  "❄️codelinquency",
  "❄️trypanolysis",
  "❄️pathophobia",
  "❄️incapsulation",
  "❄️nonaerating",
  "❄️feldspar",
  "❄️diaphonic",
  "❄️epiglottic",
  "❄️depopulator",
  "❄️wisecracker",
  "❄️gravitational",
  "❄️kuba",
  "❄️lactesce",
  "❄️Toxotes",
  "❄️periomphalic",
  "❄️singstress",
  "❄️fannier",
  "❄️counterformula",
  "❄️Acemetae",
  "❄️repugnatorial",
  "❄️collimator",
  "❄️Acinetina",
  "❄️unpeace",
  "❄️drum",
  "❄️tetramorphic",
  "❄️descendentalism",
  "❄️cementer",
  "❄️supraloral",
  "❄️intercostal",
  "❄️Nipponize",
  "❄️negotiator",
  "❄️vacationless",
  "❄️synthol",
  "❄️fissureless",
  "❄️resoap",
  "❄️pachycarpous",
  "❄️reinspiration",
  "❄️misappropriation",
  "❄️disdiazo",
  "❄️unheatable",
  "❄️streng",
  "❄️Detroiter",
  "❄️infandous",
  "❄️loganiaceous",
  "❄️desugar",
  "❄️Matronalia",
  "❄️myxocystoma",
  "❄️Gandhiism",
  "❄️kiddier",
  "❄️relodge",
  "❄️counterreprisal",
  "❄️recentralize",
  "❄️foliously",
  "❄️reprinter",
  "❄️gender",
  "❄️edaciousness",
  "❄️chondriomite",
  "❄️concordant",
  "❄️stockrider",
  "❄️pedary",
  "❄️shikra",
  "❄️blameworthiness",
  "❄️vaccina",
  "❄️Thamnophilinae",
  "❄️wrongwise",
  "❄️unsuperannuated",
  "❄️convalescency",
  "❄️intransmutable",
  "❄️dropcloth",
  "❄️Ceriomyces",
  "❄️ponderal",
  "❄️unstentorian",
  "❄️mem",
  "❄️deceleration",
  "❄️ethionic",
  "❄️untopped",
  "❄️wetback",
  "❄️bebar",
  "❄️undecaying",
  "❄️shoreside",
  "❄️energize",
  "❄️presacral",
  "❄️undismay",
  "❄️agricolite",
  "❄️cowheart",
  "❄️hemibathybian",
  "❄️postexilian",
  "❄️Phacidiaceae",
  "❄️offing",
  "❄️redesignation",
  "❄️skeptically",
  "❄️physicianless",
  "❄️bronchopathy",
  "❄️marabuto",
  "❄️proprietory",
  "❄️unobtruded",
  "❄️funmaker",
  "❄️plateresque",
  "❄️preadventure",
  "❄️beseeching",
  "❄️cowpath",
  "❄️pachycephalia",
  "❄️arthresthesia",
  "❄️supari",
  "❄️lengthily",
  "❄️Nepa",
  "❄️liberation",
  "❄️nigrify",
  "❄️belfry",
  "❄️entoolitic",
  "❄️bazoo",
  "❄️pentachromic",
  "❄️distinguishable",
  "❄️slideable",
  "❄️galvanoscope",
  "❄️remanage",
  "❄️cetene",
  "❄️bocardo",
  "❄️consummation",
  "❄️boycottism",
  "❄️perplexity",
  "❄️astay",
  "❄️Gaetuli",
  "❄️periplastic",
  "❄️consolidator",
  "❄️sluggarding",
  "❄️coracoscapular",
  "❄️anangioid",
  "❄️oxygenizer",
  "❄️Hunanese",
  "❄️seminary",
  "❄️periplast",
  "❄️Corylus",
  "❄️unoriginativeness",
  "❄️persecutee",
  "❄️tweaker",
  "❄️silliness",
  "❄️Dabitis",
  "❄️facetiousness",
  "❄️thymy",
  "❄️nonimperial",
  "❄️mesoblastema",
  "❄️turbiniform",
  "❄️churchway",
  "❄️cooing",
  "❄️frithbot",
  "❄️concomitantly",
  "❄️stalwartize",
  "❄️clingfish",
  "❄️hardmouthed",
  "❄️parallelepipedonal",
  "❄️coracoacromial",
  "❄️factuality",
  "❄️curtilage",
  "❄️arachnoidean",
  "❄️semiaridity",
  "❄️phytobacteriology",
  "❄️premastery",
  "❄️hyperpurist",
  "❄️mobed",
  "❄️opportunistic",
  "❄️acclimature",
  "❄️outdistance",
  "❄️sophister",
  "❄️condonement",
  "❄️oxygenerator",
  "❄️acetonic",
  "❄️emanatory",
  "❄️periphlebitis",
  "❄️nonsociety",
  "❄️spectroradiometric",
  "❄️superaverage",
  "❄️cleanness",
  "❄️posteroventral",
  "❄️unadvised",
  "❄️unmistakedly",
  "❄️pimgenet",
  "❄️auresca",
  "❄️overimitate",
  "❄️dipnoan",
  "❄️chromoxylograph",
  "❄️triakistetrahedron",
  "❄️Suessiones",
  "❄️uncopiable",
  "❄️oligomenorrhea",
  "❄️fribbling",
  "❄️worriable",
  "❄️flot",
  "❄️ornithotrophy",
  "❄️phytoteratology",
  "❄️setup",
  "❄️lanneret",
  "❄️unbraceleted",
  "❄️gudemother",
  "❄️Spica",
  "❄️unconsolatory",
  "❄️recorruption",
  "❄️premenstrual",
  "❄️subretinal",
  "❄️millennialist",
  "❄️subjectibility",
  "❄️rewardproof",
  "❄️counterflight",
  "❄️pilomotor",
  "❄️carpetbaggery",
  "❄️macrodiagonal",
  "❄️slim",
  "❄️indiscernible",
  "❄️cuckoo",
  "❄️moted",
  "❄️controllingly",
  "❄️gynecopathy",
  "❄️porrectus",
  "❄️wanworth",
  "❄️lutfisk",
  "❄️semiprivate",
  "❄️philadelphy",
  "❄️abdominothoracic",
  "❄️coxcomb",
  "❄️dambrod",
  "❄️Metanemertini",
  "❄️balminess",
  "❄️homotypy",
  "❄️waremaker",
  "❄️absurdity",
  "❄️gimcrack",
  "❄️asquat",
  "❄️suitable",
  "❄️perimorphous",
  "❄️kitchenwards",
  "❄️pielum",
  "❄️salloo",
  "❄️paleontologic",
  "❄️Olson",
  "❄️Tellinidae",
  "❄️ferryman",
  "❄️peptonoid",
  "❄️Bopyridae",
  "❄️fallacy",
  "❄️ictuate",
  "❄️aguinaldo",
  "❄️rhyodacite",
  "❄️Ligydidae",
  "❄️galvanometric",
  "❄️acquisitor",
  "❄️muscology",
  "❄️hemikaryon",
  "❄️ethnobotanic",
  "❄️postganglionic",
  "❄️rudimentarily",
  "❄️replenish",
  "❄️phyllorhine",
  "❄️popgunnery",
  "❄️summar",
  "❄️quodlibetary",
  "❄️xanthochromia",
  "❄️autosymbolically",
  "❄️preloreal",
  "❄️extent",
  "❄️strawberry",
  "❄️immortalness",
  "❄️colicwort",
  "❄️frisca",
  "❄️electiveness",
  "❄️heartbroken",
  "❄️affrightingly",
  "❄️reconfiscation",
  "❄️jacchus",
  "❄️imponderably",
  "❄️semantics",
  "❄️beennut",
  "❄️paleometeorological",
  "❄️becost",
  "❄️timberwright",
  "❄️resuppose",
  "❄️syncategorematical",
  "❄️cytolymph",
  "❄️steinbok",
  "❄️explantation",
  "❄️hyperelliptic",
  "❄️antescript",
  "❄️blowdown",
  "❄️antinomical",
  "❄️caravanserai",
  "❄️unweariedly",
  "❄️isonymic",
  "❄️keratoplasty",
  "❄️vipery",
  "❄️parepigastric",
  "❄️endolymphatic",
  "❄️Londonese",
  "❄️necrotomy",
  "❄️angelship",
  "❄️Schizogregarinida",
  "❄️steeplebush",
  "❄️sparaxis",
  "❄️connectedness",
  "❄️tolerance",
  "❄️impingent",
  "❄️agglutinin",
  "❄️reviver",
  "❄️hieroglyphical",
  "❄️dialogize",
  "❄️coestate",
  "❄️declamatory",
  "❄️ventilation",
  "❄️tauromachy",
  "❄️cotransubstantiate",
  "❄️pome",
  "❄️underseas",
  "❄️triquadrantal",
  "❄️preconfinemnt",
  "❄️electroindustrial",
  "❄️selachostomous",
  "❄️nongolfer",
  "❄️mesalike",
  "❄️hamartiology",
  "❄️ganglioblast",
  "❄️unsuccessive",
  "❄️yallow",
  "❄️bacchanalianly",
  "❄️platydactyl",
  "❄️Bucephala",
  "❄️ultraurgent",
  "❄️penalist",
  "❄️catamenial",
  "❄️lynnhaven",
  "❄️unrelevant",
  "❄️lunkhead",
  "❄️metropolitan",
  "❄️hydro",
  "❄️outsoar",
  "❄️vernant",
  "❄️interlanguage",
  "❄️catarrhal",
  "❄️Ionicize",
  "❄️keelless",
  "❄️myomantic",
  "❄️booker",
  "❄️Xanthomonas",
  "❄️unimpeded",
  "❄️overfeminize",
  "❄️speronaro",
  "❄️diaconia",
  "❄️overholiness",
  "❄️liquefacient",
  "❄️Spartium",
  "❄️haggly",
  "❄️albumose",
  "❄️nonnecessary",
  "❄️sulcalization",
  "❄️decapitate",
  "❄️cellated",
  "❄️unguirostral",
  "❄️trichiurid",
  "❄️loveproof",
  "❄️amakebe",
  "❄️screet",
  "❄️arsenoferratin",
  "❄️unfrizz",
  "❄️undiscoverable",
  "❄️procollectivistic",
  "❄️tractile",
  "❄️Winona",
  "❄️dermostosis",
  "❄️eliminant",
  "❄️scomberoid",
  "❄️tensile",
  "❄️typesetting",
  "❄️xylic",
  "❄️dermatopathology",
  "❄️cycloplegic",
  "❄️revocable",
  "❄️fissate",
  "❄️afterplay",
  "❄️screwship",
  "❄️microerg",
  "❄️bentonite",
  "❄️stagecoaching",
  "❄️beglerbeglic",
  "❄️overcharitably",
  "❄️Plotinism",
  "❄️Veddoid",
  "❄️disequalize",
  "❄️cytoproct",
  "❄️trophophore",
  "❄️antidote",
  "❄️allerion",
  "❄️famous",
  "❄️convey",
  "❄️postotic",
  "❄️rapillo",
  "❄️cilectomy",
  "❄️penkeeper",
  "❄️patronym",
  "❄️bravely",
  "❄️ureteropyelitis",
  "❄️Hildebrandine",
  "❄️missileproof",
  "❄️Conularia",
  "❄️deadening",
  "❄️Conrad",
  "❄️pseudochylous",
  "❄️typologically",
  "❄️strummer",
  "❄️luxuriousness",
  "❄️resublimation",
  "❄️glossiness",
  "❄️hydrocauline",
  "❄️anaglyph",
  "❄️personifiable",
  "❄️seniority",
  "❄️formulator",
  "❄️datiscaceous",
  "❄️hydracrylate",
  "❄️Tyranni",
  "❄️Crawthumper",
  "❄️overprove",
  "❄️masher",
  "❄️dissonance",
  "❄️Serpentinian",
  "❄️malachite",
  "❄️interestless",
  "❄️stchi",
  "❄️ogum",
  "❄️polyspermic",
  "❄️archegoniate",
  "❄️precogitation",
  "❄️Alkaphrah",
  "❄️craggily",
  "❄️delightfulness",
  "❄️bioplast",
  "❄️diplocaulescent",
  "❄️neverland",
  "❄️interspheral",
  "❄️chlorhydric",
  "❄️forsakenly",
  "❄️scandium",
  "❄️detubation",
  "❄️telega",
  "❄️Valeriana",
  "❄️centraxonial",
  "❄️anabolite",
  "❄️neger",
  "❄️miscellanea",
  "❄️whalebacker",
  "❄️stylidiaceous",
  "❄️unpropelled",
  "❄️Kennedya",
  "❄️Jacksonite",
  "❄️ghoulish",
  "❄️Dendrocalamus",
  "❄️paynimhood",
  "❄️rappist",
  "❄️unluffed",
  "❄️falling",
  "❄️Lyctus",
  "❄️uncrown",
  "❄️warmly",
  "❄️pneumatism",
  "❄️Morisonian",
  "❄️notate",
  "❄️isoagglutinin",
  "❄️Pelidnota",
  "❄️previsit",
  "❄️contradistinctly",
  "❄️utter",
  "❄️porometer",
  "❄️gie",
  "❄️germanization",
  "❄️betwixt",
  "❄️prenephritic",
  "❄️underpier",
  "❄️Eleutheria",
  "❄️ruthenious",
  "❄️convertor",
  "❄️antisepsin",
  "❄️winterage",
  "❄️tetramethylammonium",
  "❄️Rockaway",
  "❄️Penaea",
  "❄️prelatehood",
  "❄️brisket",
  "❄️unwishful",
  "❄️Minahassa",
  "❄️Briareus",
  "❄️semiaxis",
  "❄️disintegrant",
  "❄️peastick",
  "❄️iatromechanical",
  "❄️fastus",
  "❄️thymectomy",
  "❄️ladyless",
  "❄️unpreened",
  "❄️overflutter",
  "❄️sicker",
  "❄️apsidally",
  "❄️thiazine",
  "❄️guideway",
  "❄️pausation",
  "❄️tellinoid",
  "❄️abrogative",
  "❄️foraminulate",
  "❄️omphalos",
  "❄️Monorhina",
  "❄️polymyarian",
  "❄️unhelpful",
  "❄️newslessness",
  "❄️oryctognosy",
  "❄️octoradial",
  "❄️doxology",
  "❄️arrhythmy",
  "❄️gugal",
  "❄️mesityl",
  "❄️hexaplaric",
  "❄️Cabirian",
  "❄️hordeiform",
  "❄️eddyroot",
  "❄️internarial",
  "❄️deservingness",
  "❄️jawbation",
  "❄️orographically",
  "❄️semiprecious",
  "❄️seasick",
  "❄️thermically",
  "❄️grew",
  "❄️tamability",
  "❄️egotistically",
  "❄️fip",
  "❄️preabsorbent",
  "❄️leptochroa",
  "❄️ethnobotany",
  "❄️podolite",
  "❄️egoistic",
  "❄️semitropical",
  "❄️cero",
  "❄️spinelessness",
  "❄️onshore",
  "❄️omlah",
  "❄️tintinnabulist",
  "❄️machila",
  "❄️entomotomy",
  "❄️nubile",
  "❄️nonscholastic",
  "❄️burnt",
  "❄️Alea",
  "❄️befume",
  "❄️doctorless",
  "❄️Napoleonic",
  "❄️scenting",
  "❄️apokreos",
  "❄️cresylene",
  "❄️paramide",
  "❄️rattery",
  "❄️disinterested",
  "❄️idiopathetic",
  "❄️negatory",
  "❄️fervid",
  "❄️quintato",
  "❄️untricked",
  "❄️Metrosideros",
  "❄️mescaline",
  "❄️midverse",
  "❄️Musophagidae",
  "❄️fictionary",
  "❄️branchiostegous",
  "❄️yoker",
  "❄️residuum",
  "❄️culmigenous",
  "❄️fleam",
  "❄️suffragism",
  "❄️Anacreon",
  "❄️sarcodous",
  "❄️parodistic",
  "❄️writmaking",
  "❄️conversationism",
  "❄️retroposed",
  "❄️tornillo",
  "❄️presuspect",
  "❄️didymous",
  "❄️Saumur",
  "❄️spicing",
  "❄️drawbridge",
  "❄️cantor",
  "❄️incumbrancer",
  "❄️heterospory",
  "❄️Turkeydom",
  "❄️anteprandial",
  "❄️neighborship",
  "❄️thatchless",
  "❄️drepanoid",
  "❄️lusher",
  "❄️paling",
  "❄️ecthlipsis",
  "❄️heredosyphilitic",
  "❄️although",
  "❄️garetta",
  "❄️temporarily",
  "❄️Monotropa",
  "❄️proglottic",
  "❄️calyptro",
  "❄️persiflage",
  "❄️degradable",
  "❄️paraspecific",
  "❄️undecorative",
  "❄️Pholas",
  "❄️myelon",
  "❄️resteal",
  "❄️quadrantly",
  "❄️scrimped",
  "❄️airer",
  "❄️deviless",
  "❄️caliciform",
  "❄️Sefekhet",
  "❄️shastaite",
  "❄️togate",
  "❄️macrostructure",
  "❄️bipyramid",
  "❄️wey",
  "❄️didynamy",
  "❄️knacker",
  "❄️swage",
  "❄️supermanism",
  "❄️epitheton",
  "❄️overpresumptuous"
  ]

public func run_SortStringsUnicode(_ n: Int) {
  for _ in 1...n {
    benchSortStrings(unicodeWords)
  }
}
