//===--- FloatingPointParsing.swift -----------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2019 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

// This test verifies the performance of parsing a floating point value
// from a String.

import TestsUtils

public let benchmarks = [
  BenchmarkInfo(
    name: "ParseFloat.Float.Exp",
    runFunction: run_ParseFloatExp,
    tags: [.validation, .api, .runtime, .String]),

  BenchmarkInfo(
    name: "ParseFloat.Double.Exp",
    runFunction: run_ParseDoubleExp,
    tags: [.validation, .api, .runtime, .String]),

  BenchmarkInfo(
    name: "ParseFloat.Float80.Exp",
    runFunction: run_ParseFloat80Exp,
    tags: [.validation, .api, .runtime, .String]),

  BenchmarkInfo(
    name: "ParseFloat.Float.Uniform",
    runFunction: run_ParseFloatUniform,
    tags: [.validation, .api, .runtime, .String]),

  BenchmarkInfo(
    name: "ParseFloat.Double.Uniform",
    runFunction: run_ParseDoubleUniform,
    tags: [.validation, .api, .runtime, .String]),

  BenchmarkInfo(
    name: "ParseFloat.Float80.Uniform",
    runFunction: run_ParseFloat80Uniform,
    tags: [.validation, .api, .runtime, .String]),
]

/// The 'Exp' test:

/// Generates values following a truncated exponential distribution on 0...1000, 
/// each rounded to a number of significant digits uniformly selected from 
/// 1...6 (Float), 1...15 (Double) and 1...18 (Float80). 
/// This is a good representative sample of "well-scaled" values humans actually
/// write in programs; in particular, it includes good coverage for integer values
/// and values with short decimal parts.

// func genExpDistributedFloat(significantDigits: Int) -> String {
//     let value = exp(Float80.random(in: 0...log(1000.0)))
//     return String(format: "%.\(significantDigits)f", Double(value))
// }

/// Each workload contains 100 elements generated from the above function.

let floatExpWorkload = [
  "840.23812", "15.9082", "319.784", "13.14857", "5.42759",
  "901.771", "339.6", "7.27", "126.88", "326.08923", "18.2804",
  "189.8467", "330.628", "8.272461", "19.337", "12.0082",
  "29.6085", "27.4508", "5.14013", "725.388972", "124.102",
  "13.315556", "21.288", "45.4670", "88.7246", "6.16",
  "1.506155", "2.16918", "779.120", "22.89751", "15.33395",
  "59.575817", "2.73305", "203.04", "321.08", "148.419",
  "249.675", "579.453", "222.2", "153.62548", "305.501",
  "96.3", "28.55095", "145.377249", "2.53048", "1.0",
  "293.2052", "2.1", "814.091552", "157.45375", "15.0",
  "1.304", "8.88", "799.458180", "11.3", "1.5645",
  "25.69062", "569.28", "2.6464", "173.792970", "6.25",
  "344.54", "2.09610", "3.547", "409.860",
  "65.866038", "3.64", "2.7", "62.304", "67.7729",
  "19.0638", "2.8", "8.89508", "20.04", "1.054648",
  "3.5479", "5.203191", "52.11968", "326.39", "43.027",
  "82.15620", "178.519010", "1.3", "5.40", "387.41",
  "500.5", "5.96", "1.7740", "96.48818", "889.583",
  "96.92098", "6.760", "199.441", "510.905", "307.726",
  "87.9055", "11.7", "6.487537", "119.1", "5.8"
]

let doubleExpWorkload = [
  "339.91", "662.95264138", "590.3312546", "44.58449489426322", "1.61025794592574",
  "266.29744353690", "34.14542", "144.968532801343116", "1.790721486700", "609.0086620368", 
  "1.79655054299720", "138.906589772754131", "1.3663343399933", "3.018134440821", "14.7117491216652", 
  "97.0", "28.630149748915", "5.4", "29.7639", "4.520193", 
  "43.574150740143", "21.26131766279", "8.332799002305356", "70.267", "792.71364", 
  "987.54396679201125", "301.4300850", "707.7375522552", "3.350899864", "41.99", 
  "78.051", "2.5251720", "28.171984464058", "6.115905648", "31.7", 
  "2.90316715974825", "3.49235954808", "13.76", "3.2", "32.465845", 
  "460.84828154", "1.0268532933", "79.607954332859777", "173.25041830", "49.0888", 
  "23.2264", "130.018100263319411", "301.8", "1.707", "2.220651", 
  "11.9744168", "13.50610", "83.276270711070708", "1.207", "11.507359", 
  "887.81742700364475", "46.8051834077896", "174.367548608815781", "19.8671230", "5.0432", 
  "3.927758869", "1.6393532610", "232.5", "17.77417107", "94.1453702714822", 
  "746.2908548477199", "28.9832376533851", "1.7432454399", "96.15", "484.00318952955", 
  "14.90238658606421", "243.704906310113", "29.5307828313724", "19.200405681021813", "24.1717308", 
  "2.7453085963749", "2.856", "677.940804020", "221.57146165", "31.5891",
  "350.74206", "3.06588790579", "171.404", "46.106851", "590.3917781324", 
  "829.052362588", "2.32566173", "7.0098461186", "306.11702882946065", "17.4345632593715", 
  "899.3720935006996", "108.31212", "3.703786329", "48.81100281168", "27.41503", 
  "169.15383", "1.978568", "3.68994208914", "29.4322212731893", "4.8719352"
]

let float80ExpWorkload = [
  "6.77555058241", "147.74", "6.03", "507.6033533228630859", "100.7", 
  "11.46", "3.264282911002", "85.7858847516568", "34.4300032", "4.9957", 
  "198.3958760", "87.67549428326", "205.373035358974988", "27.93", "831.999235134615", 
  "46.886425395152", "5.77789", "89.564807063568139256", "3.85398054", "1.021", 
  "592.504", "1.802084399", "486.4972328197284", "5.22490700277", "29.917340694598", 
  "181.48302719", "75.74373681671689", "30.48161373580532", "1.24544077730", "1.2", 
  "10.426", "55.37308819", "1.87502584", "3.1486", "9.2794677633", 
  "24.59858334079387632", "20.2896643", "4.6", "9.017375215972097", "163.1", 
  "5.50921892286", "9.93079", "13.320762298", "3.194056167117689693", "211.6671712762052380", 
  "347.0356", "209.66", "13.170751077618", "34.568", "330.5",
  "41.388619513", "625.5176", "7.3199072376", "2.40", "334.210711370", 
  "10.790469414612726240", "2.051865559526", "374.34104357856199", "1.444672057", "182.15138835680261", 
  "1.549898719", "2.2", "3.5960392119", "220.3919", "128.45273", 
  "955.052925", "441.738166", "21.365", "28.5534801613", "124.1", 
  "449.252220495138", "608.587461250119532", "107.88473705800", "2.085422", "2.5", 
  "121.0005042322", "5.4402224803576962", "90.46", "40.646406742621564945", "70.79133", 
  "4.59801271236", "1.893481804014", "17.52333", "1.3195086968", "46.70781", 
  "14.59891931096853", "402.75", "158.0343", "7.152603207", "7.3637945245", 
  "15.6", "545.740720800777012", "242.172569", "1.73940415531105", "151.14631658", 
  "26.5256384449273490", "135.236", "12.886101", "47.596174", "1.831407510"
]

/// The 'Uniform' test:

/// Generates values made up of uniform decimal significands with 9, 17 and 21 
/// digits and exponents uniform on the valid range. This is a stress test 
/// for rounding, and covers all the worst cases for tables generated by programs.
/// The exponent range is -45...38 for Floats, -324...308 for Doubles, and 
/// -4951...4932 for Float80.
// func genUniformFloat(significandDigits: Int, exponentRange: ClosedRange<Int>) -> String {
//     let afterDecimalPoint = (0..<significandDigits).map { _ in String(Int.random(in: 0...9)) }.joined()
//     let sign = ["", "-", "+"].randomElement()!
//     return "\(sign)\(Int.random(in: 1...9)).\(afterDecimalPoint)e\(exponentRange.randomElement()!)"
// }

/// Each workload contains 100 elements generated from the above function,
/// along with five inf/nan/invalid tests.

let floatUniformWorkload = [
  "+1.253892113e-32", "+5.834995733e-41", "5.262096122e-17", "+4.917053817e-37", "8.535406338e-34", 
  "2.152837278e10", "-5.212739043e-16", "+9.799669278e-27", "+8.678995824e-6", "-5.965172043e26", 
  "-8.420371743e-10", "1.968856825e-8", "1.521071839e-19", "+1.048728457e-15", "-5.657219736e10", 
  "-7.664702071e-34", "+3.282753652e15", "-5.032906928e26", "-3.024685077e29", "+7.972511327e-22",
  "-3.941547478e-19", "-2.424139629e4", "+1.222228289e2", "+9.872675983e4", "+8.505353502e-43", 
  "7.315998745e12", "-2.879924852e-38", "5.567658036e21", "5.751274848e-39", "-2.098314138e8", 
  "-2.295512125e-13", "-9.261977483e3", "-7.717209557e26", "+6.563403126e38", "-6.988491389e-45", 
  "3.318738022e21", "5.534799334e16", "7.236228752e0", "+6.134225015e-26", "-1.801539431e-3", 
  "-8.763001980e37", "-4.810306387e-30", "-8.902359860e5", "-4.654434339e17", "4.103749478e11", 
  "+1.624005001e0", "+6.810516979e-8", "+4.509584369e0", "7.115062319e15", "-3.275835669e24", 
  "-2.225975117e17", "+9.765601399e-27", "9.271660327e13", "-7.957489335e4", "+1.279815043e-30", 
  "-6.140235958e-3", "+2.925804509e-11", "-2.902478935e-36", "+2.870673595e-37", "+8.788759496e-20", 
  "+7.279719040e13", "-9.516217056e20", "2.306171183e21", "-2.655002939e22", "-8.678845371e32", 
  "6.086700440e20", "+6.768130785e12", "1.675300343e11", "+8.156722194e-16", "-6.040145895e35", 
  "-6.928961416e-26", "-5.119323586e27", "-4.566748978e-5", "-3.394147506e-7", "6.171944831e-19", 
  "+8.883811091e-11", "-3.390953266e-44", "-1.912771939e-7", "+8.506344503e-23", "-3.437927939e21", 
  "-3.515331652e1", "8.610555796e9", "2.340195107e-20", "+9.018470750e-42", "+8.321248518e27", 
  "-6.959418594e32", "-5.342372027e21", "+2.744186726e-24", "9.948785682e-37", "+6.310898794e-6", 
  "-2.477472268e16", "8.590689853e1", "+7.811554461e-24", "+1.508151084e17", "3.071428780e-7",
  "4.545415458e-9", "7.075010280e11", "+8.616159616e-29", "4.613265893e-10", "7.770003218e-33",
  "+inf", "Invalid", "nan", "NAN", "snan"
]

let doubleUniformWorkload = [
  "-5.099347166e-78", "3.584151187e-255", "-7.555973282e17", "+6.217083912e-164", "-6.063585254e8", 
  "-5.374097872e-252", "2.688487062e208", "+1.030241589e-272", "2.120162986e-50", "-2.617585299e-69",
  "8.560348373e282", "4.323584117e-168", "5.899559411e215", "+3.630548207e220", "-8.420738030e-73", 
  "-6.832994185e-129", "-9.751163985e90", "-3.923652856e222", "-5.337925604e-12", "+4.360166940e-75", 
  "+6.207675792e-164", "-8.275068142e-195", "+3.318047866e-71", "-9.162983038e197", "+9.330784575e-147", 
  "9.208831616e-322", "-5.688921689e270", "+2.871328480e-258", "-8.071161900e261", "-4.191368176e222", 
  "+5.792498976e-167", "5.835667380e235", "+9.402094050e6", "2.079961640e180", "+4.037655106e86", 
  "-1.267141442e106", "+2.361395667e138", "+2.101128051e142", "5.301258292e42", "-8.822348131e-280", 
  "-3.775817054e208", "-5.080405399e93", "+9.686534601e-77", "4.586641905e-175", "3.135576124e77", 
  "4.688137331e138", "+6.893752397e-189", "6.812711913e278", "3.812796443e115", "+3.744289703e7", 
  "+5.932500106e157", "+4.846313692e16", "-8.881077959e-290", "+1.535288334e-275", "7.250519901e-75",
  "2.787321374e41", "+3.519991812e-183", "+7.589975072e79", "5.848655303e122", "-3.328972789e161", 
  "-2.190752323e104", "8.864042297e147", "+8.584292050e-239", "-7.972816817e219", "9.352363266e-99", 
  "+9.435082870e83", "+4.297178676e-122", "+8.699490866e-300", "+5.562137865e-57", "+9.063928579e-92", 
  "2.743744209e82", "+9.960619645e-39", "-1.962409303e90", "-4.371512402e-287", "4.790326011e-137", 
  "+1.295921853e-273", "9.137852847e-96", "2.934099551e35", "-8.176544555e-65", "-7.098452895e-220", 
  "+6.665950037e103", "+9.297850239e-254", "-8.529931750e-216", "-9.541329616e-324", "-4.761545594e148", 
  "3.507981964e-314", "-3.745430665e-89", "8.799599593e137", "8.736852803e181", "+8.328604940e291", 
  "-2.207530122e-202", "-4.259268955e-271", "+1.633874531e-225", "+6.167040542e211", "-2.632062534e-52", 
  "-2.296105107e69", "4.935681094e205", "+4.696549581e-117", "+5.032486364e-105", "6.718199716e48",
  "-inf", "Invalid", "nan", "NAN", "snan"
]

let float80UniformWorkload = [
  "-4.252484660e-718", "+7.789429046e-2371", "-2.914666745e-2986", "8.287524889e4854", "+4.792668657e-3693", 
  "-5.187192176e-756", "+2.452045856e-2309", "+7.133017664e-1055", "-6.749416450e-3803", "+6.808813002e-1771", 
  "+7.355881301e3600", "-9.209599679e4848", "+4.142753329e-3268", "3.950199398e-863", "-3.170838470e-4779", 
  "9.354087375e718", "2.769339314e-3567", "+1.889983496e3717", "+9.912545495e-2419", "-6.284830753e2041", 
  "+9.061812480e3682", "6.551587084e1912", "+6.485690673e-1591", "-3.522511676e-2344", "-6.846303741e-2830", 
  "-7.995042826e896", "6.268882688e937", "-3.776199447e-4134", "-2.353057456e801", "5.875638854e-3889", 
  "1.245553459e814", "4.593376472e-2139", "-8.277421726e1497", "+1.606488487e1221", "6.433878090e-2220", 
  "-2.515502287e2543", "-4.347251565e-1330", "1.101969004e-4525", "-7.602718782e-4037", "-7.289917475e-4547", 
  "+1.920523398e2160", "-8.279745071e4493", "+6.383586105e-3771", "-3.784311609e-1828", "-1.193395125e1296", 
  "-2.012225848e1954", "+2.238968379e-1455", "6.331805949e-2127", "-7.584066626e-717", "-9.040205012e1614", 
  "+1.953864302e4255", "+5.103307458e528", "9.491061048e531", "+2.165292603e54", "-6.471469370e654", 
  "-9.275772875e4856", "-4.070772582e1488", "+2.063335882e-2112", "+4.853853159e-3841", "-9.058842001e3955", 
  "2.897215261e3511", "1.389094534e-384", "-9.749518987e1602", "+3.103609399e-3559", "-7.156672429e3879",
  "9.385923023e2310", "7.593508637e-2100", "-2.656332678e2833", "6.143253335e-340", "5.794793573e2972", 
  "3.869110366e2609", "+2.884288161e-1513", "-5.316488863e4336", "4.948197725e-3829", "3.755250612e-3011", 
  "+1.550352355e-767", "-1.625440305e-4354", "+3.086601758e-929", "6.042347288e-357", "-6.176954358e3288", 
  "1.594019881e480", "-8.613112966e4863", "+9.864076072e2498", "3.540199292e-821", "+3.770221960e4915", 
  "-4.115310178e-3958", "-8.343495037e-4238", "1.165941010e-317", "6.039736339e-693", "6.912425733e1200", 
  "-9.307038635e335", "-4.656175810e-637", "-6.342156592e-3848", "-1.221105578e1780", "+6.097066301e-2159", 
  "-5.823123345e1389", "+5.404800369e1379", "2.988649766e-736", "-3.733572715e2405", "-1.597565079e-377",
  "+inf", "Invalid", "nan", "NAN", "snan"
]

@inline(__always)
public func parseFloatWorkload<T : LosslessStringConvertible>(_ n: Int, workload: [String], type: T.Type) {
  for _ in 0..<n {
    for element in workload {
      let f = T(element)
      blackHole(f)
    }
  }
}

@inline(never)
public func run_ParseFloatExp(_ n: Int) {
  parseFloatWorkload(n, workload: floatExpWorkload, type: Float.self)
}

@inline(never)
public func run_ParseDoubleExp(_ n: Int) {
  parseFloatWorkload(n, workload: doubleExpWorkload, type: Double.self)
}

@inline(never)
public func run_ParseFloat80Exp(_ n: Int) {
#if canImport(Darwin) || os(Linux)
// On Darwin, long double is Float80 on x86, and Double otherwise.
// On Linux, Float80 is at aleast available on x86.
#if arch(x86_64) || arch(i386)
  parseFloatWorkload(n, workload: float80ExpWorkload, type: Float80.self)
#endif // x86
#endif // Darwin/Linux
}

@inline(never)
public func run_ParseFloatUniform(_ n: Int) {
  parseFloatWorkload(n, workload: floatUniformWorkload, type: Float.self)
}

@inline(never)
public func run_ParseDoubleUniform(_ n: Int) {
  parseFloatWorkload(n, workload: doubleUniformWorkload, type: Double.self)
}

@inline(never)
public func run_ParseFloat80Uniform(_ n: Int) {
#if canImport(Darwin) || os(Linux)
// On Darwin, long double is Float80 on x86, and Double otherwise.
// On Linux, Float80 is at aleast available on x86.
#if arch(x86_64) || arch(i386)
  parseFloatWorkload(n, workload: float80UniformWorkload, type: Float80.self)
#endif // x86
#endif // Darwin/Linux
}
