use core:lang;
use lang:bs;
use lang:bs:macro;
use core:asm;

/**
 * Handler expression.
 */
class HandleExpr extends Expr {
	Expr body;
	Handler handler;
	Expr handlerCtor;

	init(SrcPos pos, Block parent, SExpr body, Expr handler) {
		var h = if (type = handler.result.type.type as Handler) {
			type;
		} else {
			throw SyntaxError(handler.pos, "The expression after 'with' must evaluate to an effect. It is a ${handler.result.type}.");
		};

		var handlerFn = fnType([h.result, h]);
		SStr paramName("@handler");

		// Inject the return clause and a return point into the lambda:
		body = WrapLambda(body, paramName, h.result, h.returnClause);
		LambdaExpr lambdaBody(body.pos, parent, [ValParam(h, paramName)], body);

		init(pos) {
			body = lambdaBody.expectCastTo(handlerFn, parent.scope);
			handler = h;
			handlerCtor = handler;
		}
	}

	// Helper class to wrap the entire body of the lambda in a ReturnPoint, and optionally pass the
	// result value through a function as well.
	private class WrapLambda extends SExpr {
		SExpr body;
		HandlerClause? wrap;
		SStr paramName;
		Value finalResult;

		init(SExpr body, SStr paramName, Value result, HandlerClause? wrap) {
			init() {
				body = body;
				wrap = wrap;
				paramName = paramName;
				finalResult = result;
			}
			pos = body.pos;
		}

		Expr transform(Block block) : override {
			Value result = finalResult;

			// If we have a function, we should use it:
			if (wrap) {
				if (wrap.handler.params.count > 1)
					result = wrap.handler.params[1];
				else
					result = Value();
			}

			// Wrap in a return point to make returns work as we would expect.
			ReturnPoint p(pos, block, result);
			p.body = body.transform(p);


			if (wrap) {
				// A function to wrap:
				unless (effectVar = block.variable(SimplePart(paramName.v)))
					throw InternalError("Failed to find effect variable.");

				Actuals actuals(LocalVarAccess(pos, effectVar));

				if (result.any) {
					actuals.add(p);
					return FnCall(pos, block.scope, wrap.handler, actuals);
				} else {
					ExprBlock seq(pos, block);
					seq.add(p);
					seq.add(FnCall(pos, block.scope, wrap.handler, actuals));
					return seq;
				}
			} else {
				// No function to wrap:
				return p;
			}
		}
	}

	void code(CodeGen code, CodeResult result) : override {
		unless (frameType = named{}.find("HandlerFrame", handler, Scope()) as Type)
			throw InternalError("Failed to find HandlerFrame!");

		Value bodyFnType = fnType([handler.result, handler]);
		unless (mainFn = frameType.find("main", [frameType, bodyFnType], Scope()) as Function)
			throw InternalError("Failed to find the main function of the frame.");

		// Create the handler object.
		CodeResult handlerObj(handler, code.block);
		handlerCtor.code(code, handlerObj);

		// Create a frame:
		var frame = {
			unless (ctorFn = frameType.find("__init", [frameType, handler], Scope()) as Function)
				throw InternalError("Failed to find the constructor of the frame.");
			var handlerAccess = BareVarAccess(SrcPos(), handler, handlerObj.location(code));
			CtorCall call(SrcPos(), Scope(), ctorFn, Actuals(handlerAccess));

			CodeResult frame(frameType, code.block);
			call.code(code, frame);
			frame.location(code);
		};

		// Set "prev". Note: Representation of pointer-maybe is a pointer
		{
			// Get our handler frame:
			code.l << fnCall(named{currentHandlerFrame}.ref, false, ptrDesc, ptrC);

			// Set it.
			var prevVar = named{HandlerFrame:prev<HandlerFrame>};
			code.l << mov(ptrA, frame);
			code.l << mov(ptrRel(ptrA, prevVar.offset), ptrC);
		}

		// Set "effects"
		{
			Map<Effect, FnBase> toHandle;
			for (h in handler.effectClauses)
				toHandle.put(h.effect, h.handler.pointer);

			var effectsVar = named{HandlerFrame:effects<HandlerFrame>};
			code.l << mov(ptrA, frame);
			code.l << mov(ptrRel(ptrA, effectsVar.offset), objPtr(toHandle));
		}

		// Create the lambda function:
		CodeResult lambda(bodyFnType, code.block);
		body.code(code, lambda);

		// Call 'main' on the desired thread. Note: not using the result calls detach automatically.
		mainFn.asyncLocalCall(code, [frame, lambda.location(code)], CodeResult());

		// Wait for the result to be ready:
		code.l << fnParam(ptrDesc, frame);
		code.l << fnCall(named{HandlerFrame:wait<HandlerFrame>}.ref, true);

		// Note: Due to how maybe types work, we can actually use it as if it was a regular value.
		// We know that it is set at this point.

		if (handler.result.any) {
			unless (frameResult = frameType.find("result", frameType, Scope()) as MemberVar)
				throw InternalError("Failed to find 'result' in HandlerFrame!");

			if (result.needed()) {
				// Just copy the result.
				if (handler.result.isAsmType) {
					code.l << mov(ptrA, frame);
					code.l << mov(result.location(code), xRel(handler.result.size, ptrA, frameResult.offset));
				} else {
					code.l << mov(ptrA, frame);
					code.l << add(ptrA, ptrConst(frameResult.offset));
					code.l << lea(ptrC, result.location(code));
					code.l << fnParam(ptrDesc, ptrC);
					code.l << fnParam(ptrDesc, ptrA);
					code.l << fnCall(handler.result.copyCtor, true);
					result.created(code);
				}
			}
		}
	}

	ExprResult result() : override {
		ExprResult(handler.result);
	}

	void toS(StrBuf to) : override {
		to << "handle " << body << " with " << handler.identifier();
	}
}
