use core:unsafe;

/**
 * A stored continuation.
 *
 * It is essentially an array of stored stacks that can be resumed.
 */
class Continuation {
	// Create an empty continuation.
	init(EffectFrame resume) {
		init {
			resumeFrame = resume;
		}
	}

	// All stacks that are contained in the continuation.
	private Stack[] stacks;

	// The object used to resume execution.
	private EffectFrame resumeFrame;

	// A single stack in the execution.
	private value Stack {
		// Stored stack.
		StackMirror mirror;

		// The handler frame that launched the stack.
		HandlerFrame frame;

		// Create.
		init(HandlerFrame frame) {
			init {
				frame = frame;
				mirror = StackMirror:save(frame.threadId);
			}
		}
	}

	// Add an element to the stack.
	void add(HandlerFrame frame) {
		stacks << Stack(frame);
	}

	// Get the resume frame so that it is possible to set the result there.
	EffectFrame frame() {
		resumeFrame;
	}

	// Resume. Returns the HandlerFrame that contains the result.
	HandlerFrame resume() {
		if (stacks.empty)
			throw InternalError("Trying to resume an empty continuation!");

		// Previous handler on this thread:
		HandlerFrame? prevFrame = currentHandlerFrame();

		// Resume all stacks. For all but the last, we need to ask the HandlerFrame object to wait
		// once more, otherwise the thread will start too early.
		Bool last = true;
		for (Nat i = stacks.count; i > 0; i--) {
			var stack = stacks[i - 1];
			if (!last) {
				stack.frame.resurrected();
			}

			// Link it into the chain of handlers to make nested invocations work as expected.
			stack.frame.prev = prevFrame;
			setHandlerFrame(stack.frame.threadId, stack.frame);

			// Finally, we can resume the thread.
			stack.mirror.restore();

			prevFrame = stack.frame;
			last = false;
		}

		// Finally, wait and return for the last frame.
		HandlerFrame lastFrame = stacks.last.frame;
		lastFrame.wait();
		return lastFrame;
	}
}
