use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * Type used to represent a row with typed data from an iterator.
 *
 * Note: We use ValParam to associate strings to types.
 */
class TypedRow extends Type {
	init(ValParam[] types) {
		init("<row>", TypeFlags:typeClass) {}

		parentLookup = named{};

		// Add variables.
		MemberVar[] vars;
		for (t in types) {
			MemberVar v(t.name, t.type, this);
			vars << v;
			add(v);
		}

		// Add a constructor that reads data from a Row instance.
		BSTreeCtor ctor([thisParam(this), ValParam(named{Row}, "row")], SrcPos());
		CtorBody body(ctor, Scope(this));

		unless (rowVar = body.variable(SimplePart("row")))
			throw InternalError("Failed to find the local variable 'row'.");
		LocalVarAccess row(SrcPos(), rowVar);

		{
			InitBlock init(SrcPos(), body, null);
			for (i, t in types) {
				init.init(Initializer(SStr(t.name), getColumn(body, row, t.type, i)));
			}
			body.add(init);
		}

		ctor.body = body;
		add(ctor);

		// To string.
		add(toString(vars));

		// Other default operators.
		add(TypeDeepCopy(this));
		add(TypeCopyCtor(this));
	}

	// Generate a to-string function.
	private Function toString(MemberVar[] vars) {
		BSTreeFn fn(Value(), SStr("toS"), [thisParam(this), ValParam(named{StrBuf}, "to")], null);
		FnBody body(fn, Scope(this, BSLookup()));

		unless (toVar = body.variable(SimplePart("to")))
			throw InternalError("Failed to find local variable 'to'.");
		unless (thisVar = body.variable(SimplePart("this")))
			throw InternalError("Failed to find local variable 'this'.");
		LocalVarAccess to(SrcPos(), toVar);
		LocalVarAccess me(SrcPos(), thisVar);

		for (i, var in vars) {
			Str s = var.name + ": ";
			if (i > 0)
				s = ", " + s;
			body.add(namedExpr(body, SrcPos(), "<<", to, Actuals(StrLiteral(SrcPos(), s))));

			MemberVarAccess access(SrcPos(), me, var, true);
			Expr value = access;
			if (access.result.type.type !is named{Str})
				value = namedExpr(body, SrcPos(), "toS", access);
			body.add(namedExpr(body, SrcPos(), "<<", to, Actuals(value)));
		}

		fn.body = body;
		fn;
	}
}

// Get an expression for extracting a particular column.
private Expr getColumn(Block block, Expr row, Value type, Nat id) on Compiler {
	if (isMaybe(type)) {
		var fn = named{Row:isNull<Row, Nat>};
		Actuals params;
		params.add(row);
		params.add(NumLiteral(SrcPos(), id.long));
		If check(block, FnCall(SrcPos(), Scope(), fn, params));

		CondSuccess ok(SrcPos(), block, check.condition);
		ok.set(callCtor(block, type, Actuals()));
		check.trueCode = ok;

		check.falseCode = callCtor(block, type, Actuals(getPlainColumn(row, unwrapMaybe(type), id)));

		check;
	} else {
		getPlainColumn(row, type, id);
	}
}

// Call a constructor.
private Expr callCtor(Block block, Value type, Actuals actuals) on Compiler {
	unless (t = type.type)
		throw InternalError("Could not find a proper type.");

	var params = actuals.values();
	params.insert(0, type);
	unless (fn = t.find("__init", params, Scope()) as Function)
		throw InternalError("Could not find a constructor for ${type}");

	return CtorCall(SrcPos(), Scope(), fn, actuals);
}

// Get an expression for extracting a particular column. Does not handle Maybe<>
private Expr getPlainColumn(Expr row, Value type, Nat id) on Compiler {
	var fn = getColumnFn(type.type);
	Actuals params;
	params.add(row);
	params.add(NumLiteral(SrcPos(), id.long));
	return FnCall(SrcPos(), Scope(), fn, params);
}


/**
 * Type used to denote an iterator that produces instances of a TypedRow class.
 *
 * Follows the "simple" iterator type that only has a "next" function.
 */
class TypedIter extends Type {
	init(ValParam[] types) {
		init("<iter>", TypeFlags:typeValue) {}

		parentLookup = named{};

		TypedRow toCreate = getTypedRow(types);
		MemberVar iter("base", Value(named{Statement:Result}), this);
		iter.visibility = typePrivate();
		add(iter);

		addCtor();
		addNext(toCreate, iter);
		addIter();
		add(TypeCopyCtor(this));
		add(TypeAssign(this));
		add(TypeDefaultDtor(this));
	}

	// Add constructor.
	private void addCtor() {
		BSTreeCtor ctor([thisParam(this), ValParam(named{Statement:Result}, "result")], SrcPos());
		CtorBody body(ctor, Scope(this, BSLookup()));

		unless (resultVar = body.variable(SimplePart("result")))
			throw InternalError("Failed to find local variable 'result'.");
		LocalVarAccess result(SrcPos(), resultVar);

		InitBlock init(SrcPos(), body, null);
		init.init(Initializer(SStr("base"), result));
		body.add(init);

		ctor.body = body;
		add(ctor);
	}

	// Add "next" member.
	private void addNext(TypedRow row, MemberVar iter) {
		BSTreeFn fn(wrapMaybe(Value(row)), SStr("next"), [thisParam(this)], null);
		FnBody body(fn, Scope(this, BSLookup()));

		unless (thisVar = body.variable(SimplePart("this")))
			throw InternalError("Failed to find local variable 'this'.");
		LocalVarAccess me(SrcPos(), thisVar);

		unless (typeCtor = row.find("__init", [Value(row), Value(named{Row})], Scope()) as Function)
			throw InternalError("Failed to find the constructor of the generated row-type.");

		// Call "next" and see if it returned null.
		WeakMaybeCast cast(namedExpr(body, SrcPos(), "next", MemberVarAccess(SrcPos(), me, iter, true)));
		cast.name(SStr("x"));
		If check(body, cast);
		body.add(check);

		CondSuccess success(SrcPos(), body, cast);
		check.trueCode = success;
		if (created = cast.result) {
			CtorCall c(SrcPos(), Scope(), typeCtor, Actuals(LocalVarAccess(SrcPos(), created)));
			success.set(Return(SrcPos(), success, c));
		}

		// Just return null otherwise.
		body.add(NullExpr(SrcPos()));

		fn.body = body;
		add(fn);
	}

	// Add "iter" member. We just return ourselves. Otherwise, we can't use it in foreach-loops. The
	// "clean" option would be to make a separate class that is "some returned rows" that can be
	// iterated. But this makes more sense overall.
	private void addIter() {
		BSTreeFn fn(Value(this), SStr("iter"), [thisParam(this)], null);
		FnBody body(fn, Scope(this));

		unless (thisVar = body.variable(SimplePart("this")))
			throw InternalError("Failed to find local variable 'this'.");
		LocalVarAccess me(SrcPos(), thisVar);
		body.add(me);

		fn.body = body;
		add(fn);
	}
}

// Global cache of data types.
private Map<Str, TypedRow> createdRowTypes on Compiler;
private Map<Str, TypedIter> createdIterTypes on Compiler;

// Compute a key for use in the map.
private Str typeKey(ValParam[] types) on Compiler {
	StrBuf s;
	for (x in types) {
		s << "!" << x.type << ":" << x.name;
	}
	return s.toS;
}

// Get a TypedRow instance.
TypedRow getTypedRow(ValParam[] types) on Compiler {
	Str key = typeKey(types);
	if (createdRowTypes.has(key))
		return createdRowTypes.get(key);

	TypedRow created(types);
	createdRowTypes.put(key, created);
	return created;
}

// Get a TypedIter instance.
TypedIter getTypedIter(ValParam[] types) on Compiler {
	Str key = typeKey(types);
	if (createdIterTypes.has(key))
		return createdIterTypes.get(key);

	TypedIter created(types);
	createdIterTypes.put(key, created);
	return created;
}
