use std::collections::HashMap;
use std::collections::hash_map::Entry;
use std::sync::Arc;
use std::sync::Mutex;
use std::sync::MutexGuard;
use std::sync::Weak;
use std::ops::Deref;
use std::ops::DerefMut;
use std::path::Path;
use std::path::PathBuf;

use anyhow::Context;

use sequoia_openpgp as openpgp;
use openpgp::Cert;
use openpgp::cert::prelude::*;
use openpgp::crypto::Decryptor;
use openpgp::crypto::mpi;
use openpgp::crypto::Password;
use openpgp::crypto::SessionKey;
use openpgp::crypto::Signer;
use openpgp::Result;
use openpgp::Fingerprint;
use openpgp::packet;
use openpgp::parse::Parse;
use openpgp::types::HashAlgorithm;
use openpgp::types::PublicKeyAlgorithm;

use sequoia_keystore_backend as backend;
use backend::Backend as _;
use backend::KeyHandle as _;
use backend::Error;

#[cfg(test)]
mod testdata;

/// Known soft keys.
#[derive(Clone)]
pub struct Backend(Arc<Mutex<BackendInternal>>);

struct BackendInternal {
    home: PathBuf,

    // Map from certificate fingerprint to Device.
    certs: HashMap<Fingerprint, Device>,

    // Map from key to certificate fingerprint.
    keys: HashMap<Fingerprint, Key>,
}

// If we do: backend.0.lock().unwrap().method(), then method doesn't
// have a reference to the `Arc`, which we sometimes need.  This
// conveniently encapsulates an `Arc` and a `MutexGuard`.
struct BackendRef<'a>(MutexGuard<'a, BackendInternal>, Arc<Mutex<BackendInternal>>);

impl<'a> Deref for BackendRef<'a> {
    type Target = MutexGuard<'a, BackendInternal>;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl<'a> DerefMut for BackendRef<'a> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl Backend {
    fn lock(&self) -> BackendRef {
        BackendRef(self.0.lock().unwrap(), self.0.clone())
    }
}

/// A Device corresponds to a certificate.
#[derive(Clone)]
pub struct Device(Arc<Mutex<DeviceInternal>>);

struct DeviceInternal {
    #[allow(dead_code)]
    backend: Weak<Mutex<BackendInternal>>,

    // The device's id (this is just the certificate's fingerprint).
    id: String,

    cert: Cert,

    keys: Vec<Weak<Mutex<KeyInternal>>>,
}

#[derive(Clone)]
pub struct Key(Arc<Mutex<KeyInternal>>);

/// A secret key.
struct KeyInternal {
    #[allow(dead_code)]
    backend: Weak<Mutex<BackendInternal>>,

    // The key's id (this is just the key's fingerprint).
    id: String,

    // The Key data structure.
    //
    // The first one is the variant as read from the TSK.  The second
    // is the unlocked variant (if any).
    variants: Vec<packet::Key<packet::key::SecretParts,
                              packet::key::UnspecifiedRole>>,

    // The unlocked variant, if any.
    unlocked: Option<packet::Key<packet::key::SecretParts,
                                 packet::key::UnspecifiedRole>>,
}

impl Backend {
    /// Initializes a SoftKeys backend.
    ///
    /// `home` is the directory where the backend will look for its
    /// configuration.  If this is `None`, then it will look in
    /// `$HOME/.sq/keystore/softkeys`.
    ///
    /// Currently, this backend looks for certificates containing
    /// secret key material in files with an 'asc' or 'pgp' extension.
    /// Other files are silently ignored.
    pub fn init<P: AsRef<Path>>(home: Option<P>) -> Result<Box<Self>> {
        log::trace!("Backend::init");

        let home = if let Some(home) = home {
            home.as_ref().into()
        } else if let Some(home) = dirs::home_dir() {
            home.join(".sq").join("keystore").join("softkeys")
        } else {
            return Err(anyhow::anyhow!("Failed get the home directory"));
        };

        log::info!("Home directory is: {:?}", home);

        let mut backend = Backend(Arc::new(Mutex::new(BackendInternal {
            home: home.clone(),
            keys: Default::default(),
            certs: Default::default(),
        })));

        if let Err(err) = backend.scan() {
            log::error!("Scanning {:?} for keys: {}", home, err);
        }

        Ok(Box::new(backend))
    }
}

impl BackendRef<'_> {
    /// Ingests a certificate.
    ///
    /// This merges the secret key of a certificate into the backend.
    /// If the certificate does not contain any secret key material,
    /// then an error is returned.
    pub fn ingest(&mut self, cert: Cert) -> Result<()> {
        log::trace!("BackendRef::ingest");

        if ! cert.is_tsk() {
            return Err(anyhow::anyhow!(
                "{} does not contain any secret key material",
                cert.fingerprint()));
        }

        // Add entries for the secret keys.
        let mut keys = Vec::with_capacity(cert.keys().secret().count());
        for ka in cert.keys().secret() {
            log::info!("Adding {} ({})", ka.keyid(), cert.keyid());

            match self.0.keys.entry(ka.fingerprint()) {
                Entry::Occupied(oe) => {
                    log::info!("{} ({}) already present",
                               ka.keyid(), cert.keyid());

                    keys.push(Arc::downgrade(&oe.get().0));

                    // We collect all secret key variants, but avoid
                    // duplicates.
                    let variants = &mut oe.get().0.lock().unwrap().variants;
                    if ! variants.contains(ka.key()) {
                        log::info!("{} ({}): new variant",
                                   ka.keyid(), cert.keyid());

                        variants.push(ka.key().clone());
                    }
                }
                Entry::Vacant(ve) => {
                    let sk = Key(Arc::new(Mutex::new(KeyInternal {
                        backend: Arc::downgrade(&self.1),
                        id: ka.fingerprint().to_hex(),
                        variants: vec![ ka.key().clone() ],
                        unlocked: None,
                    })));
                    keys.push(Arc::downgrade(&sk.0));
                    ve.insert(sk);
                }
            }
        }

        // Add an entry for the certificate.
        match self.0.certs.entry(cert.fingerprint()) {
            Entry::Occupied(oe) => {
                // Merge cert.
                let mut e = oe.get().0.lock().unwrap();

                e.cert = e.cert.clone().merge_public_and_secret(cert)
                    .expect("same certificate");

                e.keys.append(&mut keys);

                e.keys.sort_by_key(|k| {
                    k.upgrade().expect("valid").lock().unwrap().id.clone()
                });
                e.keys.dedup_by_key(|k| {
                    k.upgrade().expect("valid").lock().unwrap().id.clone()
                });
            }
            Entry::Vacant(ve) => {
                ve.insert(
                    Device(Arc::new(Mutex::new(DeviceInternal {
                        backend: Arc::downgrade(&self.1),
                        id: cert.fingerprint().to_hex(),
                        cert: cert,
                        keys,
                    }))));
            }
        }

        Ok(())
    }
}

impl backend::Backend for Backend {
    fn id(&self) -> String {
        "softkeys".into()
    }

    fn scan(&mut self) -> Result<()> {
        log::trace!("Backend::scan");

        let mut backend = self.lock();

        log::info!("Scanning: {:?}", backend.home);
        for entry in backend.home.read_dir()
            .with_context(|| format!("{:?}", backend.home))?
        {
            let entry = match entry {
                Ok(entry) => entry,
                Err(err) => {
                    log::debug!("While listing {:?}: {}", backend.home, err);
                    continue;
                }
            };

            let filename = entry.path();

            let read = if let Some(extension) = filename.extension() {
                if extension == "asc" || extension == "pgp" {
                    true
                } else {
                    false
                }
            } else {
                false
            };

            if ! read {
                log::debug!("Ignoring {:?}", filename);
                continue;
            }
            log::debug!("Considering: {:?}", filename);

            let parser = CertParser::from_file(&filename)?;
            for certr in parser {
                let cert = match certr {
                    Ok(cert) => cert,
                    Err(err) => {
                        log::info!("{:?} partially corrupted: {}",
                                   filename, err);
                        continue;
                    }
                };

                log::debug!("Found certificate {} in {:?}",
                            cert.keyid(), filename);

                backend.ingest(cert)?;
            }
        }

        Ok(())
    }

    fn list<'a>(&'a self)
        -> Box<dyn Iterator<Item=Box<dyn backend::DeviceHandle + Send + Sync + 'a>>
               + Send + Sync + 'a>
    {
        log::trace!("Backend::list");

        let backend = self.0.lock().unwrap();

        Box::new(
            backend.certs.values()
                .map(|sc| {
                    Box::new(sc.clone())
                        as Box<dyn backend::DeviceHandle + Send + Sync + 'a>
                })
                .collect::<Vec<_>>()
                .into_iter())
    }

    fn find_device<'a>(&self, id: &str)
        -> Result<Box<dyn backend::DeviceHandle + Send + Sync + 'a>>
    {
        log::trace!("Backend::find_device");

        let backend = self.0.lock().unwrap();

        backend.certs.get(&id.parse()?)
            .map(|sc| {
                Box::new(sc.clone())
                    as Box<dyn backend::DeviceHandle + Send + Sync + 'a>
            })
            .ok_or_else(|| Error::NotFound(id.into()).into())
    }

    fn find_key<'a>(&self, id: &str)
        -> Result<Box<dyn backend::KeyHandle + Send + Sync + 'a>>
    {
        log::trace!("Backend::find_key");

        let backend = self.lock();

        backend.keys.get(&id.parse()?)
            .map(|sk| {
                Box::new(sk.clone()) as Box<dyn backend::KeyHandle + Send + Sync + 'a>
            })
            .ok_or_else(|| Error::NotFound(id.into()).into())
    }
}

impl backend::DeviceHandle for Device {
    fn id(&self) -> String {
        self.0.lock().unwrap().id.clone()
    }

    /// Certificates cannot be registered so if it exists, it is
    /// available.
    fn available(&self) -> bool {
        true
    }

    /// Certificates cannot be unconfigured.
    fn configured(&self) -> bool {
        true
    }

    /// Certificates cannot be registered.  (They are always available.)
    fn registered(&self) -> bool {
        false
    }

    fn lock(&mut self) -> Result<()> {
        log::trace!("DeviceHandle::lock");

        let mut dh = self.0.lock().unwrap();

        for k in dh.keys.iter_mut() {
            if let Some(kh) = k.upgrade() {
                let mut kh = Key(kh);
                if let Err(err) = kh.lock() {
                    log::debug!("Error locking key {}: {}",
                                kh.id(), err);
                }
            }
        }

        Ok(())
    }

    fn list<'a>(&'a self)
        -> Box<dyn Iterator<Item=Box<dyn backend::KeyHandle + Send + Sync + 'a>>
               + Send + Sync + 'a>
    {
        log::trace!("DeviceHandle::list");

        let cert = self.0.lock().unwrap();

        Box::new(
            cert
                .keys
                .iter()
                .filter_map(|k| {
                    Some(Box::new(Key(k.upgrade()?))
                         as Box<dyn sequoia_keystore_backend::KeyHandle
                                + Send + Sync>)
                })
                .collect::<Vec<_>>()
                .into_iter())
    }
}

impl backend::KeyHandle for Key {
    fn id(&self) -> String {
        self.0.lock().unwrap().id.clone()
    }

    fn available(&self) -> bool {
        true
    }

    fn locked(&self) -> bool {
        let kh = self.0.lock().unwrap();
        let unlocked = kh.unlocked.is_some()
            || kh.variants.iter().any(|k| k.has_unencrypted_secret());
        ! unlocked
    }

    fn decryption_capable(&self) -> bool {
        let kh = self.0.lock().unwrap();
        let key = &kh.variants[0];

        // XXX: Also look up the certificate in the cert store and
        // check the key flags.
        key.pk_algo().for_encryption()
    }

    fn signing_capable(&self) -> bool {
        let kh = self.0.lock().unwrap();
        let key = &kh.variants[0];

        // XXX: Also look up the certificate in the cert store and
        // check the key flags.
        key.pk_algo().for_signing()
    }

    fn unlock(&mut self, password: &Password) -> Result<()> {
        log::trace!("KeyHandle::unlock");

        let mut kh = self.0.lock().unwrap();

        if kh.unlocked.is_some() {
            return Err(Error::AlreadyUnlocked(kh.id.clone()).into());
        }

        let mut err = None;
        for v in kh.variants.iter() {
            let mut k = v.clone();
            match k.secret_mut().decrypt_in_place(v.pk_algo(), password) {
                Ok(()) => {
                    kh.unlocked = Some(k);
                    return Ok(());
                }
                Err(err_) => {
                    err = Some(err_);
                }
            }
        }

        Err(err.expect("at least one key variant"))
    }

    fn lock(&mut self) -> Result<()> {
        log::trace!("KeyHandle::lock");

        let mut kh = self.0.lock().unwrap();
        kh.unlocked = None;
        Ok(())
    }

    fn fingerprint(&self) -> Fingerprint {
        log::trace!("KeyHandle::fingerprint");

        self.0.lock().unwrap().variants[0].fingerprint()
    }

    fn public_key(&self)
        -> packet::Key<packet::key::PublicParts,
                       packet::key::UnspecifiedRole>
    {
        log::trace!("KeyHandle::public_key");

        let k = self.0.lock().unwrap().variants[0].clone();
        k.take_secret().0
    }

    fn decrypt_ciphertext(&mut self,
                          ciphertext: &mpi::Ciphertext,
                          plaintext_len: Option<usize>)
        -> Result<SessionKey>
    {
        log::trace!("KeyHandle::decrypt_ciphertext");

        let kh = self.0.lock().unwrap();

        // We may have multiple variants.  We try them in turn unless
        // a key is already unlocked in which case we only try that
        // one.
        let variants: Box<dyn Iterator<Item=&packet::Key<_, _>>>
            = if let Some(k) = kh.unlocked.as_ref() {
                Box::new(std::iter::once(k))
            } else {
                Box::new(kh.variants.iter())
            };

        let mut err = None;
        for k in variants {
            if k.secret().is_encrypted() {
                if err.is_none() {
                    err = Some(Error::Locked(k.fingerprint().to_hex()).into());
                }
                continue;
            }

            // `into_keypair` fails if the key is encrypted, but we
            // already checked that it isn't.
            let mut keypair = k.clone().into_keypair().expect("decrypted");

            match keypair.decrypt(ciphertext, plaintext_len) {
                Ok(sk) => {
                    log::info!("Decrypted ciphertext using {}", k.keyid());
                    return Ok(sk);
                }
                Err(err_) => {
                    log::info!("Decrypting ciphertext using {}: {}",
                               k.keyid(), err_);
                    if err.is_none() {
                        err = Some(err_);
                    }
                }
            }
        }

        Err(err.expect("have key variants that failed"))
    }

    fn sign(&mut self, hash_algo: HashAlgorithm, digest: &[u8])
        -> Result<(PublicKeyAlgorithm, mpi::Signature)>
    {
        log::trace!("KeyHandle::sign");

        let kh = self.0.lock().unwrap();

        // We may have multiple variants.  We try them in turn unless
        // a key is already unlocked in which case we only try that
        // one.
        let variants: Box<dyn Iterator<Item=&packet::Key<_, _>>>
            = if let Some(k) = kh.unlocked.as_ref() {
                Box::new(std::iter::once(k))
            } else {
                Box::new(kh.variants.iter())
            };

        // Remember the first error.
        let mut err = None;
        for k in variants {
            if k.secret().is_encrypted() {
                if err.is_none() {
                    err = Some(Error::Locked(k.fingerprint().to_hex()).into());
                }
                continue;
            }

            // `into_keypair` fails if the key is encrypted, but we
            // already checked that it isn't.
            let mut keypair = k.clone().into_keypair().expect("decrypted");

            match keypair.sign(hash_algo, digest) {
                Ok(sig) => return Ok((k.pk_algo(), sig)),
                Err(err_) => err = Some(err_),
            }
        }

        Err(err.expect("At least one key variant"))
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    use std::io::Read;

    use test_log::test;

    use openpgp::Result;
    use openpgp::packet::Packet;
    use openpgp::packet::signature::SignatureBuilder;
    use openpgp::parse::PacketParser;
    use openpgp::parse::PacketParserResult;
    use openpgp::parse::Parse;
    use openpgp::parse::stream::VerifierBuilder;
    use openpgp::serialize::stream::LiteralWriter;
    use openpgp::serialize::stream::Message;
    use openpgp::serialize::stream::Signer;
    use openpgp::types::SignatureType;
    use sequoia_openpgp as openpgp;

    /// Initialize the backend from some keyrings.
    ///
    /// The backend, and the certificates are returned.
    fn init_with(keys: &[ &[u8] ]) -> Result<(Box<Backend>, Vec<Cert>)> {
        let backend = Backend::init(Some(PathBuf::from("/dev/null")))?;
        let mut certs = Vec::new();

        for key in keys {
            let parser = CertParser::from_bytes(key)?;
            for certr in parser {
                let cert = match certr {
                    Ok(cert) => {
                        certs.push(cert.clone());
                        cert
                    }
                    Err(err) => {
                        log::debug!("cert partially corrupted: {}", err);
                        continue;
                    }
                };

                backend.lock().ingest(cert)?;
            }
        }

        Ok((backend, certs))
    }

    /// Try to decrypt a message.
    ///
    /// `ciphertext` is the well-formed encrypted message.  plaintext
    /// is the expected plaintext.  If it is not `None`, then it must
    /// match what is decrypted.
    ///
    /// `password` is an optional password.  If supplied, then it must
    /// be used to unlock the key.
    fn try_decrypt(backend: &mut Backend,
                   ciphertext: &[u8], plaintext: Option<&[u8]>,
                   password: Option<Password>) -> Result<()> {
        let mut good = false;

        let mut session_key = None;
        let mut ppr = PacketParser::from_bytes(ciphertext)?;
        while let PacketParserResult::Some(mut pp) = ppr {
            match pp.packet {
                Packet::PKESK(ref pkesk) => {
                    // Try all the keys in turn.
                    'done: for k in backend.lock().keys.values_mut() {
                        if let Some(password) = password.as_ref() {
                            k.lock()?;
                            k.unlock(password)?;
                        }

                        if let Some((algo, sk)) = k.decrypt_pkesk(&pkesk) {
                            session_key = Some((algo, sk));
                            break 'done;
                        }
                    }
                    if session_key.is_none() {
                        return Err(anyhow::anyhow!("Decryption failed".to_string()));
                    }
                }
                Packet::SEIP(_) => {
                    let (algo, sk) = session_key.take().expect("No session key.");
                    pp.decrypt(algo, &sk)?;
                }
                Packet::Literal(_) => {
                    pp.buffer_unread_content()?;
                    if let Packet::Literal(l) = &pp.packet {
                        if let Some(plaintext) = plaintext {
                            if plaintext != l.body() {
                                return Err(anyhow::anyhow!(format!(
                                    "Plaintext does not match:\n\
                                     got:      {:?},\n\
                                     expected: {:?}",
                                    l.body(), plaintext)));
                            }
                        }
                        good = true;
                    }
                }
                _ => (),
            }

            let (_packet, next_ppr) = pp.recurse()?;
            ppr = next_ppr;
        }

        if let PacketParserResult::EOF(ppr) = ppr {
            assert!(ppr.is_message().is_ok());
            // A possible reason this may fail is if the message uses
            // compression, but compression support is not enabled.
            assert!(good);
        }

        Ok(())
    }

    #[test]
    fn decrypt() -> Result<()> {
        // A simple test: given a secret key and a message encrypted
        // to the key, can we use the soft key backend to decrypt it?

        let key: &[u8] = testdata::key("no-password.asc");
        let msg: &[u8] = testdata::message("no-password.asc");

        let (mut backend, _certs) = init_with(&[ key ])?;
        try_decrypt(&mut backend, msg, Some(b"foo\n"), None).unwrap();

        Ok(())
    }

    #[test]
    fn decrypt_missing_password() -> Result<()> {
        // Load two variants of a certificate: one where the keys are
        // encrypted with the password "bob" and other where they are
        // encrypted with the password "smithy".  Make sure we can't
        // decrypt the message if we don't supply a password.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-password-smithy.asc"),
        ];
        let msg: &[u8] = testdata::message("bob.asc");

        let (mut backend, _certs) = init_with(keys)?;

        assert!(try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                            None).is_err());

        Ok(())
    }

    #[test]
    fn decrypt_with_password() -> Result<()> {
        // Load two variants of a certificate: one where the keys are
        // encrypted with the password "bob" and other where they are
        // encrypted with the password "smithy".  Make sure we can
        // decrypt the message using both passwords.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-password-smithy.asc"),
        ];
        let msg: &[u8] = testdata::message("bob.asc");

        let (mut backend, _certs) = init_with(keys)?;

        try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                    Some(Password::from("bob"))).unwrap();
        try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                    Some(Password::from("smithy"))).unwrap();
        assert!(try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                            Some(Password::from("sup3r s3cur3"))).is_err());

        Ok(())
    }

    #[test]
    fn decrypt_with_passwords_and_without() -> Result<()> {
        // Load two variants of a certificate: one where the keys are
        // encrypted with the password "bob" and other where they are
        // not encrypted.  Make sure we can decrypt the message using
        // the password or no password.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-no-password.asc"),
        ];
        let msg: &[u8] = testdata::message("bob.asc");

        let (mut backend, _certs) = init_with(keys)?;

        try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                    Some(Password::from("bob"))).unwrap();
        try_decrypt(&mut backend, msg, Some(b"hi bob\n"),
                    None).unwrap();

        Ok(())
    }

    /// Helpers to verify a message.
    mod verify {
        use super::*;

        use openpgp::crypto;
        use openpgp::parse::stream::GoodChecksum;
        use openpgp::parse::stream::MessageLayer;
        use openpgp::parse::stream::MessageStructure;
        use openpgp::parse::stream::VerificationHelper;

        pub struct KeyHandleSigner(
            Box<dyn backend::KeyHandle + Send + Sync>,
            packet::Key<packet::key::PublicParts, packet::key::UnspecifiedRole>);

        impl From<Box<dyn backend::KeyHandle + Send + Sync>> for KeyHandleSigner {
            fn from(kh: Box<dyn backend::KeyHandle + Send + Sync>) -> Self {
                let key = kh.public_key();
                Self(kh, key)
            }
        }

        impl crypto::Signer for KeyHandleSigner {
            fn public(&self) -> &packet::Key<packet::key::PublicParts,
                                             packet::key::UnspecifiedRole>
            {
                &self.1
            }

            fn sign(&mut self, hash_algo: HashAlgorithm, digest: &[u8])
                    -> Result<mpi::Signature>
            {
                self.0.sign(hash_algo, digest).map(|(_pk_algo, sig)| sig)
            }
        }

        pub struct Helper {
            // The certificates.
            pub keyring: Vec<Cert>,
            // The good signatures.
            pub signers: Vec<Fingerprint>,
        }

        impl Helper {
            pub fn new(keyring: &[ Cert ]) -> Self {
                Self {
                    keyring: keyring.to_vec(),
                    signers: Vec::new(),
                }
            }
        }

        impl VerificationHelper for &mut Helper {
            fn get_certs(&mut self, ids: &[openpgp::KeyHandle]) -> Result<Vec<Cert>> {
                Ok(self.keyring
                   .iter()
                   .filter(|cert| {
                       cert.keys().any(|k| {
                           ids.iter().any(|id| {
                               id.aliases(openpgp::KeyHandle::from(k.fingerprint()))
                           })
                       })
                   })
                   .cloned()
                   .collect::<Vec<_>>())
            }
            fn check(&mut self, structure: MessageStructure) -> Result<()> {
                for (i, layer) in structure.into_iter().enumerate() {
                    match layer {
                        MessageLayer::Encryption { .. } if i == 0 => (),
                        MessageLayer::Compression { .. } if i == 1 => (),
                        MessageLayer::SignatureGroup { ref results } => {
                            for r in results {
                                if let Ok(GoodChecksum { ka, .. }) = r {
                                    self.signers.push(ka.fingerprint());
                                } else {
                                    log::info!("Bad signature: {:?}", r);
                                }
                            }
                        }
                        _ => return Err(anyhow::anyhow!(
                            "Unexpected message structure")),
                    }
                }

                Ok(())
            }
        }

        /// Sign a message and verify that the signature is good.
        ///
        /// If signing_keys is None, this is expected to fail.
        pub fn sign_verify(signers: Vec<verify::KeyHandleSigner>,
                           keyring: Vec<Cert>,
                           signing_keys: Option<&[ Fingerprint ]>) -> Result<()>
        {
            let p = &openpgp::policy::StandardPolicy::new();

            let mut output = Vec::new();
            let message = Message::new(&mut output);

            let builder = SignatureBuilder::new(SignatureType::Binary);

            let mut signers = signers.into_iter();
            let mut signer = Signer::with_template(
                message, signers.next().expect("a signer"), builder);
            for s in signers {
                signer = signer.add_signer(s);
            }
            let signer = signer.build().context("Failed to create signer")?;
            let mut writer = LiteralWriter::new(signer).build()
                .context("Failed to create literal writer")?;

            std::io::copy(&mut std::io::Cursor::new(b"hello"), &mut writer)
                .context("Failed to sign")?;
            if let Err(err) = writer.finalize().context("Failed to sign") {
                if signing_keys.is_none() {
                    // Expected failure.
                    return Ok(());
                } else {
                    panic!("Failed to finalize writer: {}", err);
                }
            }

            let mut h = verify::Helper::new(&keyring);
            let mut v = VerifierBuilder::from_bytes(&output[..])?
                .with_policy(p, None, &mut h)?;

            let mut message = Vec::new();
            let r = v.read_to_end(&mut message);

            if let Some(signing_keys) = signing_keys {
                assert!(r.is_ok());
                assert_eq!(message, b"hello");
                assert_eq!(&h.signers, signing_keys);
            } else {
                assert!(r.is_err());
            }

            Ok(())
        }
    }

    #[test]
    fn verify() -> Result<()> {
        // A simple test: sign a message using a key with no password,
        // then verify it.

        let key: &[u8] = testdata::key("no-password.asc");

        let (backend, keyring) = init_with(&[ key ])?;

        let signing_key = "88647F753426C4F16D1ED1E50EF5A4511075ACD7";

        let k = backend.find_key(signing_key)?;
        let signer = verify::KeyHandleSigner::from(k);

        verify::sign_verify(vec![ signer ],
                            keyring,
                            Some(&[ signing_key.parse()? ][..]))?;

        Ok(())
    }

    #[test]
    fn verify_missing_password() -> Result<()> {
        // sign a message using a key that is password, but don't
        // provide the password.  Make sure it fails.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-password-smithy.asc"),
        ];

        let (backend, keyring) = init_with(keys)?;

        let signing_key = "9410E96E68643821794608269CB5006116833EE7";

        let k = backend.find_key(signing_key)?;
        let signer = verify::KeyHandleSigner::from(k);

        verify::sign_verify(vec![ signer ],
                            keyring,
                            None)?;

        Ok(())
    }

    #[test]
    fn verify_with_passwords() -> Result<()> {
        // Sign a message using a key that is password protected.  To
        // make it trickier, use a key with multiple variants.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-password-smithy.asc"),
        ];

        let (backend, keyring) = init_with(keys)?;

        let signing_key = "9410E96E68643821794608269CB5006116833EE7";

        let mut k = backend.find_key(signing_key)?;
        k.unlock(&Password::from("bob"))?;

        let signer = verify::KeyHandleSigner::from(k);

        verify::sign_verify(vec![ signer ],
                            keyring,
                            Some(&[ signing_key.parse()? ][..]))?;

        Ok(())
    }

    #[test]
    fn verify_with_passwords_and_without() -> Result<()> {
        // Sign a message using a key that is password protected.  Use
        // a key that has two variants: one that is password
        // protected, and one that isn't.

        let keys: &[ &[u8] ] = &[
            testdata::key("bob-password-bob.asc"),
            testdata::key("bob-no-password.asc"),
        ];

        let (backend, keyring) = init_with(keys)?;

        let signing_key = "9410E96E68643821794608269CB5006116833EE7";

        // Try without a password.
        let k = backend.find_key(signing_key)?;

        let signer = verify::KeyHandleSigner::from(k);

        verify::sign_verify(vec![ signer ],
                            keyring.clone(),
                            Some(&[ signing_key.parse()? ][..]))?;


        // Try unlocking the variant with a password and make sure it
        // still works.
        let mut k = backend.find_key(signing_key)?;
        k.unlock(&Password::from("bob"))?;

        let signer = verify::KeyHandleSigner::from(k);

        verify::sign_verify(vec![ signer ],
                            keyring,
                            Some(&[ signing_key.parse()? ][..]))?;

        Ok(())
    }

    #[test]
    fn list_keys() -> Result<()> {
        // We have two variants of a certificate with disjoint
        // subkeys.  When we load both certificates, make sure both
        // sets of variants appear.

        let keyring_a: &[u8] = testdata::key("carol-subkeys-a.asc");
        let keys_a = &[
            "B0B75BE1004E71FC9FEEF319CC160DCFDE911D68",
            "4BA88DAF3998EF1A1A4DBC9349F15C0BB3DD62B5",
        ];

        let keyring_b: &[u8] = testdata::key("carol-subkeys-b.asc");
        let keys_b = &[
            "B0B75BE1004E71FC9FEEF319CC160DCFDE911D68",
            "61D8031424FC2C58BBA6CA7F11B1E2EDC6E4D1D1",
            "5606791C6C58CD2A8F1306397189B15EF88BB1D5",
        ];

        // Try with just a, then just b, then both variants.
        for (a, b) in [(true, false), (false, true), (true, true)].iter() {
            let mut keyring: Vec<&[u8]> = Vec::new();
            let mut keys = Vec::new();

            if *a {
                keyring.push(keyring_a);
                keys.extend_from_slice(keys_a);
            }
            if *b {
                keyring.push(keyring_b);
                keys.extend_from_slice(keys_b);
            }

            let (backend, _certs) = init_with(&keyring[..])?;

            // List the devices.
            let devices = backend.list().map(|d| d.id()).collect::<Vec<String>>();
            assert_eq!(devices, vec![ keys_a[0] ]);

            // Check that we can find the cert using Backend::find_device.
            let device = backend.find_device(keys_a[0])
                .expect("certificate");

            // List the keys.
            let mut got = device.list().map(|k| k.id()).collect::<Vec<String>>();
            got.sort();

            keys.sort();
            keys.dedup();

            assert_eq!(
                keys.into_iter().map(String::from).collect::<Vec<String>>(),
                got);

            // Check that Backend::find_key returns all of the keys.
            for (i, k) in keys_a.iter().enumerate() {
                if *a || i == 0 {
                    assert!(backend.find_key(k).is_ok());
                } else {
                    assert!(backend.find_key(k).is_err());
                }
            }
            for (i, k) in keys_b.iter().enumerate() {
                if *b || i == 0 {
                    assert!(backend.find_key(k).is_ok());
                } else {
                    assert!(backend.find_key(k).is_err());
                }
            }
        }

        Ok(())
    }

    #[test]
    fn list_only_returns_secret_keys() -> Result<()> {
        // Make sure only keys with secret key material are returned.

        let keyring: &[u8] = testdata::key("dave-not-all-keys-have-secrets.asc");
        let cert = "4FD198623B8726286C7159A6032CCEF246B60258";
        let mut keys = vec![
            "AC32461C491647DA6DAC794712CCFC305567FA9D",
        ];
        let public_keys = vec![
            "4FD198623B8726286C7159A6032CCEF246B60258",
            "AA91F4FA667D5B17E8ACCA9F6BC89BD62B55B802",
        ];

        let (backend, _certs) = init_with(&[ keyring ])?;

        // List the devices.
        let devices = backend.list().map(|d| d.id()).collect::<Vec<String>>();
        assert_eq!(devices, vec![ cert ]);

        // Check that we can find the cert using Backend::find_device.
        let device = backend.find_device(cert)
            .expect("certificate");

        // List the keys.
        let mut got = device.list().map(|k| k.id()).collect::<Vec<String>>();
        got.sort();

        keys.sort();
        keys.dedup();

        assert_eq!(
            keys.iter().map(|&k| k.into()).collect::<Vec<String>>(),
            got);

        // Check that Backend::find_key returns all of the keys.
        for k in keys {
            assert!(backend.find_key(k).is_ok());
        }
        // And doesn't return the public keys.
        for k in public_keys {
            assert!(backend.find_key(k).is_err());
        }

        Ok(())
    }
}
