/* -*- Mode: C++; tab-width: 8; c-basic-offset: 2; indent-tabs-mode: nil; -*- */

//#define DEBUGTAG "ReplaySession"

#define USE_BREAKPOINT_TARGET 1

#include "ReplaySession.h"

#include <syscall.h>
#include <sys/prctl.h>

#include <algorithm>

#include "AutoRemoteSyscalls.h"
#include "fast_forward.h"
#include "kernel_metadata.h"
#include "log.h"
#include "replay_syscall.h"
#include "task.h"
#include "util.h"

using namespace rr;
using namespace std;

/* Why a skid region?  Interrupts generated by perf counters don't
 * fire at exactly the programmed point (as of 2013 kernel/HW);
 * there's a variable slack region, which is technically unbounded.
 * This means that an interrupt programmed for retired branch k might
 * fire at |k + 50|, for example.  To counteract the slack, we program
 * interrupts just short of our target, by the |SKID_SIZE| region
 * below, and then more slowly advance to the real target.
 *
 * How was this magic number determined?  Trial and error: we want it
 * to be as small as possible for efficiency, but not so small that
 * overshoots are observed.  If all other possible causes of overshoot
 * have been ruled out, like memory divergence, then you'll know that
 * this magic number needs to be increased if the following symptom is
 * observed during replay.  Running with DEBUGLOG enabled (see above),
 * a sequence of log messages like the following will appear
 *
 * 1. programming interrupt for [target - SKID_SIZE] ticks
 * 2. Error: Replay diverged.  Dumping register comparison.
 * 3. Error: [list of divergent registers; arbitrary]
 * 4. Error: overshot target ticks=[target] by [i]
 *
 * The key is that no other replayer log messages occur between (1)
 * and (2).  This spew means that the replayer programmed an interrupt
 * for ticks=[target-SKID_SIZE], but the tracee was actually interrupted
 * at ticks=[target+i].  And that in turn means that the kernel/HW
 * skidded too far past the programmed target for rr to handle it.
 *
 * If that occurs, the SKID_SIZE needs to be increased by at least
 * [i].
 *
 * NB: there are probably deeper reasons for the target slack that
 * could perhaps let it be deduced instead of arrived at empirically;
 * perhaps pipeline depth and things of that nature are involved.  But
 * those reasons if they exit are currently not understood.
 */
static const int SKID_SIZE = 70;

static void debug_memory(Task* t) {
  if (should_dump_memory(t, t->current_trace_frame())) {
    dump_process_memory(t, t->current_trace_frame().time(), "rep");
  }
  if (t->session().can_validate() &&
      should_checksum(t, t->current_trace_frame())) {
    /* Validate the checksum we computed during the
     * recording phase. */
    validate_process_memory(t, t->current_trace_frame().time());
  }
}

ReplaySession::~ReplaySession() {
  // We won't permanently leak any OS resources by not ensuring
  // we've cleaned up here, but sessions can be created and
  // destroyed many times, and we don't want to temporarily hog
  // resources.
  kill_all_tasks();
  assert(task_map.empty() && vm_map.empty());
  gc_emufs();
  assert(emufs().size() == 0);
}

ReplaySession::shr_ptr ReplaySession::clone() {
  LOG(debug) << "Deepforking ReplaySession " << this << " ...";

  finish_initializing();

  shr_ptr session(new ReplaySession(*this));
  LOG(debug) << "  deepfork session is " << session.get();

  copy_state_to(*session, session->emufs());

  return session;
}

/**
 * Return true if it's possible/meaningful to make a checkpoint at the
 * |frame| that |t| will replay.
 */
static bool can_checkpoint_at(Task* t, const TraceFrame& frame) {
  const Event& ev = frame.event();
  if (ev.has_ticks_slop()) {
    return false;
  }
  switch (ev.type()) {
    case EV_EXIT:
    case EV_UNSTABLE_EXIT:
    // At exits, we can't clone the exiting tasks, so
    // don't event bother trying to checkpoint.
    case EV_SYSCALLBUF_RESET:
    // RESETs are usually inserted in between syscall
    // entry/exit.  Do not attempting to checkpoint at
    // RESETs.  Users would never want to do that anyway.
    case EV_TRACE_TERMINATION:
      // There's nothing to checkpoint at the end of an
      // early-terminated trace.
      return false;
    default:
      return true;
  }
}

bool ReplaySession::can_clone() {
  finish_initializing();

  Task* t = current_task();
  return t && can_validate() && can_checkpoint_at(t, current_trace_frame());
}

DiversionSession::shr_ptr ReplaySession::clone_diversion() {
  finish_initializing();

  LOG(debug) << "Deepforking ReplaySession " << this
             << " to DiversionSession...";

  DiversionSession::shr_ptr session(new DiversionSession(*this));
  LOG(debug) << "  deepfork session is " << session.get();

  copy_state_to(*session, session->emufs());
  session->finish_initializing();

  return session;
}

void ReplaySession::gc_emufs() { emu_fs->gc(*this); }

void ReplaySession::maybe_gc_emufs(SupportedArch arch, int syscallno) {
  if (is_close_syscall(syscallno, arch) || is_munmap_syscall(syscallno, arch)) {
    gc_emufs();
  }
}

/*static*/ ReplaySession::shr_ptr ReplaySession::create(const string& dir) {
  shr_ptr session(new ReplaySession(dir));

  // Because we execvpe() the tracee, we must ensure that $PATH
  // is the same as in recording so that libc searches paths in
  // the same order.  So copy that over now.
  //
  // And because we use execvpe(), the exec'd tracee will start
  // with a fresh environment guaranteed to be the same as in
  // replay, so we don't have to worry about any mutation here
  // affecting post-exec execution.
  for (auto& e : session->trace_in.initial_envp()) {
    if (e.find("PATH=") == 0) {
      // NB: intentionally leaking this string.
      putenv(strdup(e.c_str()));
    }
  }

  Task* t = Task::spawn(*session, session->trace_in,
                        session->trace_reader().peek_frame().tid());
  session->on_create(t);

  return session;
}

void ReplaySession::advance_to_next_trace_frame(TraceFrame::Time stop_at_time) {
  if (trace_in.at_end()) {
    return;
  }

  trace_frame = trace_in.read_frame();
}

bool ReplaySession::is_ignored_signal(int sig) {
  switch (sig) {
    // TIME_SLICE_SIGNALs can be queued but not delivered before we stop
    // execution for some other reason. Ignore them.
    case PerfCounters::TIME_SLICE_SIGNAL:
      return true;
    default:
      return false;
  }
}

static bool compute_ticks_request(
    Task* t, const ReplaySession::StepConstraints& constraints,
    TicksRequest* ticks_request) {
  *ticks_request = RESUME_UNLIMITED_TICKS;
  if (constraints.ticks_target > 0) {
    Ticks ticks_period = constraints.ticks_target - SKID_SIZE - t->tick_count();
    if (ticks_period <= 0) {
      // Behave as if we actually executed something. Callers assume we did.
      t->clear_wait_status();
      return false;
    }
    *ticks_request = (TicksRequest)ticks_period;
  }
  return true;
}

/**
 * Continue until reaching either the "entry" of an emulated syscall,
 * or the entry or exit of an executed syscall.  |emu| is nonzero when
 * we're emulating the syscall.  Return COMPLETE when the next syscall
 * boundary is reached, or INCOMPLETE if advancing to the boundary was
 * interrupted by an unknown trap.
 * When |syscall_trace_frame| is non-null, we continue to the syscall by
 * setting a breakpoint instead of running until we execute a system
 * call instruction. In that case we will not actually enter the kernel.
 */
Completion ReplaySession::cont_syscall_boundary(
    Task* t, const StepConstraints& constraints) {
  TicksRequest ticks_request;
  if (!compute_ticks_request(t, constraints, &ticks_request)) {
    return INCOMPLETE;
  }

  if (constraints.command == RUN_SINGLESTEP_FAST_FORWARD) {
    // ignore ticks_period. We can't add more than one tick during a
    // fast_forward so it doesn't matter.
    did_fast_forward |= fast_forward_through_instruction(
        t, RESUME_SYSEMU_SINGLESTEP, constraints.stop_before_states);
  } else {
    ResumeRequest resume_how =
        constraints.is_singlestep() ? RESUME_SYSEMU_SINGLESTEP : RESUME_SYSEMU;
    t->resume_execution(resume_how, RESUME_WAIT, ticks_request);
  }

  if (t->pending_sig() == PerfCounters::TIME_SLICE_SIGNAL) {
    // This would normally be triggered by constraints.ticks_target but it's
    // also possible to get stray signals here.
    return INCOMPLETE;
  }

  if (is_ignored_signal(t->pending_sig())) {
    return cont_syscall_boundary(t, constraints);
  }

  if (SIGTRAP == t->pending_sig()) {
    return INCOMPLETE;
  }
  ASSERT(t, !t->pending_sig()) << "Replay got unrecorded signal "
                               << t->pending_sig() << " ("
                               << signal_name(t->pending_sig()) << ")";

  return COMPLETE;
}

/**
 * Advance to the next syscall entry (or virtual entry) according to
 * |step|.  Return COMPLETE if successful, or INCOMPLETE if an unhandled trap
 * occurred.
 */
Completion ReplaySession::enter_syscall(Task* t,
                                        const StepConstraints& constraints) {
  bool use_breakpoint_optimization = false;
  remote_code_ptr syscall_instruction;

  if (can_validate()) {
    syscall_instruction =
        current_trace_frame().regs().ip().decrement_by_syscall_insn_length(
            t->arch());
    // Skip this optimization if we can't set the breakpoint, or if it's
    // in writeable or shared memory, since in those cases it could be
    // overwritten by the tracee. It could even be dynamically generated and
    // not generated yet.
    if (t->vm()->is_breakpoint_in_private_read_only_memory(
            syscall_instruction) &&
        t->vm()->add_breakpoint(syscall_instruction, TRAP_BKPT_INTERNAL)) {
      use_breakpoint_optimization = true;
    }
  }

  if (cont_syscall_boundary(t, constraints) == INCOMPLETE) {
    bool reached_target =
        use_breakpoint_optimization && SIGTRAP == t->pending_sig() &&
        t->ip().decrement_by_bkpt_insn_length(t->arch()) ==
            syscall_instruction &&
        t->vm()->get_breakpoint_type_at_addr(syscall_instruction) ==
            TRAP_BKPT_INTERNAL;
    if (reached_target) {
      // Emulate syscall state change
      Registers r = t->regs();
      r.set_ip(syscall_instruction.increment_by_syscall_insn_length(t->arch()));
      r.set_original_syscallno(r.syscallno());
      r.set_syscall_result(-ENOSYS);
      t->emulate_syscall_entry(r);
      t->validate_regs();
    }
    if (use_breakpoint_optimization) {
      t->vm()->remove_breakpoint(syscall_instruction, TRAP_BKPT_INTERNAL);
    }
    if (!reached_target) {
      return INCOMPLETE;
    }
  } else {
    // If we use the breakpoint optimization, we must get a SIGTRAP before
    // reaching a syscall, so cont_syscall_boundary must return INCOMPLETE.
    ASSERT(t, !use_breakpoint_optimization);
    t->validate_regs();
    t->finish_emulated_syscall();
  }

  return COMPLETE;
}

/**
 * Advance past the reti (or virtual reti) according to |step|.
 * Return COMPLETE if successful, or INCOMPLETE if an unhandled trap occurred.
 */
Completion ReplaySession::exit_syscall(Task* t,
                                       const StepConstraints& constraints) {
  t->on_syscall_exit(current_step.syscall.number, current_trace_frame().regs());

  t->apply_all_data_records_from_trace();
  t->set_return_value_from_trace();

  uint32_t flags = 0;
  if (t->arch() == SupportedArch::x86 &&
      (X86Arch::pwrite64 == current_step.syscall.number ||
       X86Arch::pread64 == current_step.syscall.number)) {
    flags |= Task::IGNORE_ESI;
  }
  t->validate_regs(flags);

  return COMPLETE;
}

void ReplaySession::check_pending_sig(Task* t) {
  ASSERT(t, 0 < t->pending_sig())
      << "Replaying `" << trace_frame.event()
      << "': expecting tracee signal or trap, but instead at `"
      << t->syscall_name(t->regs().original_syscallno())
      << "' (ticks: " << t->tick_count() << ")";
}

/**
 * Advance |t| to the next signal or trap.  If |stepi| is |SINGLESTEP|,
 * then execution resumes by single-stepping.  Otherwise it continues
 * normally. |t->pending_sig()| contains any pending signal.
 *
 * Default |resume_how| is RESUME_SYSCALL for error checking:
 * since the next event is supposed to be a signal,
 * entering a syscall here means divergence.  There
 * shouldn't be any straight-line execution overhead
 * for SYSCALL vs. CONT, so the difference in cost
 * should be neglible.
 *
 * Some callers pass RESUME_CONT because they want to execute any syscalls
 * encountered.
 */
void ReplaySession::continue_or_step(Task* t,
                                     const StepConstraints& constraints,
                                     TicksRequest tick_request,
                                     ResumeRequest resume_how) {
  if (constraints.command == RUN_SINGLESTEP) {
    t->resume_execution(RESUME_SINGLESTEP, RESUME_WAIT, tick_request);
  } else if (constraints.command == RUN_SINGLESTEP_FAST_FORWARD) {
    did_fast_forward |= fast_forward_through_instruction(
        t, RESUME_SINGLESTEP, constraints.stop_before_states);
  } else {
    t->resume_execution(resume_how, RESUME_WAIT, tick_request);
  }
  check_pending_sig(t);
}

/**
 * Return nonzero if |t| was stopped for a breakpoint trap (int3),
 * as opposed to a trace trap.  Return zero in the latter case.
 */
static bool is_breakpoint_trap(Task* t) {
  const siginfo_t& si = t->get_siginfo();
  assert(SIGTRAP == si.si_signo);

  /* XXX unable to find docs on which of these "should" be
   * right.  The SI_KERNEL code is seen in the int3 test, so we
   * at least need to handle that. */
  return SI_KERNEL == si.si_code || TRAP_BRKPT == si.si_code;
}

/**
 * Return one of the (non-zero) enumerated TRAP_* debugger-trap types
 * above if the SIGTRAP generated by the child is intended for the
 * debugger, or zero if it's meant for rr internally.
 *
 * NB: calling this function while advancing the ticks counter through hpc
 * interrupts when emulating asynchronous signal delivery *will*
 * result in bad results.  Don't call this function from there; it's
 * not necessary.
 */
enum ExecStateType { UNKNOWN, NOT_AT_TARGET, AT_TARGET };
TrapType ReplaySession::compute_trap_type(Task* t, int target_sig,
                                          SignalDeterministic deterministic,
                                          ExecStateType exec_state,
                                          const StepConstraints& constraints) {
  TrapType trap_type;

  assert(SIGTRAP == t->pending_sig());

  /* We're not replaying a trap, and it was clearly raised on
   * behalf of the debugger.  (The debugger will verify
   * that.) */
  if (SIGTRAP != target_sig
      /* Replay of deterministic signals never internally
       * single-steps or sets internal breakpoints. */
      && (DETERMINISTIC_SIG == deterministic
          /* Replay of async signals will sometimes internally
           * single-step when advancing to an execution target,
           * so the trap was only clearly for the debugger if
           * the debugger was requesting single-stepping. */
          || (constraints.is_singlestep() && NOT_AT_TARGET == exec_state))) {
    return constraints.is_singlestep() ? TRAP_STEPI : TRAP_BKPT_USER;
  }

  /* We're trying to replay a deterministic SIGTRAP, or we're
   * replaying an async signal. */

  trap_type = t->vm()->get_breakpoint_type_for_retired_insn(t->ip());
  if (TRAP_BKPT_USER == trap_type || TRAP_BKPT_INTERNAL == trap_type) {
    assert(is_breakpoint_trap(t));
    return trap_type;
  }

  if (is_breakpoint_trap(t)) {
    /* We successfully replayed a recorded deterministic
     * SIGTRAP.  (Because it must have been raised by an
     * |int3|, but not one we injected.)  Not for the
     * debugger, although we'll end up notifying it
     * anyway. */
    assert(DETERMINISTIC_SIG == deterministic);
    return TRAP_NONE;
  }

  if (DETERMINISTIC_SIG == deterministic) {
    /* If the delivery of SIGTRAP is supposed to be
     * deterministic and we didn't just retire an |int 3|
     * and this wasn't a breakpoint, we must have been
     * single stepping.  So definitely for the
     * debugger. */
    assert(constraints.is_singlestep());
    return TRAP_STEPI;
  }

  /* We're replaying an async signal. */

  if (AT_TARGET == exec_state) {
    /* If we're at the target of the async signal
     * delivery, prefer delivering the signal to retiring
     * a possible debugger single-step; we'll notify the
     * debugger anyway. */
    return TRAP_NONE;
  }

  /* Otherwise, we're not at the execution target, so may have
   * been internally single-stepping.  We'll notify the debugger
   * if it was also requesting single-stepping.  The debugger
   * won't care about the rr-internal trap if it wasn't
   * requesting single-stepping. */
  return constraints.is_singlestep() ? TRAP_STEPI : TRAP_NONE;
}

/**
 * Shortcut for callers that don't care about internal breakpoints.
 * Return nonzero if |t|'s |pending_sig()| is for the debugger, zero otherwise.
 */
bool ReplaySession::is_debugger_trap(Task* t, int target_sig,
                                     SignalDeterministic deterministic,
                                     ExecStateType exec_state,
                                     const StepConstraints& constraints) {
  TrapType type =
      compute_trap_type(t, target_sig, deterministic, exec_state, constraints);
  assert(TRAP_BKPT_INTERNAL != type);
  return TRAP_NONE != type;
}

static void guard_overshoot(Task* t, const Registers& target_regs,
                            Ticks target_ticks, Ticks remaining_ticks,
                            const Registers* closest_matching_regs) {
  if (remaining_ticks < 0) {
    remote_code_ptr target_ip = target_regs.ip();

    /* Cover up the internal breakpoint that we may have
     * set, and restore the tracee's $ip to what it would
     * have been had it not hit the breakpoint (if it did
     * hit the breakpoint).*/
    t->vm()->remove_breakpoint(target_ip, TRAP_BKPT_INTERNAL);
    if (t->regs().ip() == target_ip.increment_by_bkpt_insn_length(t->arch())) {
      t->move_ip_before_breakpoint();
    }
    if (closest_matching_regs) {
      LOG(error)
          << "Replay diverged; target registers at ticks target mismatched: ";
      Registers::compare_register_files(t, "rep overshoot", t->regs(), "rec",
                                        *closest_matching_regs, LOG_MISMATCHES);
    } else {
      LOG(error) << "Replay diverged; target registers mismatched: ";
      Registers::compare_register_files(t, "rep overshoot", t->regs(), "rec",
                                        target_regs, LOG_MISMATCHES);
    }
    ASSERT(t, false) << "overshot target ticks=" << target_ticks << " by "
                     << -remaining_ticks;
  }
}

static void guard_unexpected_signal(Task* t) {
  if (ReplaySession::is_ignored_signal(t->pending_sig()) ||
      SIGTRAP == t->pending_sig()) {
    return;
  }

  Event ev;
  if (t->pending_sig()) {
    ev = SignalEvent(t->pending_sig(), NONDETERMINISTIC_SIG, t->arch());
  } else {
    ev = SyscallEvent(max(0L, (long)t->regs().original_syscallno()), t->arch());
  }
  ASSERT(t, false) << "Replay got unrecorded event " << ev
                   << " while awaiting signal";
}

static bool is_same_execution_point(Task* t, const Registers& rec_regs,
                                    Ticks ticks_left,
                                    Registers* mismatched_regs,
                                    const Registers** mismatched_regs_ptr) {
  MismatchBehavior behavior =
#ifdef DEBUGTAG
      LOG_MISMATCHES
#else
      EXPECT_MISMATCHES
#endif
      ;
  if (ticks_left != 0) {
    LOG(debug) << "  not same execution point: " << ticks_left
               << " ticks left (@" << rec_regs.ip() << ")";
#ifdef DEBUGTAG
    Registers::compare_register_files(t, "(rep)", t->regs(), "(rec)", rec_regs,
                                      LOG_MISMATCHES);
#endif
    return false;
  }
  if (!Registers::compare_register_files(t, "rep", t->regs(), "rec", rec_regs,
                                         behavior)) {
    LOG(debug) << "  not same execution point: regs differ (@" << rec_regs.ip()
               << ")";
    *mismatched_regs = t->regs();
    *mismatched_regs_ptr = mismatched_regs;
    return false;
  }
  LOG(debug) << "  same execution point";
  return true;
}

/**
 * Run execution forwards for |t| until |ticks| is reached, and the $ip
 * reaches the recorded $ip.  Return COMPLETE if successful or INCOMPLETE if an
 * unhandled interrupt occurred.  |sig| is the pending signal to be
 * delivered; it's only used to distinguish debugger-related traps
 * from traps related to replaying execution.  |ticks| is an inout param
 * that will be decremented by branches retired during this attempted
 * step.
 */
Completion ReplaySession::advance_to(Task* t, const Registers& regs, int sig,
                                     const StepConstraints& constraints,
                                     Ticks ticks) {
  remote_code_ptr ip = regs.ip();
  Ticks ticks_left;
  bool did_set_internal_breakpoint = false;

  /* Step 1: advance to the target ticks (minus a slack region) as
   * quickly as possible by programming the hpc. */
  ticks_left = ticks - t->tick_count();

  LOG(debug) << "advancing " << ticks_left << " ticks to reach " << ticks << "/"
             << ip;

  /* XXX should we only do this if (ticks > 10000)? */
  while (ticks_left - SKID_SIZE > SKID_SIZE) {
    LOG(debug) << "  programming interrupt for " << (ticks_left - SKID_SIZE)
               << " ticks";

    continue_or_step(t, constraints, (TicksRequest)(ticks_left - SKID_SIZE));
    guard_unexpected_signal(t);

    ticks_left = ticks - t->tick_count();

    if (SIGTRAP == t->pending_sig()) {
      /* We proved we're not at the execution
       * target, and we haven't set any internal
       * breakpoints, and we're not temporarily
       * internally single-stepping, so we must have
       * hit a debugger breakpoint or the debugger
       * was single-stepping the tracee.  (The
       * debugging code will verify that.) */
      return INCOMPLETE;
    }
  }
  guard_overshoot(t, regs, ticks, ticks_left, NULL);

  /* True when our advancing has triggered a tracee SIGTRAP that needs to
   * be dealt with. */
  bool pending_SIGTRAP = false;

  /* Step 2: more slowly, find our way to the target ticks and
   * execution point.  We set an internal breakpoint on the
   * target $ip and then resume execution.  When that *internal*
   * breakpoint is hit (i.e., not one incidentally also set on
   * that $ip by the debugger), we check again if we're at the
   * target ticks and execution point.  If not, we temporarily
   * remove the breakpoint, single-step over the insn, and
   * repeat.
   *
   * What we really want to do is set a (precise)
   * retired-instruction interrupt and do away with all this
   * cruft. */
  Registers mismatched_regs;
  const Registers* mismatched_regs_ptr = NULL;
  while (true) {
    /* Invariants here are
     *  o ticks_left is up-to-date
     *  o ticks_left >= 0
     *
     * Possible state of the execution of |t|
     *  0. at a debugger trap (breakpoint or stepi)
     *  1. at an internal breakpoint
     *  2. at the execution target
     *  3. not at the execution target, but incidentally
     *     at the target $ip
     *  4. otherwise not at the execution target
     *
     * Determining whether we're at a debugger trap is
     * surprisingly complicated, but we delegate the work
     * to |compute_debugger_trap()|.  The rest can be
     * straightforwardly computed with ticks value and
     * registers. */
    bool at_target = is_same_execution_point(
        t, regs, ticks_left, &mismatched_regs, &mismatched_regs_ptr);
    if (pending_SIGTRAP) {
      TrapType trap_type =
          compute_trap_type(t, sig, NONDETERMINISTIC_SIG,
                            at_target ? AT_TARGET : NOT_AT_TARGET, constraints);
      switch (trap_type) {
        case TRAP_BKPT_USER:
        case TRAP_STEPI:
          /* Case (0) above: interrupt for the
           * debugger. */
          LOG(debug) << "    trap was debugger interrupt " << trap_type;
          if (did_set_internal_breakpoint) {
            t->vm()->remove_breakpoint(ip, TRAP_BKPT_INTERNAL);
            did_set_internal_breakpoint = false;
          }
          return INCOMPLETE;
        case TRAP_BKPT_INTERNAL: {
          /* Case (1) above: cover the tracks of
           * our internal breakpoint, and go
           * check again if we're at the
           * target. */
          LOG(debug) << "    trap was for target $ip";
          /* (The breakpoint would have trapped
           * at the $ip one byte beyond the
           * target.) */
          assert(!at_target);

          pending_SIGTRAP = false;
          t->move_ip_before_breakpoint();
          /* We just backed up the $ip, but
           * rewound it over an |int $3|
           * instruction, which couldn't have
           * retired a branch.  So we don't need
           * to adjust |ticks_count()|. */
          continue;
        }
        case TRAP_NONE:
          /* Otherwise, we must have been forced
           * to single-step because the tracee's
           * $ip was incidentally the same as
           * the target.  Unfortunately, it's
           * awkward to assert that here, so we
           * don't yet.  TODO. */
          LOG(debug) << "    (SIGTRAP; stepi'd target $ip)";
          break;
      }
    }
    /* We had to keep the internal breakpoint set (if it
     * was when we entered the loop) for the checks above.
     * But now we're either done (at the target) or about
     * to resume execution in one of a variety of ways,
     * and it's simpler to start out knowing that the
     * breakpoint isn't set. */
    if (did_set_internal_breakpoint) {
      t->vm()->remove_breakpoint(ip, TRAP_BKPT_INTERNAL);
      did_set_internal_breakpoint = false;
    }

    if (at_target) {
      /* Case (2) above: done. */
      return COMPLETE;
    }

    /* At this point, we've proven that we're not at the
     * target execution point, and we've ensured the
     * internal breakpoint is unset. */
    if (USE_BREAKPOINT_TARGET && regs.ip() != t->regs().ip()) {
      /* Case (4) above: set a breakpoint on the
       * target $ip and PTRACE_CONT in an attempt to
       * execute as many non-trapped insns as we
       * can.  (Unless the debugger is stepping, of
       * course.)  Trapping and checking
       * are-we-at-target is slow.  It bears
       * repeating that the ideal implementation
       * would be programming a precise counter
       * interrupt (insns-retired best of all), but
       * we're forced to be conservative by observed
       * imprecise counters.  This should still be
       * no slower than single-stepping our way to
       * the target execution point. */
      LOG(debug) << "    breaking on target $ip";
      t->vm()->add_breakpoint(ip, TRAP_BKPT_INTERNAL);
      did_set_internal_breakpoint = true;
      continue_or_step(t, constraints, RESUME_UNLIMITED_TICKS);
    } else {
      /* Case (3) above: we can't put a breakpoint
       * on the $ip, because resuming execution
       * would just trap and we'd be back where we
       * started.  Single-step or fast-forward past it. */
      LOG(debug) << "    (fast-forwarding over target $ip)";
      if (constraints.command == RUN_SINGLESTEP) {
        continue_or_step(t, constraints, RESUME_UNLIMITED_TICKS);
      } else {
        vector<const Registers*> states = constraints.stop_before_states;
        // This state may not be relevant if we don't have the correct tick
        // count yet. But it doesn't hurt to push it on anyway.
        states.push_back(&regs);
        did_fast_forward |=
            fast_forward_through_instruction(t, RESUME_SINGLESTEP, states);
        check_pending_sig(t);
      }
    }
    pending_SIGTRAP = SIGTRAP == t->pending_sig();

    /* Maintain the "'ticks_left'-is-up-to-date"
     * invariant. */
    ticks_left = ticks - t->tick_count();

    /* Sometimes (e.g. in the ptrace_signal_32 test), we're in almost
     * the correct state when we enter |advance_to|, except that exotic
     * registers (i.e. segment registers) need to be normalized by the kernel
     * by continuing and hitting a deterministic signal without actually
     * advancing execution. So we allow |advance_to| to proceed and actually
     * reach the desired state.
     */
    if (!is_same_execution_point(t, regs, ticks_left, &mismatched_regs,
                                 &mismatched_regs_ptr)) {
      guard_unexpected_signal(t);
    }

    guard_overshoot(t, regs, ticks, ticks_left, mismatched_regs_ptr);
  }
}

static bool is_fatal_default_action(int sig) {
  signal_action action = default_action(sig);
  return action == DUMP_CORE || action == TERMINATE;
}

/**
 * Emulates delivery of |sig| to |oldtask|.  Returns INCOMPLETE if
 * emulation was interrupted, COMPLETE if completed.
 */
Completion ReplaySession::emulate_signal_delivery(
    Task* oldtask, int sig, const StepConstraints& constraints) {
  Task* t = current_task();
  if (!t) {
    // Trace terminated abnormally.  We'll pop out to code
    // that knows what to do.
    return INCOMPLETE;
  }
  ASSERT(oldtask, t == oldtask) << "emulate_signal_delivery changed task";

  const Event& ev = trace_frame.event();
  ASSERT(t, ev.type() == EV_SIGNAL_DELIVERY || ev.type() == EV_SIGNAL_HANDLER)
      << "Unexpected signal disposition";
  // Entering a signal handler seems to clear FP/SSE registers for some
  // reason. So we saved those cleared values, and now we restore that
  // state so they're cleared during replay.
  if (ev.type() == EV_SIGNAL_HANDLER) {
    t->set_extra_regs(trace_frame.extra_regs());
  }

  /* Restore the signal-hander frame data, if there was one. */
  SignalDeterministic deterministic = ev.Signal().deterministic;
  bool restored_sighandler_frame = 0 < t->set_data_from_trace();
  if (restored_sighandler_frame) {
    t->push_event(SignalEvent(sig, deterministic, t->arch()));
    t->ev().transform(EV_SIGNAL_DELIVERY);
    LOG(debug) << "--> restoring sighandler frame for " << signal_name(sig);
    t->ev().transform(EV_SIGNAL_HANDLER);
  }
  // Note that fatal signals are not actually injected into the task!
  // This is very important; we must never actually inject fatal signals
  // into a task. All replay task death must go through exit_task.
  /* If this signal had a user handler, and we just set up the
   * callframe, and we need to restore the $sp for continued
   * execution. */
  t->set_regs(trace_frame.regs());

  t->validate_regs();
  return COMPLETE;
}

void ReplaySession::check_ticks_consistency(Task* t, const Event& ev) {
  if (!can_validate()) {
    return;
  }

  Ticks ticks_now = t->tick_count();
  Ticks trace_ticks = trace_frame.ticks();

  ASSERT(t, ticks_now == trace_ticks) << "ticks mismatch for '" << ev
                                      << "'; expected " << trace_ticks
                                      << ", got " << ticks_now << "";
}

static bool treat_signal_event_as_deterministic(const SignalEvent& ev) {
  return ev.deterministic == DETERMINISTIC_SIG;
}

/**
 * Advance to the delivery of the deterministic signal |sig| and
 * update registers to what was recorded.  Return COMPLETE if successful or
 * INCOMPLETE  if an unhandled interrupt occurred.
 */
Completion ReplaySession::emulate_deterministic_signal(
    Task* t, int sig, const StepConstraints& constraints) {
  if (t->regs().matches(trace_frame.regs()) &&
      t->tick_count() == trace_frame.ticks()) {
    // We're already at the target. This can happen when multiple signals
    // are delivered with no intervening execution.
    return COMPLETE;
  }

  continue_or_step(t, constraints, RESUME_UNLIMITED_TICKS);
  if (is_ignored_signal(t->pending_sig())) {
    return emulate_deterministic_signal(t, sig, constraints);
  }
  if (SIGTRAP == t->pending_sig() &&
      is_debugger_trap(t, sig, DETERMINISTIC_SIG, UNKNOWN, constraints)) {
    return INCOMPLETE;
  }
  ASSERT(t, t->pending_sig() == sig) << "Replay got unrecorded signal "
                                     << t->pending_sig() << " (expecting "
                                     << sig << ")";
  const Event& ev = trace_frame.event();
  check_ticks_consistency(t, ev);

  if (EV_SEGV_RDTSC == ev.type()) {
    t->set_regs(trace_frame.regs());
  }

  return COMPLETE;
}

/**
 * Run execution forwards for |t| until |t->trace.ticks| is reached,
 * and the $ip reaches the recorded $ip.  After that, deliver |sig| if
 * nonzero.  Return COMPLETE if successful or INCOMPLETE if an unhandled
 * interrupt occurred.
 */
Completion ReplaySession::emulate_async_signal(
    Task* t, const StepConstraints& constraints, Ticks ticks) {
  return advance_to(t, trace_frame.regs(), 0, constraints, ticks);
}

/**
 * Restore the recorded syscallbuf data to the tracee, preparing the
 * tracee for replaying the records.  Return the number of record
 * bytes and a pointer to the first record through outparams.
 */
void ReplaySession::prepare_syscallbuf_records(Task* t) {
  // Read the recorded syscall buffer back into the buffer
  // region.
  auto buf = t->trace_reader().read_raw_data();
  ASSERT(t, buf.data.size() >= sizeof(struct syscallbuf_hdr));
  ASSERT(t, buf.data.size() <= SYSCALLBUF_BUFFER_SIZE);
  ASSERT(t, buf.addr == t->syscallbuf_child.cast<void>());

  struct syscallbuf_hdr recorded_hdr;
  memcpy(&recorded_hdr, buf.data.data(), sizeof(struct syscallbuf_hdr));
  // Don't overwrite t->syscallbuf_hdr. That needs to keep tracking the current
  // syscallbuf state.
  memcpy(t->syscallbuf_hdr + 1, buf.data.data() + sizeof(struct syscallbuf_hdr),
         buf.data.size() - sizeof(struct syscallbuf_hdr));

  ASSERT(t, recorded_hdr.num_rec_bytes + sizeof(struct syscallbuf_hdr) <=
                SYSCALLBUF_BUFFER_SIZE);

  current_step.flush.stop_breakpoint_addr =
      t->stopping_breakpoint_table.to_data_ptr<void>().as_int() +
      (recorded_hdr.num_rec_bytes / 8) *
          t->stopping_breakpoint_table_entry_size;

  LOG(debug) << "Prepared " << (uint32_t)recorded_hdr.num_rec_bytes
             << " bytes of syscall records";
}

/**
 * Replay all the syscalls recorded in the interval between |t|'s
 * current execution point and the next non-syscallbuf event (the one
 * that flushed the buffer).  Return COMPLETE if successful or INCOMPLETE if an
 * unhandled interrupt occurred.
 */
Completion ReplaySession::flush_syscallbuf(Task* t,
                                           const StepConstraints& constraints) {
  struct syscallbuf_record* next_rec = next_record(t->syscallbuf_hdr);

  TicksRequest ticks_request;
  if (!compute_ticks_request(t, constraints, &ticks_request)) {
    return INCOMPLETE;
  }

  bool added = t->vm()->add_breakpoint(current_step.flush.stop_breakpoint_addr,
                                       TRAP_BKPT_INTERNAL);
  ASSERT(t, added);
  continue_or_step(t, constraints, ticks_request, RESUME_CONT);
  bool user_breakpoint_at_addr =
      t->vm()->get_breakpoint_type_at_addr(
          current_step.flush.stop_breakpoint_addr) != TRAP_BKPT_INTERNAL;
  t->vm()->remove_breakpoint(current_step.flush.stop_breakpoint_addr,
                             TRAP_BKPT_INTERNAL);

  // Account for buffered syscalls just completed
  struct syscallbuf_record* end_rec = next_record(t->syscallbuf_hdr);
  while (next_rec != end_rec) {
    accumulate_syscall_performed();
    maybe_gc_emufs(t->arch(), next_rec->syscallno);
    next_rec = (struct syscallbuf_record*)((uint8_t*)next_rec +
                                           stored_record_size(next_rec->size));
  }

  if (t->pending_sig() == PerfCounters::TIME_SLICE_SIGNAL) {
    // This would normally be triggered by constraints.ticks_target but it's
    // also possible to get stray signals here.
    return INCOMPLETE;
  }

  if (is_ignored_signal(t->pending_sig())) {
    return flush_syscallbuf(t, constraints);
  }

  ASSERT(t, t->pending_sig() == SIGTRAP)
      << "Replay got unexpected signal (or none) " << t->pending_sig();
  if (t->ip().decrement_by_bkpt_insn_length(t->arch()) ==
          remote_code_ptr(current_step.flush.stop_breakpoint_addr) &&
      !user_breakpoint_at_addr) {
    Registers r = t->regs();
    r.set_ip(current_step.flush.stop_breakpoint_addr);
    t->set_regs(r);
    return COMPLETE;
  }

  return INCOMPLETE;
}

Completion ReplaySession::patch_next_syscall(
    Task* t, const StepConstraints& constraints) {
  if (cont_syscall_boundary(t, constraints) == INCOMPLETE) {
    return INCOMPLETE;
  }

  t->exit_syscall_and_prepare_restart();

  // All patching effects have been recorded to the trace.
  // First, replay any memory mapping done by Monkeypatcher. There should be
  // at most one but we might as well be general.
  while (true) {
    TraceReader::MappedData data;
    bool found;
    KernelMapping km = t->trace_reader().read_mapped_region(&data, &found);
    if (!found) {
      break;
    }
    AutoRemoteSyscalls remote(t);
    ASSERT(t, km.flags() & MAP_ANONYMOUS);
    remote.infallible_mmap_syscall(km.start(), km.size(), km.prot(),
                                   km.flags() | MAP_FIXED, -1, 0);
    t->vm()->map(km.start(), km.size(), km.prot(), km.flags(), 0, string(),
                 KernelMapping::NO_DEVICE, KernelMapping::NO_INODE, &km,
                 TraceWriter::PATCH_MAPPING);
  }

  // Now replay all data records.
  t->apply_all_data_records_from_trace();
  return COMPLETE;
}

/**
 * Return true if replaying |ev| by running |step| should result in
 * the target task having the same ticks value as it did during
 * recording.
 */
static bool has_deterministic_ticks(const Event& ev,
                                    const ReplayTraceStep& step) {
  if (ev.has_ticks_slop()) {
    return false;
  }
  // We won't necessarily reach the same ticks when replaying an
  // async signal, due to debugger interrupts and other
  // implementation details.  This is checked in |advance_to()|
  // anyway.
  return TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT != step.action;
}

void ReplaySession::check_approaching_ticks_target(
    Task* t, const StepConstraints& constraints, BreakStatus& break_status) {
  if (constraints.ticks_target > 0) {
    Ticks ticks_left = constraints.ticks_target - t->tick_count();
    if (ticks_left <= SKID_SIZE) {
      break_status.approaching_ticks_target = true;
    }
  }
}

Completion ReplaySession::advance_to_ticks_target(
    Task* t, const StepConstraints& constraints) {
  while (true) {
    TicksRequest ticks_request;
    if (!compute_ticks_request(t, constraints, &ticks_request)) {
      return INCOMPLETE;
    }
    continue_or_step(t, constraints, ticks_request);
    if (SIGTRAP == t->pending_sig()) {
      return INCOMPLETE;
    }
  }
}

/**
 * Try to execute |step|, adjusting for |req| if needed.  Return COMPLETE if
 * |step| was made, or INCOMPLETE if there was a trap or |step| needs
 * more work.
 */
Completion ReplaySession::try_one_trace_step(
    Task* t, const StepConstraints& constraints) {
  if (constraints.ticks_target > 0 && !trace_frame.event().has_ticks_slop() &&
      t->current_trace_frame().ticks() > constraints.ticks_target) {
    // Instead of doing this step, just advance to the ticks_target, since
    // that happens before this event completes.
    // Unfortunately we can't do this for TSTEP_FLUSH_SYSCALLBUF
    // because its tick count can't be trusted.
    // cont_syscall_boundary handles the ticks constraint for those cases.
    return advance_to_ticks_target(t, constraints);
  }

  switch (current_step.action) {
    case TSTEP_RETIRE:
      return COMPLETE;
    case TSTEP_ENTER_SYSCALL:
      return enter_syscall(t, constraints);
    case TSTEP_EXIT_SYSCALL:
      return exit_syscall(t, constraints);
    case TSTEP_DETERMINISTIC_SIGNAL:
      return emulate_deterministic_signal(t, current_step.target.signo,
                                          constraints);
    case TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT:
      return emulate_async_signal(t, constraints, current_step.target.ticks);
    case TSTEP_DELIVER_SIGNAL:
      return emulate_signal_delivery(t, current_step.target.signo, constraints);
    case TSTEP_FLUSH_SYSCALLBUF:
      return flush_syscallbuf(t, constraints);
    case TSTEP_PATCH_SYSCALL:
      return patch_next_syscall(t, constraints);
    case TSTEP_EXIT_TASK:
      return exit_task(t, constraints);
    default:
      FATAL() << "Unhandled step type " << current_step.action;
      return COMPLETE;
  }
}

/**
 * Task death during replay always goes through here (except for
 * Session::kill_all_tasks when we forcibly kill all tasks in the session at
 * once). |exit| and |exit_group| syscalls are both emulated so the real
 * task doesn't die until we reach the EXIT/UNSTABLE_EXIT events in the trace.
 * This ensures the real tasks are alive and available as long as our Task
 * object exists, which simplifies code like Session cloning.
 *
 * Killing tasks with fatal signals doesn't work because a fatal signal will
 * try to kill all the tasks in the task group. Instead we inject an |exit|
 * syscall, which is apparently the only way to kill one specific thread.
 */
static void end_task(Task* t) {
  ASSERT(t, t->ptrace_event() != PTRACE_EVENT_EXIT);

  // Emulate what the kernel would do during a task exit. We don't let the
  // kernel do these during replay. The kernel would also do a FUTEX_WAKE on
  // this address, but we don't need to do that.
  if (!t->tid_addr().is_null()) {
    bool ok = true;
    // Ignore writes to invalid locations; the kernel does
    t->write_mem(t->tid_addr(), 0, &ok);
  }

  Registers r = t->regs();
  r.set_ip(t->vm()->privileged_traced_syscall_ip());
  r.set_syscallno(syscall_number_for_exit(t->arch()));
  t->set_regs(r);
  // Enter the syscall.
  t->resume_execution(RESUME_SYSCALL, RESUME_WAIT, RESUME_NO_TICKS);
  ASSERT(t, t->pending_sig() == 0);

  do {
    // Singlestep to collect the PTRACE_EVENT_EXIT event.
    t->resume_execution(RESUME_SINGLESTEP, RESUME_WAIT, RESUME_NO_TICKS);
  } while (t->is_ptrace_seccomp_event() ||
           ReplaySession::is_ignored_signal(t->pending_sig()));

  ASSERT(t, t->ptrace_event() == PTRACE_EVENT_EXIT);
  delete t;
}

Completion ReplaySession::exit_task(Task* t,
                                    const StepConstraints& constraints) {
  ASSERT(t, !t->seen_ptrace_exit_event);
  // Apply robust-futex updates captured during recording.
  t->apply_all_data_records_from_trace();
  end_task(t);
  /* |t| is dead now. */
  gc_emufs();
  return COMPLETE;
}

/**
 * Set up rep_trace_step state in t's Session to start replaying towards
 * the event given by the session's current_trace_frame --- but only if
 * it's not already set up.
 * Return true if we should continue replaying, false if the debugger
 * requested a restart. If this returns false, t's Session state was not
 * modified.
 */
void ReplaySession::setup_replay_one_trace_frame(Task* t) {
  const Event& ev = trace_frame.event();

  LOG(debug) << "[line " << trace_frame.time() << "] " << t->rec_tid
             << ": replaying " << Event(ev) << "; state "
             << (ev.is_syscall_event() ? state_name(ev.Syscall().state)
                                       : " (none)");
  if (t->syscallbuf_hdr) {
    LOG(debug) << "    (syscllbufsz:"
               << (uint32_t)t->syscallbuf_hdr->num_rec_bytes
               << ", abrtcmt:" << bool(t->syscallbuf_hdr->abort_commit)
               << ", locked:" << bool(t->syscallbuf_hdr->locked) << ")";
  }

  /* Ask the trace-interpretation code what to do next in order
   * to retire the current frame. */
  memset(&current_step, 0, sizeof(current_step));

  switch (ev.type()) {
    case EV_UNSTABLE_EXIT:
    case EV_EXIT:
      current_step.action = TSTEP_EXIT_TASK;
      break;
    case EV_SYSCALLBUF_ABORT_COMMIT:
      t->syscallbuf_hdr->abort_commit = 1;
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_SYSCALLBUF_FLUSH:
      current_step.action = TSTEP_FLUSH_SYSCALLBUF;
      prepare_syscallbuf_records(t);
      break;
    case EV_SYSCALLBUF_RESET:
      // Reset syscallbuf_hdr->num_rec_bytes and zero out the recorded data.
      // Zeroing out the data is important because we only save and restore
      // the recorded data area when making checkpoints. We want the checkpoint
      // to have the same syscallbuf contents as its original, i.e. zero outside
      // the recorded data area. This is important because stray reads such
      // as those performed by return_addresses should be consistent.
      t->reset_syscallbuf();
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_PATCH_SYSCALL:
      current_step.action = TSTEP_PATCH_SYSCALL;
      break;
    case EV_SCHED:
      current_step.action = TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT;
      current_step.target.ticks = trace_frame.ticks();
      current_step.target.signo = 0;
      break;
    case EV_SEGV_RDTSC:
      current_step.action = TSTEP_DETERMINISTIC_SIGNAL;
      current_step.target.ticks = -1;
      current_step.target.signo = SIGSEGV;
      break;
    case EV_GROW_MAP:
      process_grow_map(t);
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_INTERRUPTED_SYSCALL_NOT_RESTARTED:
      LOG(debug) << "  popping interrupted but not restarted " << t->ev();
      t->pop_syscall_interruption();
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_EXIT_SIGHANDLER:
      LOG(debug) << "<-- sigreturn from " << t->ev();
      t->pop_signal_handler();
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_SIGNAL:
      if (treat_signal_event_as_deterministic(ev.Signal())) {
        current_step.action = TSTEP_DETERMINISTIC_SIGNAL;
        current_step.target.signo = ev.Signal().siginfo.si_signo;
        current_step.target.ticks = -1;
      } else {
        current_step.action = TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT;
        current_step.target.signo = ev.Signal().siginfo.si_signo;
        current_step.target.ticks = trace_frame.ticks();
      }
      break;
    case EV_SIGNAL_DELIVERY:
    case EV_SIGNAL_HANDLER:
      current_step.action = TSTEP_DELIVER_SIGNAL;
      current_step.target.signo = ev.Signal().siginfo.si_signo;
      break;
    case EV_SYSCALL:
      rep_process_syscall(t, &current_step);
      if (trace_frame.event().Syscall().state == EXITING_SYSCALL &&
          current_step.action == TSTEP_RETIRE) {
        t->on_syscall_exit(current_step.syscall.number, trace_frame.regs());
        maybe_gc_emufs(t->arch(), trace_frame.regs().syscallno());
      }
      break;
    default:
      FATAL() << "Unexpected event " << ev;
  }
}

bool ReplaySession::next_step_is_syscall_exit(int syscallno) {
  return current_step.action == TSTEP_NONE &&
         trace_frame.event().is_syscall_event() &&
         trace_frame.event().Syscall().number == syscallno &&
         trace_frame.event().Syscall().state == EXITING_SYSCALL;
}

ReplayResult ReplaySession::replay_step(const StepConstraints& constraints) {
  finish_initializing();

  ReplayResult result(REPLAY_CONTINUE);

  Task* t = current_task();

  if (EV_TRACE_TERMINATION == trace_frame.event().type()) {
    result.status = REPLAY_EXITED;
    return result;
  }

  /* If we restored from a checkpoint, the steps might have been
   * computed already in which case step.action will not be TSTEP_NONE.
   */
  if (current_step.action == TSTEP_NONE) {
    setup_replay_one_trace_frame(t);
    if (current_step.action == TSTEP_NONE) {
      // Already at the destination event.
      advance_to_next_trace_frame(constraints.stop_at_time);
    }
    if (current_step.action == TSTEP_EXIT_TASK) {
      result.break_status.task = t;
      result.break_status.task_exit = true;
    }
    return result;
  }

  did_fast_forward = false;

  // Now we know |t| hasn't died, so save it in break_status.
  result.break_status.task = t;

  /* Advance towards fulfilling |current_step|. */
  if (try_one_trace_step(t, constraints) == INCOMPLETE) {
    if (EV_TRACE_TERMINATION == trace_frame.event().type()) {
      // An irregular trace step had to read the
      // next trace frame, and that frame was an
      // early-termination marker.  Otherwise we
      // would have seen the marker above.
      result.status = REPLAY_EXITED;
      return result;
    }

    // We got INCOMPLETE because there was some kind of debugger trap or
    // we got close to ticks_target.
    result.break_status = diagnose_debugger_trap(t);
    ASSERT(t, !result.break_status.signal)
        << "Expected either SIGTRAP at $ip " << t->ip()
        << " or USER breakpoint just after it";
    ASSERT(t, !result.break_status.singlestep_complete ||
                  constraints.is_singlestep());

    check_approaching_ticks_target(t, constraints, result.break_status);
    result.did_fast_forward = did_fast_forward;
    return result;
  }

  result.did_fast_forward = did_fast_forward;

  switch (current_step.action) {
    case TSTEP_DETERMINISTIC_SIGNAL:
    case TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT:
      if (trace_frame.event().type() != EV_SEGV_RDTSC) {
        result.break_status.signal = current_step.target.signo;
      }
      if (constraints.is_singlestep()) {
        result.break_status.singlestep_complete = true;
      }
      break;
    case TSTEP_DELIVER_SIGNAL:
      // When we deliver a terminating signal, do not let the singlestep
      // complete; proceed on to report our synthetic SIGKILL or task death.
      if (constraints.is_singlestep() &&
          !(trace_frame.event().type() == EV_SIGNAL_DELIVERY &&
            is_fatal_default_action(current_step.target.signo))) {
        result.break_status.singlestep_complete = true;
      }
      break;
    case TSTEP_EXIT_TASK:
      t = result.break_status.task = nullptr;
      assert(!result.break_status.any_break());
      break;
    case TSTEP_ENTER_SYSCALL:
      cpuid_bug_detector.notify_reached_syscall_during_replay(t);
      break;
    default:
      break;
  }

  if (t) {
    const Event& ev = trace_frame.event();
    if (can_validate() && ev.is_syscall_event() &&
        ::Flags::get().check_cached_mmaps) {
      t->vm()->verify(t);
    }

    if (has_deterministic_ticks(ev, current_step)) {
      check_ticks_consistency(t, ev);
    }

    debug_memory(t);

    check_for_watchpoint_changes(t, result.break_status);
    check_approaching_ticks_target(t, constraints, result.break_status);
  }

  // Advance to next trace frame before doing rep_after_enter_syscall,
  // so that FdTable notifications run with the same trace timestamp during
  // replay as during recording
  advance_to_next_trace_frame(constraints.stop_at_time);
  if (TSTEP_ENTER_SYSCALL == current_step.action) {
    // Advance to next trace frame before we call rep_after_enter_syscall,
    // since that matches what we do during recording and it matters for
    // reporting event numbers on stdio.
    rep_after_enter_syscall(t, current_step.syscall.number);
  }
  // Record that this step completed successfully.
  current_step.action = TSTEP_NONE;

  Task* next_task = current_task();
  if (next_task && !next_task->vm()->first_run_event() && can_validate()) {
    next_task->vm()->set_first_run_event(trace_frame.time());
  }
  if (next_task) {
    ticks_at_start_of_event = next_task->tick_count();
  }

  return result;
}
