///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "inv_mass.h"
#include "rheolef/ublas_matrix_range.h"
#include "rheolef/ublas-invert.h"
using namespace rheolef;
using namespace std;
using namespace ublas;

void
inv_mass::operator() (const geo_element& K, ublas::matrix<Float>& inv_m) const
{
  // space is vectorial or tensorial case : weight could be scalar, tensor or tensor4
  size_type n_comp = get_first_space().n_component();
  bool singular = false;
  if (n_comp == 1) {
    ublas::matrix<Float> m;
    build_scalar_mass (K, m);
    inv_m = invert (m, singular);
    check_macro (!singular, "singular mass element matrix on " << K.index() << "-th element");
    return;
  }
  reference_element hat_K = K;
  size_type n = get_second_basis().size(hat_K);
  fem_helper::coordinate_type   sys_coord     = get_first_space().coordinate_system_type();
  fem_helper::valued_field_type space_valued  = get_first_space().get_valued_type();
  fem_helper::valued_field_type weight_valued = fem_helper::scalar;
  if (is_weighted()) {
    weight_valued = _wh.get_valued_type();
  }
  if (weight_valued == fem_helper::scalar) {
    // --------------------------------------------------------------------------
    // all components have the same weight and the local matrix is block-diagonal
    // --------------------------------------------------------------------------
    ublas::matrix<Float> m_ij;
    build_scalar_mass (K, m_ij);
    check_macro (m_ij.size1() == m_ij.size2(),
  	"inv_mass: incompatible `" << get_first_basis().name()
  	<< "' and `" << get_second_basis().name() << "' basis");
    matrix<Float> inv_m_ij = invert (m_ij, singular);
    check_macro (!singular, "singular weighted-mass element matrix on " << K.index() << "-th element");
    inv_m.resize (n_comp*n, n_comp*n);
    inv_m.clear();
    fem_helper::valued_field_type valued = get_first_space().get_valued_type();
    switch (valued) {
     case fem_helper::scalar:
     case fem_helper::vectorial:
     case fem_helper::unsymmetric_tensorial: {
      for (size_type i = 0; i < n_comp; i++)
        mr_set (inv_m, range(i*n,(i+1)*n), range(i*n,(i+1)*n), inv_m_ij);  
      break;
     }
     case fem_helper::tensorial: {
      fem_helper::coordinate_type sys_coord = get_first_space().coordinate_system_type();
      // symmetric tensor => 1/2 factor for (12,13,23) subscripts
      for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
        fem_helper::pair_size_type ij = fem_helper::tensor_subscript (valued, sys_coord, ij_comp);
        size_t i = ij.first;
        size_t j = ij.second;
        Float factor_ij = ((i == j) ? 1 : 0.5); // symmetry => multiplicity factor: when i!=j : 2*w_ij
        mr_set (inv_m, range(ij_comp*n,(ij_comp+1)*n), range(ij_comp*n,(ij_comp+1)*n), factor_ij*inv_m_ij);  
      }
      break;
     }
     default: {
      string valued_name = get_first_space().get_valued();
      error_macro ("unexpected `" << valued_name << "'-valued space for the `inv_mass' form.");
     }
    }
  } else { // weight_valued != fem_helper::scalar
    // ------------------------------------------------------------------------------
    // components have different weights : the local matrix is no more block-diagonal
    // ------------------------------------------------------------------------------
    ublas::matrix<Float> m;
    build_general_mass (K, m);
    inv_m = invert (m, singular);
    check_macro (!singular, "singular weighted-mass element matrix on " << K.index() << "-th element");
#undef DO_CHECK_INV
#ifdef  DO_CHECK_INV
    ublas::matrix<Float> id;
    id.resize (m.size1(), m.size2());
    id.clear();
    for (size_t i = 0; i < id.size1(); i++) id(i,i)=1;
    ublas::matrix<Float> check1 = prod(m,inv_m) - id;
    Float err = 0;
    for (size_t i = 0; i < id.size1(); i++)
      for (size_t j = 0; j < id.size2(); j++)
	    err += sqr(check1(i,j));
    if (err > 1e-16) {
	warning_macro ("K" << K.index() << ": invert error = " << err);
        cerr << setprecision(16);
        cerr << "M" << K.index() << " = [";
        for (size_t i = 0; i < id.size1(); i++) {
          for (size_t j = 0; j < id.size2(); j++) {
	    if (j == 0) cerr << "      ";
            cerr << m(i,j);
	    if (j != id.size2()-1) cerr << ",";
	    else if (i != id.size1()-1) cerr << ";" << endl;
	    else cerr << "];" << endl;
          }
        }
        cerr << "cond(M" << K.index() << ")" << endl;
    }
#endif //  DO_CHECK_INV
  }
}
void
inv_mass::check_after_initialize () const
{
  check_macro ( "P0" == get_first_space().get_approx()
              || "P1d" == get_first_space().get_approx()
              || "P2d" == get_first_space().get_approx(),
	"unsupported continuous approximation space for `inv_mass' form");
  
  check_macro (get_first_space().get_approx() == get_second_space().get_approx(),
	"unsupported different approximation space for `inv_mass' form");

  // suppose also that multi-component spaces are homogeneous,
  // i.e. that all components have the same approx
  check_macro (get_first_space().n_component() == get_second_space().n_component(),
    "incompatible spaces for the `mass' form.");
  if (is_weighted()) {
    fem_helper::valued_field_type weight_valued = _wh.get_valued_type();
    if (weight_valued != fem_helper::scalar) {
      // first case : vector space and tensor weight
      //     m(u,v) = int_Omega w_ij u_j v_i dx
      // second case : tensor space and tensor4 weight
      //     m(tau,gamma) = int_Omega w_ijkl tau_kj gamma_ij dx
      fem_helper::valued_field_type space_valued = get_first_space().get_valued_type();
      if (weight_valued == fem_helper::tensorial) {
	  check_macro (space_valued == fem_helper::vectorial, "inv_mass form: unexpected tensorial weight and "
			  << get_first_space().get_valued() << " space");
      } else if (weight_valued == fem_helper::tensorial_4) {
	  check_macro (space_valued == fem_helper::tensorial, "inv_mass form: unexpected tensorial weight and "
			  << get_first_space().get_valued() << " space");
      } else {
          error_macro ("inv_mass form: unexpected " << _wh.get_valued() << " weight and "
			  << get_first_space().get_valued() << " space");
      }
    }
  }
}
