% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/callr-package.R
\docType{package}
\name{callr-package}
\alias{callr}
\alias{callr-package}
\title{Call R from R}
\description{
It is sometimes useful to perform a computation in a separate
R process, without affecting the current R process at all.  This
packages does exactly that.
}
\section{callr}{
\subsection{Features}{
\itemize{
\item Calls an R function, with arguments, in a subprocess.
\item Copies function arguments to the subprocess and copies the return value of the function back, seamlessly.
\item Copies error objects back from the subprocess, including a stack trace.
\item Shows and/or collects the standard output and standard error of the subprocess.
\item Supports both one-off and persistent R subprocesses.
\item Calls the function synchronously or asynchronously (in the background).
\item Can call \verb{R CMD} commands, synchronously or asynchronously.
\item Can call R scripts, synchronously or asynchronously.
\item Provides extensible \code{r_process}, \code{rcmd_process} and \code{rscript_process} R6 classes, based on \code{processx::process}.
}
}

\subsection{Installation}{

Install the stable version from CRAN:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{install.packages("callr")
}\if{html}{\out{</div>}}

Install the development version from GitHub:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pak::pak("r-lib/callr")
}\if{html}{\out{</div>}}
}

\subsection{Synchronous, one-off R processes}{

Use \code{r()} to run an R function in a new R process.
The results are passed back seamlessly:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{callr::r(function() var(iris[, 1:4]))
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#>              Sepal.Length Sepal.Width Petal.Length Petal.Width                  
#> Sepal.Length    0.6856935  -0.0424340    1.2743154   0.5162707                  
#> Sepal.Width    -0.0424340   0.1899794   -0.3296564  -0.1216394                  
#> Petal.Length    1.2743154  -0.3296564    3.1162779   1.2956094                  
#> Petal.Width     0.5162707  -0.1216394    1.2956094   0.5810063                  
</pre></div>
}}

\subsection{Passing arguments}{

You can pass arguments to the function by setting \code{args} to the list of arguments.
This is often necessary as these arguments are explicitly copied to the child process, whereas the evaluated function cannot refer to variables in the parent.
For example, the following does not work:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mycars <- cars
callr::r(function() summary(mycars))
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> <span style="font-weight: bold;color: #B58900;">Error</span>:                                                                          
#> <span style="color: #B58900;">!</span> in callr subprocess.                                                          
#> <span style="font-weight: bold;">Caused by error</span> in `summary(mycars)`:                                           
#> <span style="color: #B58900;">!</span> object 'mycars' not found                                                     
#> <span style="color: #525252;">Type .Last.error to see the more details.</span>                                       
</pre></div>
}}


But this does:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mycars <- cars
callr::r(function(x) summary(x), args = list(mycars))
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#>      speed           dist                                                       
#>  Min.   : 4.0   Min.   :  2.00                                                  
#>  1st Qu.:12.0   1st Qu.: 26.00                                                  
#>  Median :15.0   Median : 36.00                                                  
#>  Mean   :15.4   Mean   : 42.98                                                  
#>  3rd Qu.:19.0   3rd Qu.: 56.00                                                  
#>  Max.   :25.0   Max.   :120.00                                                  
</pre></div>
}}


Note that the arguments will be serialized and saved to a file, so if they are large R objects, it might take a long time for the child process to start up.
}

\subsection{Using packages}{

You can use any R package in the child process, just make sure to refer to it explicitly with the \code{::} operator.
For example, the following code creates an \href{https://github.com/igraph/rigraph}{igraph} graph in the child, and calculates some metrics of it.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{callr::r(function() \{ g <- igraph::sample_gnp(1000, 4/1000); igraph::diameter(g) \})
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] 11                                                                          
</pre></div>
}}

}

\subsection{Error handling}{

callr copies errors from the child process back to the main R session:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{callr::r(function() 1 + "A")
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> <span style="font-weight: bold;color: #B58900;">Error</span>:                                                                          
#> <span style="color: #B58900;">!</span> in callr subprocess.                                                          
#> <span style="font-weight: bold;">Caused by error</span> in `1 + "A"`:                                                   
#> <span style="color: #B58900;">!</span> non-numeric argument to binary operator                                       
#> <span style="color: #525252;">Type .Last.error to see the more details.</span>                                       
</pre></div>
callr sets the `.Last.error` variable, and after an error you can inspect this for more details about the error, including stack traces both from the main R process and the subprocess.
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{.Last.error
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> <span style="font-weight: bold;color: #B58900;">Error</span>:                                                                          
#> <span style="color: #B58900;">!</span> in callr subprocess.                                                          
#> <span style="font-weight: bold;">Caused by error</span> in `1 + "A"`:                                                   
#> <span style="color: #B58900;">!</span> non-numeric argument to binary operator                                       
#> ---                                                                             
#> Backtrace:                                                                      
#> <span style="color: #525252;">1. </span>callr::<span style="color: #2AA198;">r</span><span style="color: #B58900;">(</span><span style="color: #DC322F;">function</span><span style="color: #268BD2;">()</span> <span style="color: #268BD2;">1</span> <span style="color: #859900;">+</span> <span style="color: #B58900;">"A")</span>                                                 
#> <span style="color: #525252;">2. </span>callr:::<span style="color: #2AA198;">get_result</span><span style="color: #B58900;">(</span>output = out, options<span style="color: #B58900;">)</span>                                    
#> <span style="color: #525252;">3. </span>callr:::<span style="color: #2AA198;">throw</span><span style="color: #B58900;">(</span><span style="color: #2AA198;">callr_remote_error</span><span style="color: #268BD2;">(</span>remerr, output<span style="color: #268BD2;">)</span>, parent = <span style="color: #2AA198;">fix_msg</span><span style="color: #268BD2;">(</span>remerr<span style="color: #2AA198;">[[</span><span style="color: #268BD2;">3</span><span style="color: #2AA198;">]</span>
#> <span style="color: #2AA198;">]</span><span style="color: #268BD2;">)</span><span style="color: #B58900;">)</span>                                                                             
#> ---                                                                             
#> Subprocess backtrace:                                                           
#> <span style="color: #525252;">1. </span>base::.handleSimpleError(function (e) …                                      
#> <span style="color: #525252;">2. </span>global <span style="color: #2AA198;">h</span><span style="color: #B58900;">(</span><span style="color: #2AA198;">simpleError</span><span style="color: #268BD2;">(</span>msg, call<span style="color: #268BD2;">)</span><span style="color: #B58900;">)</span>                                             
</pre></div>
}}


The error objects has two parts.
The first belongs to the main process, and the second belongs to the subprocess.

\code{.Last.error} also includes a stack trace, that includes both the main R process and the subprocess:

The top part of the trace contains the frames in the main process, and the bottom part contains the frames in the subprocess, starting with the anonymous function.
}

\subsection{Standard output and error}{

By default, the standard output and error of the child is lost, but you can request callr to redirect them to files, and then inspect the files in the parent:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- callr::r(function() \{ print("hello world!"); message("hello again!") \},
  stdout = "/tmp/out", stderr = "/tmp/err"
)
readLines("/tmp/out")
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] "[1] \\"hello world!\\""                                                      
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{readLines("/tmp/err")
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] "hello again!"                                                              
</pre></div>
}}


With the \code{stdout} option, the standard output is collected and can be examined once the child process finished.
The \code{show = TRUE} options will also show the output of the child, as it is printed, on the console of the parent.
}

}

\subsection{Background R processes}{

\code{r_bg()} is similar to \code{r()} but it starts the R process in the background.
It returns an \code{r_process} R6 object, that provides a rich API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rp <- callr::r_bg(function() Sys.sleep(.2))
rp
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> PROCESS 'R', running, pid 58242.                                                
</pre></div>
}}


This is a list of all \code{r_process} methods:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{ls(rp)
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#>  [1] "as_ps_handle"          "clone"                 "finalize"                 
#>  [4] "format"                "get_cmdline"           "get_cpu_times"            
#>  [7] "get_error_connection"  "get_error_file"        "get_exe"                  
#> [10] "get_exit_status"       "get_input_connection"  "get_input_file"           
#> [13] "get_memory_info"       "get_name"              "get_output_connection"    
#> [16] "get_output_file"       "get_pid"               "get_poll_connection"      
#> [19] "get_result"            "get_start_time"        "get_status"               
#> [22] "get_username"          "get_wd"                "has_error_connection"     
#> [25] "has_input_connection"  "has_output_connection" "has_poll_connection"      
#> [28] "initialize"            "interrupt"             "is_alive"                 
#> [31] "is_incomplete_error"   "is_incomplete_output"  "is_supervised"            
#> [34] "kill"                  "kill_tree"             "poll_io"                  
#> [37] "print"                 "read_all_error"        "read_all_error_lines"     
#> [40] "read_all_output"       "read_all_output_lines" "read_error"               
#> [43] "read_error_lines"      "read_output"           "read_output_lines"        
#> [46] "resume"                "signal"                "supervise"                
#> [49] "suspend"               "wait"                  "write_input"              
</pre></div>
}}


These include all methods of the \code{processx::process} superclass and the new \code{get_result()} method, to retrieve the R object returned by the function call.
Some of the handiest methods are:
\itemize{
\item \code{get_exit_status()} to query the exit status of a finished process.
\item \code{get_result()} to collect the return value of the R function call.
\item \code{interrupt()} to send an interrupt to the process. This is equivalent to a \code{CTRL+C} key press, and the R process might ignore it.
\item \code{is_alive()} to check if the process is alive.
\item \code{kill()} to terminate the process.
\item \code{poll_io()} to wait for any standard output, standard error, or the completion of the process, with a timeout.
\item \verb{read_*()} to read the standard output or error.
\item \code{suspend()} and \code{resume()} to stop and continue a process.
\item \code{wait()} to wait for the completion of the process, with a timeout.
}
}

\subsection{Multiple background R processes and \code{poll()}}{

Multiple background R processes are best managed with the \code{processx::poll()} function that waits for events (standard output/error or termination) from multiple processes.
It returns as soon as one process has generated an event, or if its timeout has expired.
The timeout is in milliseconds.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rp1 <- callr::r_bg(function() \{ Sys.sleep(1/2); "1 done" \})
rp2 <- callr::r_bg(function() \{ Sys.sleep(1/1000); "2 done" \})
processx::poll(list(rp1, rp2), 1000)
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [[1]]                                                                           
#>   output    error  process                                                      
#> "silent" "silent" "silent"                                                      
#>                                                                                 
#> [[2]]                                                                           
#>  output   error process                                                         
#> "ready" "ready" "ready"                                                         
#>                                                                                 
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{rp2$get_result()
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] "2 done"                                                                    
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{processx::poll(list(rp1), 1000)
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [[1]]                                                                           
#>   output    error  process                                                      
#> "silent" "silent"  "ready"                                                      
#>                                                                                 
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{rp1$get_result()
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] "1 done"                                                                    
</pre></div>
}}

}

\subsection{Persistent R sessions}{

\code{r_session} is another \code{processx::process} subclass that represents a persistent background R session:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rs <- callr::r_session$new()
rs
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> R SESSION, alive, idle, pid 58288.                                              
</pre></div>
}}


\code{r_session$run()} is a synchronous call, that works similarly to \code{r()}, but uses the persistent session.
\code{r_session$call()} starts the function call and returns immediately.
The \code{r_session$poll_process()} method or \code{processx::poll()} can then be used to wait for the completion or other events from one or more R sessions, R processes or other \code{processx::process} objects.

Once an R session is done with an asynchronous computation, its \code{poll_process()} method returns \code{"ready"} and the \code{r_session$read()} method can read out the result.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rs <- callr::r_session$new()
rs$run(function() runif(10))
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#>  [1] 0.8047354 0.8938617 0.7142338 0.8505395 0.3118376 0.7083882 0.9514367      
#>  [8] 0.2536755 0.6727270 0.3359578                                              
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{rs$call(function() rnorm(10))
rs
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> R SESSION, alive, busy, pid 58294.                                              
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{rs$poll_process(2000)
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> [1] "ready"                                                                     
</pre></div>
}}


\if{html}{\out{<div class="sourceCode r">}}\preformatted{rs$read()
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> $code                                                                           
#> [1] 200                                                                         
#>                                                                                 
#> $message                                                                        
#> [1] "done callr-rs-result-e3324ebebc8b"                                         
#>                                                                                 
#> $result                                                                         
#>  [1] -0.60962697 -0.41063130  0.22121432  1.44621900  0.26890394  0.11432756    
#>  [7] -0.53206118  0.47493933  0.02069551  1.37348004                            
#>                                                                                 
#> $stdout                                                                         
#> [1] ""                                                                          
#>                                                                                 
#> $stderr                                                                         
#> [1] ""                                                                          
#>                                                                                 
#> $error                                                                          
#> NULL                                                                            
#>                                                                                 
#> attr(,"class")                                                                  
#> [1] "callr_session_result"                                                      
</pre></div>
}}

}

\subsection{Running \verb{R CMD} commands}{

The \code{rcmd()} function calls an \verb{R CMD} command.
For example, you can call \verb{R CMD INSTALL}, \verb{R CMD check} or \verb{R CMD config} this way:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{callr::rcmd("config", "CC")
}\if{html}{\out{</div>}}\if{html}{\out{
<div class="asciicast" style="color: #172431;font-family: 'Fira Code',Monaco,Consolas,Menlo,'Bitstream Vera Sans Mono','Powerline Symbols',monospace;line-height: 1.300000"><pre>
#> $status                                                                         
#> [1] 0                                                                           
#>                                                                                 
#> $stdout                                                                         
#> [1] "clang -arch arm64\\n"                                                       
#>                                                                                 
#> $stderr                                                                         
#> [1] ""                                                                          
#>                                                                                 
#> $timeout                                                                        
#> [1] FALSE                                                                       
#>                                                                                 
#> $command                                                                        
#> [1] "/Library/Frameworks/R.framework/Versions/4.2-arm64/Resources/bin/R"        
#> [2] "CMD"                                                                       
#> [3] "config"                                                                    
#> [4] "CC"                                                                        
#>                                                                                 
</pre></div>
}}


This returns a list with three components: the standard output, the standard error, and the exit (status) code of the \verb{R CMD} command.
}

\subsection{Code of Conduct}{

Please note that the callr project is released with a
\href{https://callr.r-lib.org/CODE_OF_CONDUCT.html}{Contributor Code of Conduct}.
By contributing to this project, you agree to abide by its terms.
}
}

\seealso{
Useful links:
\itemize{
  \item \url{https://callr.r-lib.org}
  \item \url{https://github.com/r-lib/callr#readme}
  \item Report bugs at \url{https://github.com/r-lib/callr/issues}
}

}
\author{
\strong{Maintainer}: Gábor Csárdi \email{csardi.gabor@gmail.com} (\href{https://orcid.org/0000-0001-7098-9676}{ORCID}) [copyright holder]

Authors:
\itemize{
  \item Winston Chang
}

Other contributors:
\itemize{
  \item Posit Software, PBC [copyright holder, funder]
  \item Mango Solutions [copyright holder, funder]
}

}
\keyword{internal}
