import torch

from torch_geometric.data import Data
from torch_geometric.testing import withPackage
from torch_geometric.transforms import GDC
from torch_geometric.utils import to_dense_adj


@withPackage('numba')
def test_gdc():
    edge_index = torch.tensor([[0, 0, 1, 1, 2, 2, 2, 3, 3, 4],
                               [1, 2, 0, 2, 0, 1, 3, 2, 4, 3]])

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='sym',
              normalization_out='sym',
              diffusion_kwargs=dict(method='ppr', alpha=0.15),
              sparsification_kwargs=dict(method='threshold',
                                         avg_degree=2), exact=True)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    assert torch.all(mat >= -1e-8)
    assert torch.allclose(mat, mat.t(), atol=1e-4)

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='sym',
              normalization_out='sym',
              diffusion_kwargs=dict(method='heat', t=10),
              sparsification_kwargs=dict(method='threshold',
                                         avg_degree=2), exact=True)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    assert torch.all(mat >= -1e-8)
    assert torch.allclose(mat, mat.t(), atol=1e-4)

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='col',
              normalization_out='col',
              diffusion_kwargs=dict(method='heat', t=10),
              sparsification_kwargs=dict(method='topk', k=2,
                                         dim=0), exact=True)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    col_sum = mat.sum(0)
    assert torch.all(mat >= -1e-8)
    assert torch.all(
        torch.isclose(col_sum, torch.tensor(1.0))
        | torch.isclose(col_sum, torch.tensor(0.0)))
    assert torch.all((~torch.isclose(mat, torch.tensor(0.0))).sum(0) == 2)

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='row',
              normalization_out='row',
              diffusion_kwargs=dict(method='heat', t=5),
              sparsification_kwargs=dict(method='topk', k=2,
                                         dim=1), exact=True)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    row_sum = mat.sum(1)
    assert torch.all(mat >= -1e-8)
    assert torch.all(
        torch.isclose(row_sum, torch.tensor(1.0))
        | torch.isclose(row_sum, torch.tensor(0.0)))
    assert torch.all((~torch.isclose(mat, torch.tensor(0.0))).sum(1) == 2)

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='row',
              normalization_out='row',
              diffusion_kwargs=dict(method='coeff', coeffs=[0.8, 0.3, 0.1]),
              sparsification_kwargs=dict(method='threshold',
                                         eps=0.1), exact=True)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    row_sum = mat.sum(1)
    assert torch.all(mat >= -1e-8)
    assert torch.all(
        torch.isclose(row_sum, torch.tensor(1.0))
        | torch.isclose(row_sum, torch.tensor(0.0)))

    data = Data(edge_index=edge_index, num_nodes=5)
    gdc = GDC(self_loop_weight=1, normalization_in='sym',
              normalization_out='col',
              diffusion_kwargs=dict(method='ppr', alpha=0.15, eps=1e-4),
              sparsification_kwargs=dict(method='threshold',
                                         avg_degree=2), exact=False)
    data = gdc(data)
    mat = to_dense_adj(data.edge_index, edge_attr=data.edge_attr).squeeze()
    col_sum = mat.sum(0)
    assert torch.all(mat >= -1e-8)
    assert torch.all(
        torch.isclose(col_sum, torch.tensor(1.0))
        | torch.isclose(col_sum, torch.tensor(0.0)))
