/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id: line.c 286 2008-10-27 08:46:05Z toady $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <picviz.h>

static PcvID id = 0;


PicvizLine *picviz_line_new(void)
{
        PicvizLine *line;

        line = malloc(sizeof(*line));
        if ( ! line ) {
                fprintf(stderr, "Cannot initialize line: memory exhausted.\n");
                return NULL;
        }
        INIT_LLIST_HEAD(&line->list);
        INIT_LLIST_HEAD(&line->axisplot);
        line->id = id++;
        line->hidden = 0;

        picviz_properties_new(&line->props);
        picviz_properties_set(line->props, "color", "#000000");
        picviz_properties_set(line->props, "penwidth", "0.1");

        return line;
}

void picviz_line_axisplot_append(PicvizLine *line, PicvizAxisPlot *axisplot)
{
        llist_add_tail(&axisplot->list, &line->axisplot);
}


void picviz_line_free(PicvizLine *l)
{
        picviz_properties_destroy(l->props);
        //picviz_axisplot_destroy(l->axisplot);
        free(l);

}

PcvHeight picviz_line_max_get(PicvizImage *image, struct llist_head *line, PcvID axis_id)
{
	PicvizLine *l;
	PicvizAxisPlot *axisplot;
	char init = 0;
	PcvHeight max = 0;
	PcvHeight height = 0;

//	printf("We should return the max value if the axis %d\n", axis_id);

	llist_for_each_entry(l, line, list) {
		llist_for_each_entry(axisplot, &l->axisplot, list) {
			PicvizAxis *axis = (PicvizAxis *)picviz_axis_get(image, axisplot->axis_id);
			//printf("Y VAL=%lld\n", axisplot->y);
			if (picviz_is_string_algo_basic(axis)) {
				height = picviz_line_value_get_from_string_dummy(image, axis, 0, axisplot->strval);
			} else {
				height = picviz_line_value_get_from_string_dummy(image, axis, 1, axisplot->strval);
			}
			if (axis_id == axisplot->axis_id) {
				if ( ! init ) {
//					printf("INIT\n");
					init = 1;
					max = height;
				}

//				printf("max=%lld, axisplot->y =%lld\n", max, height);

				if ( height > max ) max = height;
			}
		}
	}

	return max;
}

PcvHeight picviz_line_max_pertype_get(PicvizImage *image, PicvizDataType type)
{
        PicvizLine *line;
        PicvizAxisPlot *axisplot;
        int i;
        int nb_types = sizeof(PicvizDataType);
        char init[nb_types];
        PcvHeight max[nb_types];

        for ( i = 0; i <= nb_types; i++) {
                init[i] = 0;
        }

        llist_for_each_entry(line, &image->lines, list) {
                llist_for_each_entry(axisplot, &line->axisplot, list) {
                        PicvizAxis *axis = (PicvizAxis *)picviz_axis_get(image, axisplot->axis_id);

                        if (axis->type == type) {
                                if ( ! init[type] ) {
                                        init[type] = 1;
                                        max[type] = axisplot->y;
                                }

                                if ( axisplot->y > max[type] ) max[type] = axisplot->y;

                        }
                }
        }

        return max[type];
}

/* Sets the callback to draw */
int picviz_line_draw(PicvizImage *image, PicvizLine *line, void (*draw_line_func)(PicvizImage *image, PcvID axis_id, PicvizLine *line, PicvizAxisPlot *axisplot1, PicvizAxisPlot *axisplot2, PcvWidth x1, PcvHeight y1, PcvWidth x2, PcvHeight y2))
{
	PicvizAxisPlot *axisplot;
	PicvizAxisPlot *last_axisplot;
	PcvWidth  last_x;
	PcvHeight last_y;

	llist_for_each_entry(axisplot, &line->axisplot, list) {
		PicvizAxis *axis = (PicvizAxis *)picviz_axis_get(image, axisplot->axis_id);

		if (axis->id == 0) {
			last_y = image->height - axisplot->y;
			last_x = axis->xpos;
			last_axisplot = axisplot;
		} else {
			draw_line_func(image, axis->id, line, last_axisplot, axisplot, last_x, last_y, axis->xpos, image->height - axisplot->y);
			last_x = axis->xpos;
			last_y = image->height - axisplot->y;
			last_axisplot = axisplot;
		}
	}

	return 0;
}

PicvizLine *picviz_line_id_get(PicvizImage *image, PcvID line_id)
{
	PicvizLine *line;

	llist_for_each_entry(line, &image->lines, list) {
		if (line->id == line_id) return line;
	}

	return NULL;
}

void picviz_line_debug(PicvizLine *line)
{
	fprintf(stdout, "line->id=%llu\n", line->id);
	fprintf(stdout, "line->hidden=%d\n", line->hidden);
}

