#!/bin/bash

function explain_usage()
{
  echo "This script configures the \"commandergenius\" SDL Android port"
  echo "for the build of openMSX."
  echo ""
  echo "The script depends on the Android SDK and NDK and on the"
  echo "\"commandergenius\" SDL Android port."
  echo ""
  echo "First download and install the Android SDK and NDK"
  echo "as per the instructions on the Android website."
  echo ""
  echo "Please make sure to add the Android SDK and NDK tools locations"
  echo "to your PATH variable, as per the instructions on the Android"
  echo "website. Otherwise the setup will fail."
  echo ""
  echo "Subsequently download the \"commandergenius\" SDL Android port"
  echo "into your favorite location."
  echo ""
  echo "Example:"
  echo "> cd /opt"
  echo "> git clone https://github.com/pelya/commandergenius.git"
  echo ""
  echo "Once that is done, you can launch this script."
  echo "You must specify the path to the \"commandergenius\" SDL Android port"
  echo "on the command line or in environment parameter SDL_ANDROID_PORT_PATH."
  echo ""
  echo "Example 1:"
  echo "> export SDL_ANDROID_PORT_PATH=/opt/commandergenius"
  echo "> ./setup_anddev.sh"
  echo ""
  echo "Example 2:"
  echo "> ./setup_anddev.sh /opt/commandergenius"
  echo ""
}

if [ $# -eq 0 -a -z "$SDL_ANDROID_PORT_PATH" ]; then
  explain_usage
  exit 1
fi

if [ $# -eq 1 ]; then
    sdl_android_port_path="$1"
else
    sdl_android_port_path="$SDL_ANDROID_PORT_PATH"
fi

if [ ! -d "${sdl_android_port_path}" ]; then
  echo "Can not find directory ${sdl_android_port_path}"
  exit 1
fi
sdl_port_app_dir="${sdl_android_port_path}/project/jni/application"
if [ ! -d "${sdl_port_app_dir}" ]; then
    echo "Can not find expected sub-directory project/jni/application"
    echo "in specified directory ${sdl_android_port_path}"
    exit 1
fi

this_script_dir=$(dirname $0)
this_script_dir=$(cd ${this_script_dir}; pwd)
my_home_dir=${this_script_dir%/*}
my_home_dir=${my_home_dir%/*}
my_android_dir="${my_home_dir}/build/android/openmsx"
my_relative_dir="${my_android_dir##*/}"

tcl_build=8.5.11
tcl_version=${tcl_build%.*}
tcl_archive="${my_home_dir}/derived/3rdparty/download/tcl${tcl_build}-src.tar.gz"
tcl_sdl_dir="${sdl_android_port_path}/project/jni/tcl${tcl_version}"
if [ -d "${tcl_sdl_dir}" ]; then
	if [ ! -f "${tcl_sdl_dir}/${tcl_build}.txt" ]; then
		echo "ERROR: expecting TCL build ${tcl_build} in ${tcl_sdl_dir}"
		exit 1
	fi
fi
if [ ! -d "${tcl_sdl_dir}" ]; then
    if [ ! -f "${tcl_archive}" ]; then
		cd "${my_home_dir}"
		echo "Downloading TCL"
		python build/android_download.py
		if [ $? -ne 0 ]; then
			echo "ERROR: Failed to download TCL"
			exit 1
		fi
		if [ ! -f "${tcl_archive}" ]; then
			echo "ERROR: Failed to download TCL ${tcl_build}"
			exit 1
		fi
    fi
    cd "${sdl_android_port_path}/project/jni"
    tar xzf "${tcl_archive}"
	tcl_relative_dir=tcl${tcl_version}
    mv tcl${tcl_build} ${tcl_relative_dir}
	touch ${tcl_relative_dir}/${tcl_build}.txt
    cp -p "${my_home_dir}/build/android/tcl${tcl_build}_Android.mk" ${tcl_relative_dir}/Android.mk
    cd ${tcl_relative_dir}
    mkdir -p lib/armeabi
    mkdir -p lib/armeabi-v7a
    mkdir -p include
    cp -p generic/*.h include
    cd unix
    if [ ! -f Makefile.in.original ]; then
        cp -p Makefile.in Makefile.in.original
        patch -i "${my_home_dir}/build/android/tcl${tcl_build}_unix_Makefile.in.patch"
    fi
    export BUILD_EXECUTABLE=yes
    ../../setCrossEnvironment.sh ./configure --host=arm-eabi
    make clean
    ../../setCrossEnvironment.sh make
    mv libtcl8.5.so ../lib/armeabi
    ../../setCrossEnvironment.sh ./configure --host=arm-eabi-v7a
    make clean
    ../../setCrossEnvironment.sh make
    mv libtcl8.5.so ../lib/armeabi-v7a
fi

echo "Setting-up softlink to this application in the SDL android port."
if [ -h "${sdl_port_app_dir}/${my_relative_dir}" ]; then
    rm "${sdl_port_app_dir}/${my_relative_dir}"
fi
if [ -d "${sdl_port_app_dir}/${my_relative_dir}" ]; then
    echo "ERROR: found directory ${my_relative_dir} in ${sdl_port_app_dir}"
    echo "This seems to be another app with the same name."
    echo "Please resolve the conflict and then re-run this script."
    exit 1
fi
ln -s "${my_android_dir}" "${sdl_port_app_dir}/${my_relative_dir}"
rm -f "${sdl_port_app_dir}/src"
ln -s "${my_relative_dir}" "${sdl_port_app_dir}/src"

echo "Making environment.props file for the build script"
cat > "${my_android_dir}/environment.props" << @EOT
# Do not edit this file. It is generated by setup_anddev.sh
sdl_android_port_path="${sdl_android_port_path}"
my_home_dir="${my_home_dir}"
@EOT

cd "${my_android_dir}"
./generate_AndroidAppSettings.sh
if [ $? -ne 0 ]; then
	exit 1
fi

echo "Configuring SDL android port to build this application"
cd "${sdl_android_port_path}"
./changeAppSettings.sh -a
if [ $? -ne 0 ]; then
    echo "ERROR: an unexpected problem occurred while running changeAppSettings.sh -a"
    echo "       in ${sdl_android_port_path}"
    exit 1
fi

android update project -p project -t android-19
if [ $? -ne 0 ]; then
    echo "ERROR: an unexpected problem occurred while running \"android update ...\""
    exit 1
fi

echo ""
echo "You can now build the application from the openMSX android build directory"
echo ""
echo "Example"
echo "> cd ${my_android_dir}"
echo "> ./launch_anddev_build.sh"
echo ""
echo "Note: the first time that you run the launch_anddev_build.sh script, you may get some error"
echo "      message. It is due to a subtle bug in the SDL android port."
echo "      Simply re-run the build.sh script a second time and it will work fine."
