open Stdune

let attr_of_ansi_color_rgb8 (c : Ansi_color.RGB8.t) =
  let module A = Notty.A in
  match Ansi_color.RGB8.to_int c with
  | 0 -> A.black
  | 1 -> A.red
  | 2 -> A.green
  | 3 -> A.yellow
  | 4 -> A.blue
  | 5 -> A.magenta
  | 6 -> A.cyan
  | 7 -> A.white
  | 8 -> A.lightblack
  | 9 -> A.lightred
  | 10 -> A.lightgreen
  | 11 -> A.lightyellow
  | 12 -> A.lightblue
  | 13 -> A.lightmagenta
  | 14 -> A.lightcyan
  | 15 -> A.lightwhite
  | i when i <= 231 ->
    let i = i - 16 in
    let r = i / 36 in
    let g = i / 6 mod 6 in
    let b = i mod 6 in
    A.rgb ~r ~g ~b
  | i when i <= 255 -> A.gray (i - 232)
  | i -> Code_error.raise "invalid 8-bit color" [ ("value", Dyn.int i) ]

let attr_of_ansi_color_rgb24 (c : Ansi_color.RGB24.t) =
  let module A = Notty.A in
  A.rgb ~r:(Ansi_color.RGB24.red c) ~g:(Ansi_color.RGB24.green c)
    ~b:(Ansi_color.RGB24.blue c)

let attr_of_ansi_color_style (s : Ansi_color.Style.t) =
  let module A = Notty.A in
  match s with
  | `Fg_black -> A.(fg black)
  | `Fg_red -> A.(fg red)
  | `Fg_green -> A.(fg green)
  | `Fg_yellow -> A.(fg yellow)
  | `Fg_blue -> A.(fg blue)
  | `Fg_magenta -> A.(fg magenta)
  | `Fg_cyan -> A.(fg cyan)
  | `Fg_white -> A.(fg white)
  | `Fg_default -> A.empty
  | `Fg_bright_black -> A.(fg lightblack)
  | `Fg_bright_red -> A.(fg lightred)
  | `Fg_bright_green -> A.(fg lightgreen)
  | `Fg_bright_yellow -> A.(fg lightyellow)
  | `Fg_bright_blue -> A.(fg lightblue)
  | `Fg_bright_magenta -> A.(fg lightmagenta)
  | `Fg_bright_cyan -> A.(fg lightcyan)
  | `Fg_bright_white -> A.(fg lightwhite)
  | `Fg_8_bit_color c -> A.fg (attr_of_ansi_color_rgb8 c)
  | `Fg_24_bit_color c -> A.fg (attr_of_ansi_color_rgb24 c)
  | `Bg_black -> A.(bg black)
  | `Bg_red -> A.(bg red)
  | `Bg_green -> A.(bg green)
  | `Bg_yellow -> A.(bg yellow)
  | `Bg_blue -> A.(bg blue)
  | `Bg_magenta -> A.(bg magenta)
  | `Bg_cyan -> A.(bg cyan)
  | `Bg_white -> A.(bg white)
  | `Bg_default -> A.empty
  | `Bg_bright_black -> A.(bg lightblack)
  | `Bg_bright_red -> A.(bg lightred)
  | `Bg_bright_green -> A.(bg lightgreen)
  | `Bg_bright_yellow -> A.(bg lightyellow)
  | `Bg_bright_blue -> A.(bg lightblue)
  | `Bg_bright_magenta -> A.(bg lightmagenta)
  | `Bg_bright_cyan -> A.(bg lightcyan)
  | `Bg_bright_white -> A.(bg lightwhite)
  | `Bg_8_bit_color c -> A.bg (attr_of_ansi_color_rgb8 c)
  | `Bg_24_bit_color c -> A.bg (attr_of_ansi_color_rgb24 c)
  | `Bold -> A.(st bold)
  | `Italic -> A.(st italic)
  | `Dim -> A.(st dim)
  | `Underline -> A.(st underline)

let attr_of_user_message_style fmt t (pp : User_message.Style.t Pp.t) : unit =
  let attr =
    let module A = Notty.A in
    match (t : User_message.Style.t) with
    | Loc -> A.(st bold)
    | Error -> A.(st bold ++ fg red)
    | Warning -> A.(st bold ++ fg magenta)
    | Kwd -> A.(st bold ++ fg blue)
    | Id -> A.(st bold ++ fg yellow)
    | Prompt -> A.(st bold ++ fg green)
    | Hint -> A.(st italic ++ fg white)
    | Details -> A.(st dim ++ fg white)
    | Ok -> A.(st italic ++ fg green)
    | Debug -> A.(st underline ++ fg lightcyan)
    | Success -> A.(st bold ++ fg green)
    | Ansi_styles l ->
      List.fold_left ~init:A.empty l ~f:(fun attr s ->
          A.(attr ++ attr_of_ansi_color_style s))
  in
  Notty.I.pp_attr attr Pp.to_fmt fmt pp

let image_of_user_message_style_pp =
  Notty.I.strf "%a@."
    (Pp.to_fmt_with_tags ~tag_handler:attr_of_user_message_style)

module Tui = struct
  module Term = Notty_unix.Term

  let create () = Term.create ~nosig:false ~output:Unix.stderr ()

  let bytes = Bytes.make 64 '0'

  let sigcont_pipe = lazy (Unix.pipe ~cloexec:true ())

  let term =
    let setup =
      lazy
        (Unix.set_nonblock (Lazy.force sigcont_pipe |> fst);
         let term = ref (create ()) in
         Sys.set_signal Sys.sigcont
         @@ Sys.Signal_handle
              (fun _ ->
                Term.release !term;
                term := create ();
                assert (
                  1
                  = Unix.single_write (Lazy.force sigcont_pipe |> snd) bytes 0 1));
         let rec old =
           lazy
             (Sys.signal Sys.sigtstp
             @@ Sys.Signal_handle
                  (fun i ->
                    Term.release !term;
                    match Lazy.force old with
                    | Sys.Signal_handle f -> f i
                    | _ -> Unix.kill (Unix.getpid ()) Sys.sigstop))
         in
         ignore (Lazy.force old);
         term)
    in
    fun () -> !(Lazy.force setup)

  let start () = Unix.set_nonblock Unix.stdin

  let image ~status_line ~messages =
    let status =
      match (status_line : User_message.Style.t Pp.t option) with
      | None -> []
      | Some message -> [ image_of_user_message_style_pp message ]
    in
    let messages =
      List.map messages ~f:(fun msg ->
          image_of_user_message_style_pp (User_message.pp msg))
    in
    Notty.I.vcat (messages @ status)

  let render (state : Dune_threaded_console.state) =
    let messages = Queue.to_list state.messages in
    let image = image ~status_line:state.status_line ~messages in
    Term.image (term ()) image

  let resize mutex (state : Dune_threaded_console.state) =
    Mutex.lock mutex;
    state.dirty <- true;
    Mutex.unlock mutex;
    Unix.gettimeofday ()

  let rec handle_user_events ~now ~time_budget mutex state =
    (* We check for any user input and handle it. If we go over the
       [time_budget] we give up and continue. *)
    let input_fds =
      let sigcont_r, _ = Lazy.force sigcont_pipe in
      match Unix.select [ Unix.stdin; sigcont_r ] [] [] time_budget with
      | exception Unix.Unix_error (EINTR, _, _) -> `Restore
      | [], _, _ -> `Timeout
      | fds, _, _ -> (
        match List.exists fds ~f:(Poly.equal sigcont_r) with
        | false -> `Event
        | true ->
          ignore (Unix.read sigcont_r bytes 0 1);
          (* backgrounding could have changed the cursor settings for example.
             we need to restore all this stuff *)
          `Restore)
    in
    match input_fds with
    | `Restore -> resize mutex state
    | `Timeout ->
      now +. time_budget
      (* Nothing to do, we return the time at the end of the time budget. *)
    | `Event -> (
      (* TODO if anything fancy is done in the UI in the future we need to lock
         the state with the provided mutex *)
      match Term.event (term ()) with
      | `Key (`ASCII 'q', _) ->
        (* When we encounter q we make sure to quit by signaling termination. *)
        Unix.kill (Unix.getpid ()) Sys.sigterm;
        Unix.gettimeofday ()
      | `Resize _ -> resize mutex state
      | _ -> Unix.gettimeofday ()
      | exception Unix.Unix_error ((EAGAIN | EWOULDBLOCK), _, _) ->
        (* If we encounter an exception, we make sure to rehandle user events
           with a corrected time budget. *)
        let old_now = now in
        let now = Unix.gettimeofday () in
        let delta_now = now -. old_now in
        let time_budget = Float.max 0. (time_budget -. delta_now) in
        handle_user_events ~now ~time_budget mutex state)

  let reset () = ()

  let reset_flush_history () = ()

  let finish () =
    Term.release (term ());
    Unix.clear_nonblock Unix.stdin
end

let backend =
  let t = lazy (Dune_threaded_console.make (module Tui)) in
  fun () -> Lazy.force t
