# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re
import urllib
import urlparse
import random
import logging

import gtk
import cairo

import ninix.seriko
import ninix.pix


class Surface(object):

    # keyval/name mapping
    from ninix.keymap import keymap_old, keymap_new

    def __init__(self):
        self.window = []
        self.desc = None
        self.request_parent = lambda *a: None # dummy
        self.mikire = 0
        self.kasanari = 0

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'stick_window': self.window_stick,
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def finalize(self):
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []

    def create_gtk_window(self, title, skip_taskbar):
        window = ninix.pix.TransparentWindow()
        window.set_focus_on_map(False)
        window.set_title(title)
        window.set_decorated(False)
        window.set_resizable(False)
        if skip_taskbar:
            window.set_skip_taskbar_hint(True)
        window.connect('delete_event', self.delete)
        window.connect('key_press_event', self.key_press)
        window.connect('window_state_event', self.window_state)
        window.set_events(gtk.gdk.KEY_PRESS_MASK)
        window.realize()
        return window

    def identify_window(self, win):
        for surface_window in self.window:
            if win == surface_window.window.window:
                return True
        return False

    def window_stayontop(self, flag):
        for surface_window in self.window:
            gtk_window = surface_window.window
            gtk_window.set_keep_above(flag)
                
    def window_iconify(self, flag):
        gtk_window = self.window[0].window
        iconified = gtk_window.window.get_state() & \
            gtk.gdk.WINDOW_STATE_ICONIFIED
        if flag and not iconified:
            gtk_window.iconify()
        elif not flag and iconified:
            gtk_window.deiconify()

    def window_state(self, window, event):
        if not self.request_parent('GET', 'is_running'):
            return
        if not (event.changed_mask & gtk.gdk.WINDOW_STATE_ICONIFIED):
            return
        if event.new_window_state & gtk.gdk.WINDOW_STATE_ICONIFIED:
            if window == self.window[0].window:
                self.request_parent('NOTIFY', 'notify_iconified')
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   not gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.iconify()
        else:
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.deiconify()
            if window == self.window[0].window:
                self.request_parent('NOTIFY', 'notify_deiconified')
        return

    def delete(self, window, event):
        return True

    def key_press(self, window, event):
        name = self.keymap_old.get(event.keyval, event.string)
        keycode = self.keymap_new.get(event.keyval, event.string)
        if event.state & (gtk.gdk.CONTROL_MASK | gtk.gdk.SHIFT_MASK):
            if name == 'f12':
                logging.info('reset surface position')
                self.reset_position()
        if name or keycode:
            self.request_parent(
                'NOTIFY', 'notify_event', 'OnKeyPress', name, keycode)
        return True

    def window_stick(self, stick):
        for window in self.window:
            if stick:
                window.window.stick()
            else:
                window.window.unstick()

    re_surface_id = re.compile('^surface([0-9]+)$')

    def get_seriko(self, surface):
        seriko = {}
        for basename, (path, config) in surface.items():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            seriko[key] = ninix.seriko.get_actors(config)
        return seriko

    def new(self, desc, alias, surface, name, prefix, tooltips):
        self.desc = desc
        self.__tooltips = tooltips
        self.name = name
        self.prefix = prefix
        if alias is None:
            alias0 = alias1 = None
        else:
            alias0 = alias.get('sakura.surface.alias')
            alias1 = alias.get('kero.surface.alias')
        # load surface
        pixbufs = {}
        elements = {}
        for basename, (path, config) in surface.items():
            if path is None:
                continue
            if not os.path.exists(path):
                name, ext = os.path.splitext(path)
                dgp_path = ''.join((name, '.dgp'))
                if not os.path.exists(dgp_path):
                    ddp_path = ''.join((name, '.ddp'))
                    if not os.path.exists(ddp_path):
                        logging.error(
                            '{0}: file not found (ignored)'.format(path))
                        continue
                    else:
                        path = ddp_path
                else:
                    path = dgp_path
            elements[basename] = [path]
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            pixbufs[key] = elements[basename]
        # compose surface elements
        composite_pixbuf = {}
        for basename, (path, config) in surface.items():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            if 'element0' in config:
                logging.debug('surface {0}'.format(key))
                composite_pixbuf[key] = self.compose_elements(elements, config)
        pixbufs.update(composite_pixbuf)
        # check if necessary surfaces have been loaded
        for key in ['0', '10']:
            if key not in pixbufs:
                raise SystemExit, 'cannot load surface #{0} (abort)\n'.format(key)
        self.__pixbufs = pixbufs
        # arrange surface configurations
        region = {}
        for basename, (path, config) in surface.items():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define collision areas
            buf = []
            for n in range(256):
                # "redo" syntax
                rect = config.get(''.join(('collision', str(n))))
                if rect is None:
                    continue
                values = rect.split(',')
                if len(values) != 5:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in values[:4]]
                except ValueError:
                    continue
                buf.append((values[4].strip(), x1, y1, x2, y2))
            for part in ['head', 'face', 'bust']:
                # "inverse" syntax
                rect = config.get(''.join(('collision.', part)))
                if not rect:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in rect.split(',')]
                except ValueError:
                    pass
                buf.append((part.capitalize(), x1, y1, x2, y2))
            region[key] = buf
        self.__region = region
        # MAYUNA
        mayuna = {}
        for basename, (path, config) in surface.items():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            mayuna[key] = ninix.seriko.get_mayuna(config)
        bind = {}
        for side in ['sakura', 'kero']:
            bind[side] = {}
            for index in range(128):
                name = self.desc.get(
                    '{0}.bindgroup{1:d}.name'.format(side, index), None)
                default = self.desc.get(
                    '{0}.bindgroup{1:d}.default'.format(side, index), 0)
                if name is not None:
                    bind[side][index] = [name, default]
        self.mayuna = {}
        for side in ['sakura', 'kero']:
            self.mayuna[side] = []
            for index in range(128):
                key = self.desc.get('{0}.menuitem{1:d}'.format(side, index), None)
                if key == '-':
                    self.mayuna[side].append([key, None, 0])
                else:
                    try:
                        key = int(key)
                    except:
                        pass
                    else:
                        if key in bind[side]:
                            name = bind[side][key][0].split(',')
                            self.mayuna[side].append([key, name[1],
                                                      bind[side][key][1]])
        # create surface windows
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        self.__surface = surface
        self.add_window(0, '0', alias0, mayuna, bind['sakura'])
        self.add_window(1, '10', alias1, mayuna, bind['kero'])

    def get_menu_pixmap(self):
        top_dir = self.prefix
        name = self.desc.get('menu.background.bitmap.filename',
                             'menu_background.png')
        name = name.replace('\\', '/')
        path_background = os.path.join(top_dir, name)
        name = self.desc.get('menu.sidebar.bitmap.filename',
                             'menu_sidebar.png')
        name = name.replace('\\', '/')
        path_sidebar = os.path.join(top_dir, name)
        name = self.desc.get('menu.foreground.bitmap.filename',
                             'menu_foreground.png')
        name = name.replace('\\', '/')
        path_foreground = os.path.join(top_dir, name)
        return path_background, path_sidebar, path_foreground

    def get_menu_fontcolor(self):
        fontcolor_r = self.desc.get_with_type('menu.background.font.color.r', int, 0)
        fontcolor_g = self.desc.get_with_type('menu.background.font.color.g', int, 0)
        fontcolor_b = self.desc.get_with_type('menu.background.font.color.b', int, 0)
        background = (fontcolor_r, fontcolor_g, fontcolor_b)
        fontcolor_r = self.desc.get_with_type('menu.foreground.font.color.r', int, 0)
        fontcolor_g = self.desc.get_with_type('menu.foreground.font.color.g', int, 0)
        fontcolor_b = self.desc.get_with_type('menu.foreground.font.color.b', int, 0)
        foreground = (fontcolor_r, fontcolor_g, fontcolor_b)
        return background, foreground

    def add_window(self, side, default, alias=None, mayuna={}, bind={}):
        assert len(self.window) == side
        if side == 0:
            name = 'sakura'
            title = self.request_parent('GET', 'get_selfname') or \
                ''.join(('surface.', name))
        elif side == 1:
            name = 'kero'
            title = self.request_parent('GET', 'get_keroname') or \
                ''.join(('surface.', name))
        else:
            name = 'char{0:d}'.format(side)
            title = ''.join(('surface.', name))
        skip_taskbar = bool(side >= 1)
        gtk_window = self.create_gtk_window(title, skip_taskbar)
        seriko = self.get_seriko(self.__surface)
        tooltips = {}
        if name in self.__tooltips:
            tooltips = self.__tooltips[name]
        surface_window = SurfaceWindow(
            gtk_window, side, self.desc, alias, self.__surface, tooltips,
            self.__pixbufs, seriko, self.__region, mayuna, bind, default)
        surface_window.set_responsible(self.handle_request)
        self.window.append(surface_window)

    def get_mayuna_menu(self):
        for side, index in [('sakura', 0), ('kero', 1)]:
            for menu in self.mayuna[side]:
                if menu[0] != '-':
                    menu[2] = self.window[index].bind[menu[0]][1]
        return self.mayuna

    def compose_elements(self, elements, config):
        error = None
        for n in range(256):
            key = ''.join(('element', str(n)))
            if key not in config:
                break
            spec = [value.strip() for value in config[key].split(',')]
            try:
                method, filename, x, y = spec
                x = int(x)
                y = int(y)
            except ValueError:
                error = 'invalid element spec for {0}: {1}'.format(key, config[key])
                break
            basename, ext = os.path.splitext(filename)
            ext = ext.lower()
            if ext not in ['.png', '.dgp', '.ddp']:
                error = 'unsupported file format for {0}: {1}'.format(key, filename)
                break
            basename = basename.lower()
            if basename not in elements:
                error = '{0} file not found: {1}'.format(key, filename)
                break
            pixbuf = elements[basename][0]
            if n == 0: # base surface
                pixbuf_list = [pixbuf]
            elif method == 'overlay':
                pixbuf_list.append((pixbuf, x, y))
            elif method == 'overlayfast': # XXX
                pixbuf_list.append((pixbuf, x, y))
            elif method == 'base':
                pixbuf_list.append((pixbuf, x, y))
            else:
                error = 'unknown method for {0}: {1}'.format(key, method)
                break
            logging.debug('{0}: {1} {2}, x={3:d}, y={4:d}'.format(key, method, filename, x, y))
        if error is not None:
            logging.error(error)
            pixbuf_list = []
        return pixbuf_list

    def get_window(self, side):
        if len(self.window) > side:
            return self.window[side].window # FIXME
        else: 
            return None

    def reset_surface(self):
        for window in self.window:
            window.reset_surface()

    def set_surface_default(self, side):
        if side is None:
            for side in range(len(self.window)):
                self.window[side].set_surface_default()
        elif 0 <= side < len(self.window):
            self.window[side].set_surface_default()

    def set_surface(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].set_surface(surface_id)

    def get_surface(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface()
        else:
            return 0

    def get_surface_size(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_size()
        else:
            return 0, 0

    def get_surface_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_offset()
        else:
            return 0, 0

    def get_touched_region(self, side, x, y):
        if len(self.window) > side:
            return self.window[side].get_touched_region(x, y)
        else:
            return ''

    def get_center(self, side):
        if len(self.window) > side:
                return self.window[side].get_center()
        else:
            return None, None

    def get_kinoko_center(self, side):
        if len(self.window) > side:
                return self.window[side].get_kinoko_center()
        else:
            return None, None

    def reset_balloon_position(self):
        for side in range(len(self.window)):
            x, y = self.get_position(side)
            direction = self.window[side].direction
            ox, oy = self.get_balloon_offset(side)
            self.request_parent(
                'NOTIFY', 'set_balloon_direction', side, direction)
            if direction == 0: # left
                base_x = x + ox
            else:
                w, h = self.get_surface_size(side)
                base_x = x + w - ox
            base_y = y + oy
            self.request_parent(
                'NOTIFY', 'set_balloon_position', side, base_x, base_y)

    def reset_position(self):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        for side in range(len(self.window)):
            align = self.get_alignment(side)
            w, h = self.get_surface_size(side)
            if side == 0: # sakura
                x = left + scrn_w - w
            else:
                b0w, b0h = self.request_parent(
                    'GET', 'get_balloon_size', side - 1)
                b1w, b1h = self.request_parent(
                    'GET', 'get_balloon_size', side)
                bpx, bpy = self.request_parent(
                    'GET', 'get_balloon_windowposition', side)
                o0x, o0y = self.get_balloon_offset(side - 1)
                o1x, o1y = self.get_balloon_offset(side)
                offset = max(0, b1w - (b0w - o0x))
                if (s0x + o0x - b0w) - offset - w + o1x < left:
                    x = left
                else:
                    x = (s0x + o0x - b0w) - offset - w + o1x
            if align == 1: # top
                y = top
            else:
                y = top + scrn_h - h
            self.set_position(side, x, y)
            s0x, s0y, s0w, s0h = x, y, w, h # for next loop

    def set_position(self, side, x, y):
        if len(self.window) > side:
            self.window[side].set_position(x, y)

    def get_position(self, side):
        if len(self.window) > side:
            return self.window[side].get_position()
        else:
            return 0, 0

    def set_alignment_current(self):
        for side in range(len(self.window)):
            self.window[side].set_alignment_current()

    def set_alignment(self, side, align):
        if len(self.window) > side:
            self.window[side].set_alignment(align)

    def get_alignment(self, side):
        if len(self.window) > side:
            return self.window[side].get_alignment()
        else:
            return 0

    def reset_alignment(self):
        if self.desc.get('seriko.alignmenttodesktop') == 'free':
            align = 2
        else:
            align = 0
        for side in range(len(self.window)):
            self.set_alignment(side, align)

    def is_shown(self, side):
        if len(self.window) > side:
            return self.window[side].is_shown()
        else:
            return False

    def show(self, side):
        if len(self.window) > side:
            self.window[side].show()

    def hide_all(self):
        for side in range(len(self.window)):
            self.window[side].hide()

    def hide(self, side):
        if len(self.window) > side:
            self.window[side].hide()

    def raise_all(self):
        for side in range(len(self.window)):
            self.window[side].raise_()

    def raise_(self, side):
        if len(self.window) > side:
            self.window[side].raise_()

    def lower_all(self):
        for side in range(len(self.window)):
            self.window[side].lower()

    def lower(self, side):
        if len(self.window) > side:
            self.window[side].lower()

    def invoke(self, side, actor_id):
        if len(self.window) > side:
            self.window[side].invoke(actor_id)

    def invoke_yen_e(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].invoke_yen_e(surface_id)

    def invoke_talk(self, side, surface_id, count):
        if len(self.window) > side:
            return self.window[side].invoke_talk(surface_id, count)
        else:
            return 0

    def set_icon(self, path):
        pixbuf = None
        if path is not None:
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(path, is_pnr=False)
            except:
                pixbuf = None
        for window in self.window:
            window.window.set_icon(pixbuf)

    def get_mikire(self): ## FIXME
        return self.mikire

    def get_kasanari(self): ## FIXME
        return self.kasanari

    def get_name(self):
        return self.name

    def get_username(self):
        return None if self.desc is None else self.desc.get('user.defaultname')

    def get_selfname(self):
        return None if self.desc is None else self.desc.get('sakura.name')

    def get_selfname2(self):
        return None if self.desc is None else self.desc.get('sakura.name2')

    def get_keroname(self):
        return None if self.desc is None else self.desc.get('kero.name')

    def get_friendname(self):
        return None if self.desc is None else self.desc.get('sakura.friend.name')

    def get_balloon_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_balloon_offset()
        return 0, 0

    def toggle_bind(self, args):
        side, bind_id = args
        self.window[side].toggle_bind(bind_id)

    def get_collision_area(self, side, part):
        if len(self.window) > side:
            return self.window[side].get_collision_area(part)
        return None

    def check_mikire_kasanari(self):
        if not self.is_shown(0):
            self.mikire = self.kasanari = 0
            return
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        x0, y0 = self.get_position(0)
        s0w, s0h = self.get_surface_size(0)
        if x0 + s0w / 3 < left or x0 + s0w * 2 / 3 > left + scrn_w or \
           y0 + s0h / 3 < top or y0 + s0h * 2 / 3 > top + scrn_h:
            self.mikire = 1
        else:
            self.mikire = 0
        if not self.is_shown(1):
            self.kasanari = 0
            return
        x1, y1 = self.get_position(1)
        s1w, s1h = self.get_surface_size(1)
        if (x0 < x1 + s1w / 2 < x0 + s0w and y0 < y1 + s1h / 2 < y0 + s0h) or \
           (x1 < x0 + s0w / 2 < x1 + s1w and y1 < y0 + s0h / 2 < y1 + s1h):
            self.kasanari = 1
        else:
            self.kasanari = 0


class SurfaceWindow(object):

    # DnD data types
    dnd_targets = [
        ('text/plain', 0, 0),
        ]

    def __init__(self, window, side, desc, alias, surface, tooltips,
                 pixbuf, seriko, region, mayuna, bind, default_id):
        self.window = window
        self.side = side
        self.request_parent = lambda *a: None # dummy
        self.desc = desc
        self.alias = alias
        self.tooltips = tooltips
        self.align = 0
        self.__current_part = '' ## FIXME
        if self.alias is not None:
            default_id = self.alias.get(default_id, [default_id])[0]
        self.surface = surface
        self.surface_id = default_id
        self.pixbuf = pixbuf
        self.current_surface_pixbuf = None # XXX
        self.seriko = ninix.seriko.Controler(seriko)
        self.seriko.set_responsible(self.handle_request)
        self.region = region
        self.mayuna = mayuna
        self.bind = bind
        self.default_id = default_id
        self.__shown = False
        self.window_offset = (0, 0)
        self.position = (0, 0)
        self.__direction = 0
        self.dragged = False
        self.x_root = None
        self.y_root = None
        self.window.connect('leave_notify_event', self.window_leave_notify) # XXX
        self.window.connect('enter_notify_event', self.window_enter_notify) # XXX
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.BUTTON_RELEASE_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.POINTER_MOTION_HINT_MASK|
                              gtk.gdk.SCROLL_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('drag_data_received', self.drag_data_received)
        self.darea.connect('scroll_event', self.scroll)
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.window.add(self.darea)
        self.window.realize()
        self.window.window.set_back_pixmap(None, False)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    @property
    def direction(self):
        return self.__direction

    @direction.setter
    def direction(self, direction):
        self.__direction = direction # 0: left, 1: right
        self.request_parent(
            'NOTIFY', 'set_balloon_direction', self.side, direction)

    @property
    def scale(self):
        return self.request_parent('GET', 'get_preference', 'surface_scale')

    def drag_data_received(self, widget, context, x, y, data, info, time):
        logging.debug('Content-type: {0}'.format(data.type))
        logging.debug('Content-length: {0:d}'.format(data.get_length()))
        if str(data.type) == 'text/plain':
            filelist = []
            for line in data.data.split('\r\n'):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                pathname = urllib.url2pathname(path)
                if scheme == 'file' and os.path.exists(pathname):
                    filelist.append(pathname)
            if filelist:
                self.request_parent(
                    'NOTIFY', 'enqueue_event',
                    'OnFileDrop2', chr(1).join(filelist), self.side)
        return True

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def invoke_yen_e(self, surface_id):
        self.seriko.invoke_yen_e(self, surface_id)

    def invoke_talk(self, surface_id, count):
        return self.seriko.invoke_talk(self, surface_id, count)

    def reset_surface(self):
        surface_id = self.get_surface()
        self.set_surface(surface_id)

    def set_surface_default(self):
        self.set_surface(self.default_id)

    def set_surface(self, surface_id):
        if self.alias is not None and surface_id in self.alias:
            aliases = self.alias.get(surface_id)
            if aliases:
                surface_id = random.choice(aliases)
        if surface_id == '-2':
            self.seriko.terminate(self)
        if surface_id in ['-1', '-2']:
            pass
        elif surface_id not in self.pixbuf:
            self.surface_id = self.default_id
        else:
            self.surface_id = surface_id
        self.seriko.reset(self, surface_id)
        # define collision areas
        self.collisions = self.region[self.surface_id]
        # update window offset
        x, y = self.position # XXX: without window_offset
        w, h = self.get_surface_size(self.surface_id)
        dw, dh = self.get_surface_size(self.default_id) # default surface size
        xoffset = (dw - w) / 2
        if self.get_alignment() == 0:
            yoffset = dh - h
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            y = top + scrn_h - dh
        elif self.get_alignment() == 1:
            yoffset = 0
        else:
            yoffset = (dh - h) / 2
        self.window_offset = (xoffset, yoffset)
        # resize window
        self.darea.set_size_request(w, h)
        self.window.queue_resize()
        self.seriko.start(self)
        # relocate window
        if not self.dragged: # XXX
            self.set_position(x, y)
        if self.side < 2:
            self.request_parent('NOTIFY', 'notify_observer', 'set surface')

    def iter_mayuna(self, surface_width, surface_height, mayuna, done):
        for surface, interval, method, args in mayuna.patterns:
            if method in ['bind', 'add']:
                if surface in self.pixbuf:
                    x, y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    # overlay surface pixbuf
                    if x + w > surface_width:
                        w = surface_width - x
                    if y + h > surface_height:
                        h = surface_height - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'reduce':
                if surface in self.pixbuf:
                    dest_x, dest_y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    x = y = 0 # XXX
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'insert':
                index = args[0]
                for actor in self.mayuna[self.surface_id]:
                    actor_id = actor.get_id()
                    if actor_id == index:
                        if actor_id in self.bind and self.bind[actor_id][1] and \
                           actor_id not in done:
                            done.append(actor_id)
                            for result in self.iter_mayuna(surface_width, surface_height, actor, done):
                                yield result
                        else:
                            break
            else:
                raise RuntimeError, 'should not reach here'

    def create_pixbuf_from_file(self, pixbuf_id):
        assert pixbuf_id in self.pixbuf
        use_pna = self.request_parent('GET', 'get_preference', 'use_pna')
        try:
            pixbuf = ninix.pix.create_pixbuf_from_file(
                self.pixbuf[pixbuf_id][0], use_pna=use_pna)
        except:
            logging.debug('cannot load surface #{0}'.format(pixbuf_id))
            return ninix.pix.create_blank_pixbuf(100, 100)
        for element, x, y in self.pixbuf[pixbuf_id][1:]:
            try:
                overlay = ninix.pix.create_pixbuf_from_file(
                    element, use_pna=use_pna)
            except:
                continue
            w = overlay.get_width()
            h = overlay.get_height()
            sw = pixbuf.get_width()
            sh = pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            overlay.composite(pixbuf, dest_x, dest_y,
                              w, h, x, y, 1.0, 1.0,
                              gtk.gdk.INTERP_BILINEAR, 255)
        return pixbuf

    def get_pixbuf(self, pixbuf_id):
        if pixbuf_id not in self.pixbuf:
            logging.debug('cannot load pixbuf #{0}'.format(pixbuf_id))
            return ninix.pix.create_blank_pixbuf(100, 100)
        return self.create_pixbuf_from_file(pixbuf_id)

    def draw_region(self):
        cr = self.darea.window.cairo_create()
        cr.save()
        scale = self.scale
        for part, x1, y1, x2, y2 in self.collisions:
            x1 = x1 * scale / 100
            x2 = x2 * scale / 100
            y1 = y1 * scale / 100
            y2 = y2 * scale / 100
            cr.set_operator(cairo.OPERATOR_ATOP)
            cr.set_source_rgba(0.2, 0.0, 0.0, 0.4) # XXX
            cr.rectangle(x1, y1, x2 - x1, y2 - y1)
            cr.fill_preserve()
            cr.set_operator(cairo.OPERATOR_SOURCE)
            cr.set_source_rgba(0.4, 0.0, 0.0, 0.8) # XXX
            cr.stroke()
        cr.restore()
        del cr

    def create_surface_pixbuf(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id in self.mayuna and self.mayuna[surface_id]:
            surface_pixbuf = self.get_pixbuf(surface_id)
            done = []
            for actor in self.mayuna[surface_id]:
                actor_id = actor.get_id()
                if actor_id in self.bind and self.bind[actor_id][1] and \
                   actor_id not in done:
                    done.append(actor_id)
                    #surface_pixbuf = self.compose_surface(
                    #    surface_pixbuf, actor, done)
                    surface_width = surface_pixbuf.get_width()
                    surface_height = surface_pixbuf.get_height()
                    for method, pixbuf, dest_x, dest_y, w, h, x, y in self.iter_mayuna(surface_width, surface_height, actor, done):
                        if method in ['bind', 'add']:
                            pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                             w, h, x, y, 1.0, 1.0,
                                             gtk.gdk.INTERP_BILINEAR, 255)
                        elif method == 'reduce':
                            if pixbuf.get_has_alpha():
                                dest_w = surface_width
                                dest_h = surface_height
                                surface_array = surface_pixbuf.get_pixels_array()
                                array = pixbuf.get_pixels_array()
                                for i in range(h):
                                    for j in range(w):
                                        if array[i][j][3] == 0: # alpha
                                            x = j + dest_x
                                            y = i + dest_y
                                            if 0 <= x < dest_w and 0 <= y < dest_h:
                                                surface_array[y][x][3] = 0 # alpha
                        else:
                            raise RuntimeError, 'should not reach here'
        else:
            surface_pixbuf = self.get_pixbuf(surface_id)
        return surface_pixbuf

    def update_frame_buffer(self):
        if not self.request_parent('GET', 'get_preference', 'seriko_inactive'):
            surface_pixbuf = self.create_surface_pixbuf(self.seriko.base_id)
            assert surface_pixbuf is not None
            # draw overlays
            for pixbuf_id, x, y in self.seriko.iter_overlays():
                try:
                    pixbuf = self.get_pixbuf(pixbuf_id)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                except:
                    continue
                # overlay surface pixbuf
                sw = surface_pixbuf.get_width()
                sh = surface_pixbuf.get_height()
                if x + w > sw:
                    w = sw - x
                if y + h > sh:
                    h = sh - y
                if x < 0:
                    dest_x = 0
                    w += x
                else:
                    dest_x = x
                if y < 0:
                    dest_y = 0
                    h += y
                else:
                    dest_y = y
                pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                 w, h, x, y, 1.0, 1.0,
                                 gtk.gdk.INTERP_BILINEAR, 255)
        else:
            surface_pixbuf = self.create_surface_pixbuf()
        w = surface_pixbuf.get_width()
        h = surface_pixbuf.get_height()
        scale = self.scale
        w = max(8, w * scale / 100)
        h = max(8, h * scale / 100)
        surface_pixbuf = surface_pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR)
        mask_pixmap = gtk.gdk.Pixmap(None, w, h, 1)
        surface_pixbuf.render_threshold_alpha(
            mask_pixmap, 0, 0, 0, 0, w, h, 1)
        self.window.mask = mask_pixmap
        self.current_surface_pixbuf = surface_pixbuf
        self.darea.queue_draw()

    def redraw(self, darea, event):
        if self.current_surface_pixbuf is None: # XXX
            return
        cr = darea.window.cairo_create()
        cr.save()
        cr.set_operator(cairo.OPERATOR_CLEAR)
        cr.paint()
        cr.restore()
        cr.set_source_pixbuf(self.current_surface_pixbuf, 0, 0)
        alpha_channel = self.request_parent(
            'GET', 'get_preference', 'surface_alpha')
        cr.paint_with_alpha(alpha_channel)
        del cr
        if self.request_parent('GET', 'get_preference', 'check_collision'):
            self.draw_region()

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def add_overlay(self, actor, pixbuf_id, x, y):
        self.seriko.add_overlay(self, actor, pixbuf_id, x, y)

    def move_surface(self, xoffset, yoffset):
        x, y = self.get_position()
        self.window.resize_move(x, y, xoffset, yoffset)
        if self.side < 2:
            args = (self.side, xoffset, yoffset)
            self.request_parent(
                'NOTIFY', 'notify_observer', 'move surface', args) # animation

    def get_balloon_offset(self):
        path, config = self.surface[''.join(('surface', self.surface_id))]
        side = self.side
        if side == 0:
            name = 'sakura'
            x = config.get_with_type('{0}.balloon.offsetx'.format(name), int)
            y = config.get_with_type('{0}.balloon.offsety'.format(name), int)
        elif side == 1:
            name = 'kero'
            x = config.get_with_type('{0}.balloon.offsetx'.format(name), int)
            y = config.get_with_type('{0}.balloon.offsety'.format(name), int)
        else:
            name = 'char{0:d}'.format(side)
            x, y = None, None # XXX
        if x is None:
            x = self.desc.get_with_type('{0}.balloon.offsetx'.format(name), int, 0)
        if y is None:
            y = self.desc.get_with_type('{0}.balloon.offsety'.format(name), int, 0)
        scale = self.scale
        x = x * scale / 100
        y = y * scale / 100
        return x, y

    def get_collision_area(self, part):
        for p, x1, y1, x2, y2 in self.collisions: ## FIXME
            if p == part:
                scale = self.scale
                x1 = x1 * scale / 100
                x2 = x2 * scale / 100
                y1 = y1 * scale / 100
                y2 = y2 * scale / 100
                return x1, y1, x2, y2
        return None

    def get_surface(self):
        return self.surface_id

    def get_surface_size(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id not in self.pixbuf:
            w, h = 100, 100 # XXX
        else:
            w, h = ninix.pix.get_png_size(self.pixbuf[surface_id][0])
        scale = self.scale
        w = max(8, int(w * scale / 100))
        h = max(8, int(h * scale / 100))
        return w, h

    def get_surface_offset(self):
        return self.window_offset

    def get_touched_region(self, x, y):
        for part, x1, y1, x2, y2 in self.collisions:
            if x1 <= x <= x2 and y1 <= y <= y2:
                ##logging.debug('{0} touched'.format(part))
                return part
        return ''

    def __get_with_scaling(self, name, conv):
        basename = ''.join(('surface', self.surface_id))
        path, config = self.surface[basename]
        value = config.get_with_type(name, conv)
        if value is not None:
            scale = self.scale
            value = conv(value * scale / 100)
        return value

    def get_center(self):
        centerx = self.__get_with_scaling('point.centerx', int)
        centery = self.__get_with_scaling('point.centery', int)
        return centerx, centery

    def get_kinoko_center(self):
        centerx = self.__get_with_scaling('point.kinoko.centerx', int)
        centery = self.__get_with_scaling('point.kinoko.centery', int)
        return centerx, centery

    def set_position(self, x, y):
        self.position = (x, y)
        new_x, new_y = self.get_position()
        if self.__shown:
            self.window.resize_move(new_x, new_y)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        direction = 0 if x > left + scrn_w / 2 else 1
        self.direction = direction
        ox, oy = self.get_balloon_offset()
        if direction == 0: # left
            base_x = new_x + ox
        else:
            w, h = self.get_surface_size()
            base_x = new_x + w - ox
        base_y = new_y + oy
        self.request_parent(
            'NOTIFY', 'set_balloon_position', self.side, base_x, base_y)
        self.request_parent('NOTIFY', 'notify_observer', 'set position')
        self.request_parent('NOTIFY', 'check_mikire_kasanari')

    def get_position(self): ## FIXME: position with offset(property)
        return list(map(lambda x, y: x + y, self.position, self.window_offset))

    def set_alignment_current(self):
        self.set_alignment(self.get_alignment())

    def set_alignment(self, align):
        if align in [0, 1, 2]:
            self.align = align
        if self.dragged:
            # XXX: position will be reset after button release event
            return
        if align == 0:
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            sw, sh = self.get_surface_size(self.default_id)
            sx, sy = self.position # XXX: without window_offset
            sy = top + scrn_h - sh
            self.set_position(sx, sy)
        elif align == 1:
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            sx, sy = self.position # XXX: without window_offset
            sy = top
            self.set_position(sx, sy)
        else: # free
            pass

    def get_alignment(self):
        return self.align

    def destroy(self):
        self.seriko.destroy()
        self.window.remove(self.darea)
        self.darea.destroy()
        self.window.destroy()

    def is_shown(self):
        return self.__shown

    def show(self):
        if self.__shown:
            return
        self.__shown = True
        x, y = self.get_position()
        self.window.resize_move(x, y) # XXX: call before showing the window
        self.darea.show()
        self.window.show()
        self.request_parent('NOTIFY', 'notify_observer', 'show', (self.side))
        self.request_parent('NOTIFY', 'notify_observer', 'raise', (self.side))

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = False
            self.request_parent(
                'NOTIFY', 'notify_observer', 'hide', (self.side))

    def raise_(self):
        self.window.window.raise_()
        self.request_parent('NOTIFY', 'notify_observer', 'raise', (self.side))

    def lower(self):
        self.window.window.lower()
        self.request_parent('NOTIFY', 'notify_observer', 'lower', (self.side))

    def button_press(self, window, event):
        self.request_parent('NOTIFY', 'reset_idle_time')
        x = int(event.x)
        y = int(event.y)
        scale = self.scale
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        self.x_root = event.x_root
        self.y_root = event.y_root
        click = 1 if event.type == gtk.gdk.BUTTON_PRESS else 2
        # automagical raise
        self.request_parent('NOTIFY', 'notify_observer', 'raise', (self.side))
        self.request_parent('NOTIFY', 'notify_surface_click',
                            event.button, click, self.side, x, y)
        return True

    def button_release(self, window, event):
        if self.dragged:
            self.dragged = False
            self.set_alignment_current()
        self.x_root = None
        self.y_root = None
        return True

    def motion_notify(self, darea, event):
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        scale = self.scale
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        part = self.get_touched_region(x, y)
        if part != self.__current_part:
            if part == '':
                self.window.set_tooltip_text(None)
                self.darea.window.set_cursor(None)
                self.request_parent(
                    'NOTIFY', 'notify_event',
                    'OnMouseLeave', x, y, '', self.side, self.__current_part)
            else:
                if part in self.tooltips:
                    tooltip = self.tooltips[part]
                    self.window.set_tooltip_text(tooltip)
                else:
                    self.window.set_tooltip_text(None)
                cursor = gtk.gdk.Cursor(gtk.gdk.HAND1)
                self.darea.window.set_cursor(cursor)
                self.request_parent(
                    'NOTIFY', 'notify_event',
                    'OnMouseEnter', x, y, '', self.side, part)
        self.__current_part = part
        if not self.request_parent('GET', 'busy'):
            if state & gtk.gdk.BUTTON1_MASK:
                if self.x_root is not None and \
                   self.y_root is not None:
                    self.dragged = True
                    x_delta = int(event.x_root - self.x_root)
                    y_delta = int(event.y_root - self.y_root)
                    x, y = self.position # XXX: without window_offset
                    self.set_position(x + x_delta, y + y_delta)
                    self.x_root = event.x_root
                    self.y_root = event.y_root
            elif state & gtk.gdk.BUTTON2_MASK or \
               state & gtk.gdk.BUTTON3_MASK:
                pass
            else:
                self.request_parent('NOTIFY', 'notify_surface_mouse_motion',
                                    self.side, x, y, part)
        return True

    def scroll(self, darea, event):
        x = int(event.x)
        y = int(event.y)
        scale = self.scale
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        if event.direction == gtk.gdk.SCROLL_UP:
            count = 1
        elif event.direction == gtk.gdk.SCROLL_DOWN:
            count = -1
        else:
            count = 0
        if count != 0:
            part = self.get_touched_region(x, y)
            self.request_parent('NOTIFY', 'notify_event',
                                'OnMouseWheel', x, y, count, self.side, part)
        return True

    def toggle_bind(self, bind_id):
        if bind_id in self.bind:
            current = self.bind[bind_id][1]
            self.bind[bind_id][1] = not current
            self.reset_surface()

    def window_enter_notify(self, window, event):
        x, y, state = event.x, event.y, event.state
        scale = self.scale
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        self.request_parent('NOTIFY', 'notify_event',
                            'OnMouseEnterAll', x, y, '', self.side, '')

    def window_leave_notify(self, window, event):
        x, y, state = event.x, event.y, event.state
        scale = self.scale
        x = int(x * 100 / scale)
        y = int(y * 100 / scale)
        if self.__current_part != '': # XXX
            self.request_parent(
                'NOTIFY', 'notify_event',
                'OnMouseLeave', x, y, '', self.side, self.__current_part)
            self.__current_part = ''
        self.request_parent(
            'NOTIFY', 'notify_event',
            'OnMouseLeaveAll', x, y, '', self.side, '')
        return True


def test():
    pass

if __name__ == '__main__':
    test()
