# Copyright (C) 2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

"""Main TestCase class"""

import unittest
import testtools

from mago.application import TestApplication
from mago.config import get_config

from inspect import getfile, getmodule

class TestCase(testtools.TestCase):
    """Base class for desktop testing

    This class setup the test and start/stop the application under test"""
    setupOnce = False
    window_name = None

    # Set it static because LDTP can run only one app at once.
    application = None

    # Class and Test Configuration
    # This configuration is specific to the run and different from the
    # global configuration.
    # By default the name $(basename MODULEFILEPATH).ini
    testConfig = None

    def __init__(self, methodName = 'testMethod'):
        """Construct a TestCase instance

        Loads the configuration for the application and the test
        """
        super(TestCase, self).__init__(methodName)
        #self.magoConfig = get_config()

        moduleFile = getfile(getmodule(self))
        classConfigFile = '.'.join(moduleFile.split('.')[:-1]) + ".ini"
        self.testConfig = get_config(classConfigFile)


    @classmethod
    def setUpClass(self):
        """setUpClass

        Called once for all the tests in a TestCase.
        set setupOnce to True to activate it.
        This is used if you want to launch the application to test only once
        at the beginning of the test session or want to do some extensive setup
        """
        super(TestCase, self).setUpClass()

    def setUp(self):
        """setUp

        Launch the application when setupOnce is set to False
        """
        super(TestCase, self).setUp()

        if not hasattr(TestCase, 'application'):
            TestCase.application = None

        application = TestCase.application
        # Check mandatory propertys, launcher, launcher_args and window_name

        if not application:
            TestCase.application =\
                    TestApplication(self.launcher, window_name = self.window_name)
            #TestCase.application.launch()
        else:
            if not application.isRunning:
                TestCase.application.launch()

    @classmethod
    def tearDownClass(self):
        """tearDownClass

        Called once at the end of the class to close the application and
        cleanup the environment and destroy the application.
        Only called if setupOnce is set to True.
        """
        super(TestCase, self).tearDownClass()
        if self.setupOnce and TestCase.application:
            del TestCase.application

    def tearDown(self):
        """tearDown

        Called once at the end of the test to close the application and
        cleanup the environment and destroy the application.
        Only called if setupOnce is set to False.
        """
        super(TestCase, self).tearDown()
        if not self.setupOnce and TestCase.application:
            del TestCase.application
