#ifndef SRMLOG_H
#define SRMLOG_H

#include <SRMTypes.h>

#ifdef __cplusplus
extern "C" {
#endif

#if DOXYGEN
#define FORMAT_CHECK
#else
#define FORMAT_CHECK __attribute__((format(printf, 1, 2)))
#endif

/**
 * @defgroup SRMLog SRMLog
 *
 * @brief Module for logging and debugging information.
 *
 * The @ref SRMLog module provides functions for logging messages and debugging information with varying verbosity levels.
 * The verbosity level can be controlled by setting the **SRM_DEBUG** environment variable to a non-negative integer value.
 *
 * @{
 */

/**
 * @brief Initialize the SRMLog module.
 *
 * This function initializes the @ref SRMLog module and sets the verbosity level based on the **SRM_DEBUG** environment variable.
 * The verbosity levels are defined as follows:
 *
 * - **SRM_DEBUG**=0: Disables messages except those generated by SRMLog().
 * - **SRM_DEBUG**=1: Prints messages generated by SRMLog() and SRMFatal().
 * - **SRM_DEBUG**=2: Prints messages generated by SRMLog(), SRMFatal(), and SRMError().
 * - **SRM_DEBUG**=3: Prints messages generated by SRMLog(), SRMFatal(), SRMError(), and SRMWarning().
 * - **SRM_DEBUG**=4: Prints messages generated by SRMLog(), SRMFatal(), SRMError(), SRMWarning(), and SRMDebug().
 */
void SRMLogInit();

/**
 * @brief Get the current verbosity level.
 *
 * @return The current verbosity level as an integer.
 */
Int32 SRMLogGetLevel();

/**
 * @brief Print a general message independent of the **SRM_DEBUG** value.
 *
 * @param format A format string for the message.
 * @param ... Additional arguments for the format string.
 */
FORMAT_CHECK void SRMLog(const char *format, ...);

/**
 * @brief Report an unrecoverable error. **SRM_DEBUG** >= 1.
 *
 * @param format A format string for the error message.
 * @param ... Additional arguments for the format string.
 */
FORMAT_CHECK void SRMFatal(const char *format, ...);

/**
 * @brief Report a nonfatal error. **SRM_DEBUG** >= 2.
 *
 * @param format A format string for the error message.
 * @param ... Additional arguments for the format string.
 */
FORMAT_CHECK void SRMError(const char *format, ...);

/**
 * @brief Report messages that indicate a risk. **SRM_DEBUG** >= 3.
 *
 * @param format A format string for the warning message.
 * @param ... Additional arguments for the format string.
 */
FORMAT_CHECK void SRMWarning(const char *format, ...);

/**
 * @brief Print debugging messages. **SRM_DEBUG** >= 4.
 *
 * @param format A format string for the debug message.
 * @param ... Additional arguments for the format string.
 */
FORMAT_CHECK void SRMDebug(const char *format, ...);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif // SRMLOG_H
