/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Proxy_p.h"

#include <QColor>

#include <GTLCore/Color.h>
#include <GTLCore/Type.h>
#include <GTLCore/TypesManager.h>
#include <OpenShiva/Kernel.h>

#include "ParametersWidgetBase_p.h"
#include "Utils_p.h"

using namespace QtShiva;

Proxy::Proxy( QObject* parent, const GTLCore::String& _name, ParametersWidgetBase::Parametrisation* _kernel, const GTLCore::Type* _type ) : QObject( parent ), m_name(_name), m_kernel( _kernel ), m_type(_type)
{
  
}

Proxy::~Proxy()
{
}

void Proxy::setBoolValue( bool v )
{
  m_kernel->setParameter( m_name, v );
  emit(valueChanged());
}

void Proxy::setIntValue( int v )
{
  m_kernel->setParameter( m_name, v );
  emit(valueChanged());
}

void Proxy::setDoubleValue( double v )
{
  m_kernel->setParameter( m_name, (float)v );
  emit(valueChanged());
}

void Proxy::setRgba( const QColor& c )
{
  m_kernel->setParameter( m_name, GTLCore::Value( GTLCore::Color( c.red() / 255.0f, c.green() / 255.0f, c.blue() / 255.0f, c.alpha() / 255.0f ) ) );
  emit(valueChanged());
}

void Proxy::setVector( const QList<double>& v )
{
  std::vector< GTLCore::Value > cv;
  foreach(double vd, v)
  {
    cv.push_back((float)vd);
  }
  m_kernel->setParameter( m_name, GTLCore::Value( cv, GTLCore::TypesManager::getVector( GTLCore::Type::Float32, v.size() )  ) );
  emit(valueChanged());
}

void Proxy::setValue( const GTLCore::Value& value)
{
  switch( m_type->dataType() )
  {
    case GTLCore::Type::BOOLEAN:
    {
      emit boolValueChanged( value.asBoolean() );
    }
      break;
    case GTLCore::Type::INTEGER32:
    {
      emit intValueChanged( value.asInt32() );
    }
      break;
    case GTLCore::Type::FLOAT32:
    {
      emit doubleValueChanged(value.asFloat32());
    }
      break;
    case GTLCore::Type::VECTOR:
    {
      emit vectorValueChanged( valueToList( value) );
    }
      break;
    case GTLCore::Type::STRUCTURE:
    {
      if(m_type == GTLCore::Type::Color)
      {
        GTLCore::Color color = value.asColor();
        double r = color.red() * 255;
        double g = color.green() * 255;
        double b = color.blue() * 255;
        double a = color.alpha() * 255;
        emit( rgbaValueChanged(QColor(r,g,b,a)) );
      }
    }
      break;
  }
  emit(valueChanged());
}

#include "Proxy_p.moc"
