/*
 * Copyright (C) 2013 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This file is part of online-accounts-ui
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "application-manager.h"

#include <Accounts/Account>
#include <Accounts/Manager>
#include <QDebug>
#include <QDir>
#include <QQmlComponent>
#include <QQmlContext>
#include <QQmlEngine>
#include <QSignalSpy>
#include <QTest>

#define TEST_DIR "/tmp/tst_application_manager"

using namespace OnlineAccountsUi;

namespace QTest {
template<>
char *toString(const QSet<QString> &set)
{
    QByteArray ba = "QSet<QString>(";
    QStringList list = set.toList();
    ba += list.join(", ");
    ba += ")";
    return qstrdup(ba.data());
}
} // QTest namespace

class ApplicationManagerTest: public QObject
{
    Q_OBJECT

public:
    ApplicationManagerTest();

private Q_SLOTS:
    void initTestCase();
    void testNonExistingApp();
    void testApplicationInfo_data();
    void testApplicationInfo();
    void testAclAdd_data();
    void testAclAdd();
    void testAclRemove_data();
    void testAclRemove();
    void testApplicationFromProfile_data();
    void testApplicationFromProfile();
    void testProviderInfo_data();
    void testProviderInfo();
    void testUsefulProviders_data();
    void testUsefulProviders();

private:
    void clearApplicationsDir();
    void clearProvidersDir();
    void clearTestDir();
    void writeAccountsFile(const QString &name, const QString &contents);

private:
    QDir m_testDir;
    QDir m_accountsDir;
};

ApplicationManagerTest::ApplicationManagerTest():
    QObject(0),
    m_testDir(TEST_DIR),
    m_accountsDir(TEST_DIR "/accounts")
{
}

void ApplicationManagerTest::clearApplicationsDir()
{
    QDir applicationsDir = m_accountsDir;
    if (applicationsDir.cd("applications")) {
        applicationsDir.removeRecursively();
        applicationsDir.mkpath(".");
    }
}

void ApplicationManagerTest::clearProvidersDir()
{
    QDir providersDir = m_accountsDir;
    if (providersDir.cd("providers")) {
        providersDir.removeRecursively();
        providersDir.mkpath(".");
    }
}

void ApplicationManagerTest::clearTestDir()
{
    m_testDir.removeRecursively();
    m_testDir.mkpath(".");
}

void ApplicationManagerTest::writeAccountsFile(const QString &name,
                                               const QString &contents)
{
    /* Create files into the appropriate directory:
     * .provider files go in "providers", .application in "applications" and so
     * on: so we just need to take the file extension and add one "s".
     */
    QFileInfo fileInfo(name);
    QString subDirName = fileInfo.suffix() + "s";
    m_accountsDir.mkpath(subDirName);
    QDir subDir = m_accountsDir;
    subDir.cd(subDirName);

    QFile file(subDir.filePath(name));
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        qWarning() << "Could not write file" << name;
        return;
    }

    file.write(contents.toUtf8());
}

void ApplicationManagerTest::initTestCase()
{
    qputenv("ACCOUNTS", TEST_DIR);
    qputenv("XDG_DATA_HOME", TEST_DIR);
    qputenv("XDG_DATA_DIRS", TEST_DIR);

    clearTestDir();

    /* Create a few service files */
    writeAccountsFile("cool-mail.service",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<service id=\"cool-mail\">\n"
        "  <type>tstemail</type>\n"
        "  <name>Cool Mail</name>\n"
        "  <provider>cool</provider>\n"
        "</service>");
    writeAccountsFile("cool-sharing.service",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<service id=\"cool-sharing\">\n"
        "  <type>tstsharing</type>\n"
        "  <name>Cool Sharing</name>\n"
        "  <provider>cool</provider>\n"
        "</service>");
    writeAccountsFile("bad-mail.service",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<service id=\"bad-mail\">\n"
        "  <type>tstemail</type>\n"
        "  <name>Bad Mail</name>\n"
        "  <provider>bad</provider>\n"
        "</service>");
    writeAccountsFile("bad-sharing.service",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<service id=\"bad-sharing\">\n"
        "  <type>tstsharing</type>\n"
        "  <name>Bad Sharing</name>\n"
        "  <provider>bad</provider>\n"
        "</service>");

    /* And the relative providers */
    writeAccountsFile("cool.provider",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"cool\">\n"
        "  <name>Cool provider</name>\n"
        "</provider>");
    writeAccountsFile("bad.provider",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"bad\">\n"
        "  <name>Bad provider</name>\n"
        "</provider>");
}

void ApplicationManagerTest::testNonExistingApp()
{
    ApplicationManager manager;

    QVariantMap info = manager.applicationInfo("com.lomiri.test_MyApp",
                                               "com.lomiri.test_MyApp_0.1");
    QVERIFY(info.isEmpty());

    QStringList acl;
    acl << "one" << "two";
    QStringList newAcl = manager.addApplicationToAcl(acl, "com.lomiri.test_MyApp");
    QCOMPARE(newAcl, acl);

    newAcl = manager.removeApplicationFromAcl(acl, "com.lomiri.test_MyApp");
    QCOMPARE(newAcl, acl);
}

void ApplicationManagerTest::testApplicationInfo_data()
{
    QTest::addColumn<QString>("applicationId");
    QTest::addColumn<QString>("contents");
    QTest::addColumn<QString>("inputProfile");
    QTest::addColumn<QString>("expectedProfile");
    QTest::addColumn<QStringList>("services");

    QTest::newRow("no-services") <<
        "com.lomiri.test_MyApp" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_MyApp_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp_0.2" <<
        "com.lomiri.test_MyApp_0.2" <<
        QStringList();

    QTest::newRow("no-valid-services") <<
        "com.lomiri.test_MyApp2" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp2\">\n"
        "  <description>My application 2</description>\n"
        "  <service-types>\n"
        "    <service-type id=\"some type\">Do something</service-type>\n"
        "  </service-types>\n"
        "  <profile>com.lomiri.test_MyApp2_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp2_0.2" <<
        "com.lomiri.test_MyApp2_0.2" <<
        QStringList();

    QTest::newRow("email-services") <<
        "com.lomiri.test_MyApp3" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp3\">\n"
        "  <description>My application 3</description>\n"
        "  <service-types>\n"
        "    <service-type id=\"tstemail\">\n"
        "      <description>Send email</description>\n"
        "    </service-type>\n"
        "  </service-types>\n"
        "  <profile>com.lomiri.test_MyApp3_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp3_0.2" <<
        "com.lomiri.test_MyApp3_0.2" <<
        (QStringList() << "cool-mail" << "bad-mail");

    QTest::newRow("cool-services") <<
        "com.lomiri.test_MyApp4" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp4\">\n"
        "  <description>My application 4</description>\n"
        "  <services>\n"
        "    <service id=\"cool-mail\">\n"
        "      <description>Send email</description>\n"
        "    </service>\n"
        "    <service id=\"cool-sharing\">\n"
        "      <description>Share stuff</description>\n"
        "    </service>\n"
        "  </services>\n"
        "  <profile>com.lomiri.test_MyApp4_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp4_0.2" <<
        "com.lomiri.test_MyApp4_0.2" <<
        (QStringList() << "cool-mail" << "cool-sharing");

    QTest::newRow("unconfined app") <<
        "com.lomiri.test_UnconfinedApp" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_UnconfinedApp\">\n"
        "  <description>My application 5</description>\n"
        "  <services>\n"
        "    <service id=\"cool-mail\">\n"
        "      <description>Send email</description>\n"
        "    </service>\n"
        "  </services>\n"
        "  <profile>com.lomiri.test_UnconfinedApp_0.2</profile>\n"
        "</application>" <<
        "unconfined" <<
        "com.lomiri.test_UnconfinedApp_0.2" <<
        (QStringList() << "cool-mail");

    QTest::newRow("unconfined app, no profile") <<
        "com.lomiri.test_UnconfinedApp2" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_UnconfinedApp2\">\n"
        "  <description>My application 6</description>\n"
        "  <services>\n"
        "    <service id=\"cool-mail\">\n"
        "      <description>Send email</description>\n"
        "    </service>\n"
        "  </services>\n"
        "</application>" <<
        "unconfined" <<
        "unconfined" <<
        (QStringList() << "cool-mail");
}

void ApplicationManagerTest::testApplicationInfo()
{
    clearApplicationsDir();

    QFETCH(QString, applicationId);
    QFETCH(QString, contents);
    QFETCH(QString, inputProfile);
    QFETCH(QString, expectedProfile);
    QFETCH(QStringList, services);

    writeAccountsFile(applicationId + ".application", contents);

    ApplicationManager manager;

    QVariantMap info = manager.applicationInfo(applicationId, inputProfile);
    QCOMPARE(info.value("id").toString(), applicationId);
    QCOMPARE(info.value("profile").toString(), expectedProfile);
    QCOMPARE(info.value("services").toStringList().toSet(), services.toSet());
}

void ApplicationManagerTest::testAclAdd_data()
{
    QTest::addColumn<QString>("applicationId");
    QTest::addColumn<QStringList>("oldAcl");
    QTest::addColumn<QStringList>("newAcl");

    QTest::newRow("add to empty") <<
        "com.lomiri.test_One" <<
        QStringList() <<
        (QStringList() << "com.lomiri.test_One_0.1");

    QTest::newRow("add to one") <<
        "com.lomiri.test_One" <<
        (QStringList() << "some app") <<
        (QStringList() << "some app" << "com.lomiri.test_One_0.1");

    QTest::newRow("add existing") <<
        "com.lomiri.test_One" <<
        (QStringList() << "com.lomiri.test_One_0.1" << "some app") <<
        (QStringList() << "com.lomiri.test_One_0.1" << "some app");
}

void ApplicationManagerTest::testAclAdd()
{
    clearApplicationsDir();

    writeAccountsFile("com.lomiri.test_One.application",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_One\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_One_0.1</profile>\n"
        "</application>");
    writeAccountsFile("com.lomiri.test_Two.application",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_Two\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_Two_0.2</profile>\n"
        "</application>");

    QFETCH(QString, applicationId);
    QFETCH(QStringList, oldAcl);
    QFETCH(QStringList, newAcl);

    ApplicationManager manager;

    QStringList acl = manager.addApplicationToAcl(oldAcl, applicationId);
    QCOMPARE(acl.toSet(), newAcl.toSet());
}

void ApplicationManagerTest::testAclRemove_data()
{
    QTest::addColumn<QString>("applicationId");
    QTest::addColumn<QStringList>("oldAcl");
    QTest::addColumn<QStringList>("newAcl");

    QTest::newRow("remove from empty") <<
        "com.lomiri.test_One" <<
        QStringList() <<
        QStringList();

    QTest::newRow("remove from one") <<
        "com.lomiri.test_One" <<
        (QStringList() << "com.lomiri.test_One_0.1") <<
        QStringList();

    QTest::newRow("remove from two") <<
        "com.lomiri.test_One" <<
        (QStringList() << "com.lomiri.test_One_0.1" << "some app") <<
        (QStringList() << "some app");

    QTest::newRow("remove from missing") <<
        "com.lomiri.test_One" <<
        (QStringList() << "some other app" << "some app") <<
        (QStringList() << "some other app" << "some app");

    QTest::newRow("remove multiple versions") <<
        "com.lomiri.test_One" <<
        (QStringList() << "some other app" << "com.lomiri.test_One_0.1" <<
         "com.lomiri.test_One_0.3" << "some app" <<
         "com.lomiri.test_One_0.2") <<
        (QStringList() << "some other app" << "some app");
}

void ApplicationManagerTest::testAclRemove()
{
    clearApplicationsDir();

    writeAccountsFile("com.lomiri.test_One.application",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_One\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_One_0.1</profile>\n"
        "</application>");
    writeAccountsFile("com.lomiri.test_Two.application",
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_Two\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_Two_0.2</profile>\n"
        "</application>");

    QFETCH(QString, applicationId);
    QFETCH(QStringList, oldAcl);
    QFETCH(QStringList, newAcl);

    ApplicationManager manager;

    QStringList acl = manager.removeApplicationFromAcl(oldAcl, applicationId);
    QCOMPARE(acl.toSet(), newAcl.toSet());
}

void ApplicationManagerTest::testApplicationFromProfile_data()
{
    QTest::addColumn<QString>("applicationId");
    QTest::addColumn<QString>("contents");
    QTest::addColumn<QString>("profile");
    QTest::addColumn<bool>("isValid");

    QTest::newRow("id-same-as-profile") <<
        "com.lomiri.test_MyApp_0.2" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp\">\n"
        "  <description>My application</description>\n"
        "  <profile>com.lomiri.test_MyApp_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp_0.2" <<
        true;

    QTest::newRow("id-with-no-version") <<
        "com.lomiri.test_MyApp2" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp2\">\n"
        "  <description>My application 2</description>\n"
        "  <profile>com.lomiri.test_MyApp2_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp2_0.2" <<
        true;

    QTest::newRow("id-is-just-package") <<
        "com.lomiri.test" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp3\">\n"
        "  <description>My application 3</description>\n"
        "  <profile>com.lomiri.test_MyApp3_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp3_0.2" <<
        true;

    QTest::newRow("not found") <<
        "com.lomiri.test_MyApp5" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp4\">\n"
        "  <description>My application 4</description>\n"
        "  <profile>com.lomiri.test_MyApp4_0.2</profile>\n"
        "</application>" <<
        "com.lomiri.test_MyApp4_0.2" <<
        false;
}

void ApplicationManagerTest::testApplicationFromProfile()
{
    clearApplicationsDir();

    QFETCH(QString, applicationId);
    QFETCH(QString, contents);
    QFETCH(QString, profile);
    QFETCH(bool, isValid);

    writeAccountsFile(applicationId + ".application", contents);

    ApplicationManager manager;

    Accounts::Application app = manager.applicationFromProfile(profile);
    if (!isValid) {
        QVERIFY(!app.isValid());
    } else {
        QVERIFY(app.isValid());
        QCOMPARE(app.name(), applicationId);
    }
}

void ApplicationManagerTest::testProviderInfo_data()
{
    QTest::addColumn<QString>("providerId");
    QTest::addColumn<QString>("contents");
    QTest::addColumn<QString>("profile");
    QTest::addColumn<QString>("packageDir");
    QTest::addColumn<bool>("isSingleAccount");

    QTest::newRow("no profile") <<
        "com.lomiri.test_MyPlugin" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"com.lomiri.test_MyPlugin\">\n"
        "  <name>My Plugin</name>\n"
        "</provider>" <<
        QString() <<
        QString() <<
        false;

    QTest::newRow("single account") <<
        "com.lomiri.test_MyPlugin" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"com.lomiri.test_MyPlugin\">\n"
        "  <name>My Plugin</name>\n"
        "  <single-account>true</single-account>\n"
        "</provider>" <<
        QString() <<
        QString() <<
        true;

    QTest::newRow("no package-dir") <<
        "com.lomiri.test_MyPlugin2" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"com.lomiri.test_MyPlugin2\">\n"
        "  <name>My Plugin</name>\n"
        "  <profile>com.lomiri.test_MyPlugin2_0.2</profile>\n"
        "  <single-account>false</single-account>\n"
        "</provider>" <<
        "com.lomiri.test_MyPlugin2_0.2" <<
        QString() <<
        false;

    QTest::newRow("with package-dir") <<
        "com.lomiri.test_MyPlugin3" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<provider id=\"com.lomiri.test_MyPlugin3\">\n"
        "  <name>My Plugin</name>\n"
        "  <profile>com.lomiri.test_MyPlugin3_0.2</profile>\n"
        "  <package-dir>/opt/click.ubuntu.com/something</package-dir>\n"
        "</provider>" <<
        "com.lomiri.test_MyPlugin3_0.2" <<
        "/opt/click.ubuntu.com/something" <<
        false;
}

void ApplicationManagerTest::testProviderInfo()
{
    clearProvidersDir();

    QFETCH(QString, providerId);
    QFETCH(QString, contents);
    QFETCH(QString, profile);
    QFETCH(QString, packageDir);
    QFETCH(bool, isSingleAccount);

    writeAccountsFile(providerId + ".provider", contents);

    ApplicationManager manager;

    QVariantMap info = manager.providerInfo(providerId);
    QCOMPARE(info.value("id").toString(), providerId);
    QCOMPARE(info.value("profile").toString(), profile);
    QCOMPARE(info.value("package-dir").toString(), packageDir);
    QCOMPARE(info.value("isSingleAccount").toBool(), isSingleAccount);
}

void ApplicationManagerTest::testUsefulProviders_data()
{
    QTest::addColumn<QStringList>("applicationIds");
    QTest::addColumn<QStringList>("contents");
    QTest::addColumn<QStringList>("expectedProviders");

    QStringList applicationIds;
    QStringList contents;
    applicationIds << "com.lomiri.test_MyApp";
    contents <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp\">\n"
        "  <description>My application</description>\n"
        "</application>";
    QTest::newRow("no-services") <<
        applicationIds << contents <<
        QStringList();
    applicationIds.clear();
    contents.clear();

    applicationIds <<
        "com.lomiri.test_MyApp2";
    contents <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp2\">\n"
        "  <description>My application 2</description>\n"
        "  <service-types>\n"
        "    <service-type id=\"some type\">Do something</service-type>\n"
        "  </service-types>\n"
        "</application>";
    QTest::newRow("no-valid-services") <<
        applicationIds << contents <<
        QStringList();
    applicationIds.clear();
    contents.clear();

    applicationIds <<
        "com.lomiri.test_MyApp3";
    contents <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp3\">\n"
        "  <description>My application 3</description>\n"
        "  <service-types>\n"
        "    <service-type id=\"tstemail\">\n"
        "      <description>Send email</description>\n"
        "    </service-type>\n"
        "  </service-types>\n"
        "</application>";
    QTest::newRow("email-services") <<
        applicationIds << contents <<
        (QStringList() << "cool" << "bad");
    applicationIds.clear();
    contents.clear();

    applicationIds <<
        "com.lomiri.test_MyApp4";
    contents <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp4\">\n"
        "  <description>My application 4</description>\n"
        "  <services>\n"
        "    <service id=\"cool-mail\">\n"
        "      <description>Send email</description>\n"
        "    </service>\n"
        "    <service id=\"cool-sharing\">\n"
        "      <description>Share stuff</description>\n"
        "    </service>\n"
        "  </services>\n"
        "</application>";
    QTest::newRow("cool-services") <<
        applicationIds << contents <<
        (QStringList() << "cool");
    applicationIds.clear();
    contents.clear();

    applicationIds <<
        "com.lomiri.test_MyApp5" <<
        "com.lomiri.test_MyApp6";
    contents <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp5\">\n"
        "  <description>My application 5</description>\n"
        "  <services>\n"
        "    <service id=\"cool-mail\">\n"
        "      <description>Send email</description>\n"
        "    </service>\n"
        "  </services>\n"
        "</application>" <<
        "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
        "<application id=\"com.lomiri.test_MyApp6\">\n"
        "  <description>My application 6</description>\n"
        "  <services>\n"
        "    <service id=\"bad-sharing\">\n"
        "      <description>Share stuff</description>\n"
        "    </service>\n"
        "  </services>\n"
        "</application>";
    QTest::newRow("two apps") <<
        applicationIds << contents <<
        (QStringList() << "cool" << "bad");
    applicationIds.clear();
    contents.clear();
}

void ApplicationManagerTest::testUsefulProviders()
{
    clearApplicationsDir();

    QFETCH(QStringList, applicationIds);
    QFETCH(QStringList, contents);
    QFETCH(QStringList, expectedProviders);

    for (int i = 0; i < applicationIds.count(); i++) {
        writeAccountsFile(applicationIds[i] + ".application", contents[i]);
    }

    ApplicationManager manager;

    QStringList providers = manager.usefulProviders();
    QCOMPARE(providers.toSet(), expectedProviders.toSet());
}

QTEST_GUILESS_MAIN(ApplicationManagerTest);

#include "tst_application_manager.moc"
