/*
  ==============================================================================

   This file is part of the JUCE framework.
   Copyright (c) Raw Material Software Limited

   JUCE is an open source framework subject to commercial or open source
   licensing.

   By downloading, installing, or using the JUCE framework, or combining the
   JUCE framework with any other source code, object code, content or any other
   copyrightable work, you agree to the terms of the JUCE End User Licence
   Agreement, and all incorporated terms including the JUCE Privacy Policy and
   the JUCE Website Terms of Service, as applicable, which will bind you. If you
   do not agree to the terms of these agreements, we will not license the JUCE
   framework to you, and you must discontinue the installation or download
   process and cease use of the JUCE framework.

   JUCE End User Licence Agreement: https://juce.com/legal/juce-8-licence/
   JUCE Privacy Policy: https://juce.com/juce-privacy-policy
   JUCE Website Terms of Service: https://juce.com/juce-website-terms-of-service/

   Or:

   You may also use this code under the terms of the AGPLv3:
   https://www.gnu.org/licenses/agpl-3.0.en.html

   THE JUCE FRAMEWORK IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL
   WARRANTIES, WHETHER EXPRESSED OR IMPLIED, INCLUDING WARRANTY OF
   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, ARE DISCLAIMED.

  ==============================================================================
*/

/* This file was generated with JUCE-Utils/Unicode/generate_unicode_blob.py */
namespace juce
{
enum class LineBreakType : uint8_t
{
    al,
    bk,
    cm,
    cr,
    gl,
    lf,
    nl,
    sp,
    wj,
    zw,
    zwj,
    ai,
    b2,
    ba,
    bb,
    cb,
    cj,
    cl,
    cp,
    eb,
    em,
    ex,
    h2,
    h3,
    hl,
    hy,
    in,
    is,
    jl,
    id,
    jt,
    jv,
    ns,
    nu,
    op,
    po,
    pr,
    qu,
    ri,
    sa,
    sg,
    sy,
    xx,
    opw
};

enum class EastAsianWidthType : uint8_t
{
    neutral,
    narrow,
    ambiguous,
    full,
    half,
    wide
};

enum class BidiType : uint8_t
{
    ltr,
    rtl,
    al,
    en,
    an,
    es,
    et,
    cs,
    nsm,
    bn,
    b,
    s,
    ws,
    on,
    lre,
    lro,
    rle,
    rlo,
    pdf,
    lri,
    rli,
    fsi,
    pdi,
    none
};

enum class VerticalTransformType : uint8_t
{
    r,
    u,
    tr,
    tu
};

enum class EmojiType : uint8_t
{
    yes,
    presentation,
    modifier,
    modifierBase,
    component,
    extended,
    no
};

enum class UnicodeScriptType : uint8_t
{
    common,
    inherited,
    han,
    arabic,
    hiragana,
    adlam,
    mende_kikakui,
    ethiopic,
    wancho,
    toto,
    nyiakeng_puachue_hmong,
    glagolitic,
    latin,
    signwriting,
    greek,
    duployan,
    nushu,
    katakana,
    tangut,
    khitan_small_script,
    miao,
    medefaidrin,
    pahawh_hmong,
    bassa_vah,
    tangsa,
    mro,
    bamum,
    cypro_minoan,
    cuneiform,
    tamil,
    lisu,
    makasar,
    gunjala_gondi,
    masaram_gondi,
    marchen,
    bhaiksuki,
    pau_cin_hau,
    canadian_aboriginal,
    soyombo,
    zanabazar_square,
    nandinagari,
    dives_akuru,
    warang_citi,
    dogra,
    ahom,
    takri,
    mongolian,
    modi,
    siddham,
    tirhuta,
    newa,
    grantha,
    khudawadi,
    multani,
    khojki,
    sinhala,
    sharada,
    mahajani,
    chakma,
    sora_sompeng,
    kaithi,
    brahmi,
    elymaic,
    chorasmian,
    sogdian,
    yezidi,
    hanifi_rohingya,
    psalter_pahlavi,
    avestan,
    manichaean,
    kharoshthi,
    meroitic_cursive,
    lydian,
    phoenician,
    hatran,
    nabataean,
    palmyrene,
    imperial_aramaic,
    cypriot,
    vithkuqi,
    caucasian_albanian,
    elbasan,
    osage,
    osmanya,
    shavian,
    deseret,
    ugaritic,
    gothic,
    carian,
    lycian,
    hangul,
    cyrillic,
    hebrew,
    armenian,
    meetei_mayek,
    cherokee,
    tai_viet,
    myanmar,
    cham,
    javanese,
    rejang,
    kayah_li,
    devanagari,
    saurashtra,
    phags_pa,
    syloti_nagri,
    vai,
    yi,
    bopomofo,
    tifinagh,
    georgian,
    coptic,
    braille,
    sundanese,
    ol_chiki,
    lepcha,
    batak,
    balinese,
    tai_tham,
    buginese,
    khmer,
    limbu,
    tai_le,
    tagbanwa,
    buhid,
    hanunoo,
    tagalog,
    runic,
    ogham,
    tibetan,
    lao,
    thai,
    malayalam,
    kannada,
    telugu,
    oriya,
    gujarati,
    gurmukhi,
    bengali,
    syriac,
    mandaic,
    samaritan,
    nko,
    thaana,
    linear_a,
    linear_b,
    new_tai_lue,
    old_hungarian,
    old_turkic,
    old_uyghur,
    old_sogdian,
    old_south_arabian,
    old_north_arabian,
    old_persian,
    old_permic,
    old_italic,
    inscriptional_pahlavi,
    inscriptional_parthian,
    anatolian_hieroglyphs,
    egyptian_hieroglyphs,
    meroitic_hieroglyphs,
    emoji
};

enum class BreakOpportunity : uint8_t
{
    direct,
    indirect,
    prohibited,
    combinedIndirect,
    combinedProhibited
};

enum class GeneralCategory : uint8_t
{
    xx,
    cn,
    lu,
    ll,
    lt,
    lm,
    lo,
    mn,
    me,
    mc,
    nd,
    nl,
    no,
    zs,
    zl,
    zp,
    cc,
    cf,
    co,
    cs,
    pd,
    ps,
    pe,
    pc,
    po,
    sm,
    sc,
    sk,
    so,
    pi,
    pf
};


struct UnicodeEntry
{
    LineBreakType         bt;
    EastAsianWidthType    asian;
    BidiType              bidi;
    UnicodeScriptType     script;
    VerticalTransformType vertical;
    EmojiType             emoji;
    GeneralCategory       generalCategory;
};

class UnicodeDataTable
{
public:
    UnicodeDataTable() = delete;

    static UnicodeEntry getDataForCodepoint (uint32_t codepoint)
    {
        static const auto data = []
        {
            [[maybe_unused]] constexpr auto hash = []
            {
                uint32_t v = 5381;

                for (auto c : compressedData)
                    v = ((v << 5) + v) + (uint32_t) c;

                return v;
            }();

            jassert (hash == 0x7f9e2da);

            MemoryInputStream           mStream { compressedData, std::size (compressedData), false };
            GZIPDecompressorInputStream zStream { &mStream, false };

            Array<UnicodeEntry> arr;
            arr.resize (uncompressedSize / sizeof (UnicodeEntry));
            [[maybe_unused]] const auto r = zStream.read (arr.getRawDataPointer(), uncompressedSize);
            jassert (r == uncompressedSize);

            return arr;
        }();

        jassert (isPositiveAndBelow (codepoint, data.size()));
        return data[(int) codepoint];
    }

private:
    static constexpr auto    uncompressedSize = 7798777;
    static constexpr uint8_t compressedData[]
    {
        0x78,0x9c,0xec,0xdd,0xf9,0x83,0x23,0xd9,0x5d,0x18,0xf0,0x27,0xf5,0x48,0xd3,0xdd,0xd3,0x1a,0x7b,0xec,0x99,0xb5,0x66,0xa6,
        0xc7,0x9e,0x9e,0xd9,0xc5,0x30,0x80,0x09,0x47,0x30,0x10,0x93,0x70,0x1f,0x39,0xc9,0x9d,0x78,0x31,0x37,0x38,0x24,0x80,0x43,
        0xb0,0xc1,0x84,0x23,0x60,0x7c,0x03,0x26,0x21,0x21,0x07,0x81,0x24,0x7f,0x6a,0xaa,0xea,0x55,0xab,0x55,0x52,0xa9,0x54,0x52,
        0x95,0xa4,0xd7,0xea,0xcf,0xfe,0xd0,0x9f,0xad,0xe3,0x1d,0xdf,0x77,0x55,0x95,0xae,0x19,0x86,0xb3,0x10,0xc6,0xef,0x1d,0x6e,
        0xc9,0x24,0x3c,0xc8,0x19,0x85,0xf3,0x9c,0x41,0xdc,0x1a,0x84,0x8b,0x9c,0x93,0xb8,0x73,0xdb,0xac,0x37,0xe1,0xbc,0x8e,0xa2,
        0x2e,0xf7,0x07,0x79,0x5d,0x26,0x4f,0x06,0x93,0x8c,0xe9,0x3b,0x91,0x30,0x18,0x87,0x70,0x6f,0xfa,0x76,0xce,0xf8,0xd9,0xeb,
        0x82,0x6c,0xe7,0xfc,0x29,0xaf,0x0a,0x9e,0xbc,0xaf,0xe0,0xad,0xe2,0x58,0x9e,0x60,0x94,0x6d,0x3d,0x7d,0x3e,0xb8,0x9f,0x9f,
        0xf2,0xb4,0xd8,0x7a,0x5c,0x6e,0x7d,0x43,0xe4,0x6a,0x70,0x92,0x9d,0x79,0xde,0x91,0x32,0xcf,0xe7,0xd7,0xd5,0xcd,0x79,0x5a,
        0xe5,0xc9,0xfc,0xb1,0x8c,0xac,0xcd,0xc7,0xc3,0x5b,0x46,0xd9,0xc8,0x6f,0xc7,0x18,0xe6,0xdb,0x7a,0xfc,0x3c,0xf2,0x81,0xd9,
        0x56,0x9e,0xe0,0xe4,0x96,0x51,0xc6,0x37,0x89,0x3d,0xf6,0x68,0x3e,0xbe,0xa7,0xed,0x87,0xf7,0x78,0x7f,0xf3,0xa8,0x7f,0xee,
        0x85,0x7c,0x24,0x4f,0x5e,0x0d,0x8b,0x2e,0x8e,0x53,0xed,0x59,0x39,0xf1,0xde,0x1e,0xce,0x6f,0xc5,0x76,0xb9,0x7c,0x90,0x9d,
        0x39,0x18,0x4f,0x1f,0x0c,0x63,0xbf,0x87,0x6c,0x6b,0x94,0xed,0xcc,0xdb,0x73,0xfc,0x4e,0xc8,0x77,0xbe,0x28,0x5b,0x70,0x32,
        0xcc,0x4b,0x78,0x14,0x86,0xc5,0x29,0xe5,0x40,0x79,0x5d,0xe4,0x79,0x99,0x67,0x3d,0x18,0x3f,0x7d,0x30,0xcc,0xa6,0xd3,0xf8,
        0xa2,0xe4,0xe1,0x75,0x9e,0xd9,0x7f,0xe3,0x93,0x58,0xc2,0xb4,0xc2,0xf3,0x59,0x82,0xb9,0xf2,0x3e,0x18,0xab,0x74,0x51,0xa5,
        0x8c,0x28,0xcb,0xaa,0x18,0xd0,0x2d,0x18,0xb6,0x3e,0xb3,0xcf,0x5c,0x62,0x75,0x9f,0x6e,0x90,0x6e,0x38,0xc7,0xc9,0x02,0xa1,
        0x81,0x61,0xdd,0x56,0x6d,0xf2,0xa6,0x3c,0xdb,0x9f,0x19,0xa9,0xc4,0xd7,0x5c,0x5e,0xed,0xd6,0x7c,0xd0,0x27,0x95,0x96,0xd8,
        0xe9,0xd6,0xb0,0x97,0xd2,0x7b,0xad,0xf5,0x70,0x7d,0x9e,0xed,0xcb,0x1b,0xb6,0xce,0xb3,0xa6,0x57,0xb6,0xa8,0x6e,0x6d,0x41,
        0x2d,0x42,0xd9,0x36,0x79,0xfb,0x36,0xdb,0xdb,0x56,0xaf,0xfd,0xd7,0x69,0xab,0x7d,0x82,0xc5,0x69,0xd8,0xa9,0x84,0xda,0x63,
        0x4d,0xa7,0xf4,0x93,0x75,0xfb,0xe4,0xfd,0xb4,0x60,0x3f,0xed,0xd2,0x22,0x79,0xc9,0x78,0x7d,0xec,0xe3,0x06,0xca,0x74,0xf7,
        0xea,0xca,0x6b,0xb5,0xb3,0x69,0x52,0xf6,0xbf,0x75,0xe8,0x59,0xdc,0xb6,0x95,0xb6,0xed,0x62,0x5b,0x69,0xf4,0x6d,0x0b,0xda,
        0xaf,0x28,0xed,0x87,0x46,0x5f,0x31,0xb4,0xb8,0x1f,0xec,0x95,0x7d,0x97,0x77,0x77,0x19,0xa7,0x50,0x89,0x8e,0x8c,0xb6,0x27,
        0x7f,0xaa,0xbb,0x21,0x6c,0xca,0xec,0xa9,0x35,0x67,0x58,0xd9,0x8a,0x79,0xc6,0x07,0xcd,0xd1,0xc3,0xca,0x56,0x95,0xea,0xb1,
        0x6a,0x95,0x1e,0x0c,0x1b,0xca,0x6b,0x60,0xf6,0x7c,0x5b,0x43,0xf5,0x94,0x72,0xeb,0xe6,0x61,0xb9,0x5d,0x9b,0xad,0xab,0xc4,
        0xaf,0x0c,0xe6,0xa9,0x54,0x7e,0xc3,0x50,0x7a,0x65,0x38,0x3c,0x1d,0x84,0xf1,0x7d,0xa0,0x99,0x7b,0x3b,0xcf,0xba,0x3d,0x3b,
        0x0c,0x33,0x9b,0x8b,0x0f,0xe3,0xd5,0xfe,0x61,0x5c,0x4d,0x2b,0x5b,0xd7,0xf3,0xf6,0x61,0x9c,0xb7,0x73,0xc7,0xde,0x14,0xb3,
        0x78,0x50,0x12,0x77,0x8e,0x2a,0xb9,0x5c,0xf3,0x3c,0x5c,0xbf,0x70,0x96,0x27,0xaf,0xa6,0x5b,0xcc,0xa5,0x2c,0xe8,0x7a,0x9d,
        0x88,0xe5,0xcd,0x27,0xaf,0x52,0x29,0x7d,0x61,0x6b,0x21,0x94,0x61,0xdc,0xda,0x33,0xd7,0x55,0xda,0x38,0x79,0x6d,0x0c,0x4d,
        0x0c,0x0f,0xc3,0xb6,0xa5,0xb7,0x88,0x68,0x75,0xec,0xed,0x9b,0x67,0xd5,0xb8,0xde,0x60,0xeb,0x53,0x71,0xeb,0x53,0xbb,0xdb,
        0x6a,0xdd,0x04,0xf9,0xec,0x78,0xda,0x50,0xdd,0x96,0x8d,0xf5,0x6e,0x1c,0x67,0xef,0xce,0x6d,0xed,0x90,0xe1,0x5d,0xe0,0xe4,
        0x78,0x09,0x75,0x5b,0xbb,0x66,0x78,0xa7,0xb6,0x2e,0x87,0xe1,0xf4,0xdd,0xfc,0x92,0x3c,0x63,0x50,0xe1,0xe6,0xd8,0x69,0x49,
        0x0a,0xb5,0xb6,0xb5,0x87,0x82,0x52,0x8a,0xd6,0x96,0xad,0xa5,0xad,0x9b,0x9b,0xee,0x4f,0xc4,0x9d,0x77,0x9d,0xa5,0x07,0xa3,
        0x4f,0xc4,0x07,0xa3,0x4f,0xc4,0x87,0x98,0x76,0x9c,0xa0,0x9e,0xe7,0xb1,0x95,0x26,0x61,0x92,0xf1,0x78,0xe9,0xf1,0x31,0xdb,
        0x79,0x59,0xf2,0x76,0x18,0x67,0x3c,0x7b,0x13,0x06,0xf9,0xb1,0xec,0xba,0xf9,0x93,0xf1,0x2a,0x8a,0x0d,0x99,0x84,0x41,0xc6,
        0xe3,0x72,0x2b,0x14,0x5b,0xd3,0xea,0x29,0xb5,0x3b,0x9f,0x54,0x76,0x96,0xfd,0xb0,0x31,0xd3,0x22,0x97,0xf1,0xed,0x64,0xdb,
        0xc0,0xca,0xf6,0x2c,0xd9,0xb6,0xe9,0x6a,0x09,0xe1,0xde,0x49,0xfe,0x21,0xaa,0x66,0x8a,0xcf,0x59,0x85,0xc9,0x49,0xf1,0xbc,
        0x59,0x32,0xcc,0x79,0x9d,0x4f,0xc5,0xf1,0x74,0x46,0xbe,0xf3,0xd9,0xf3,0x10,0x3f,0xbd,0x58,0x6c,0x4d,0x63,0x82,0xcb,0x92,
        0xac,0xfb,0x4f,0xe2,0x98,0xe8,0x81,0x27,0xd9,0x63,0x48,0x31,0xa4,0xf2,0x82,0x1e,0x3d,0x89,0xe5,0xdd,0x10,0x5f,0x27,0xca,
        0xb7,0xc6,0xc7,0x4c,0x7c,0x4d,0xae,0x8f,0xcc,0x96,0x9e,0x34,0x3a,0xd2,0x43,0x17,0x5f,0x15,0x63,0xf0,0xbc,0x23,0xe5,0xf8,
        0x8c,0x5b,0xd7,0x84,0xeb,0xf1,0x59,0xd3,0x04,0x07,0xef,0x54,0x60,0x53,0x9e,0x54,0x06,0xf4,0xc6,0x53,0xad,0xb2,0xce,0x6f,
        0xb0,0x50,0xc7,0xf2,0xca,0x25,0x68,0xb4,0x78,0x6c,0x6d,0x66,0xd5,0x18,0xae,0xc2,0xc9,0x49,0x31,0x6f,0x3b,0xd1,0xd0,0x4a,
        0x97,0x15,0xf2,0x9d,0x7f,0x1c,0xdb,0x6c,0x57,0xbc,0x29,0x56,0xe8,0x41,0xdc,0x7a,0x14,0xc9,0x3b,0xe7,0x8f,0x63,0xec,0x7f,
        0x3c,0xab,0xc4,0x51,0x50,0x06,0x76,0x3b,0x29,0xbb,0xaa,0xd2,0x63,0xcb,0x61,0x1e,0x7e,0xa2,0x1f,0x17,0x7f,0x86,0x3f,0x8b,
        0x33,0xe7,0xcf,0xe2,0x50,0xec,0x81,0x32,0xcf,0xea,0x80,0xde,0x05,0x57,0x61,0xf0,0xb5,0x62,0xdd,0xed,0x44,0x28,0x18,0x1f,
        0x3b,0x59,0xe7,0x7c,0x2d,0xf6,0xd1,0x56,0xc4,0x5c,0x46,0x33,0x26,0x5f,0x8b,0xcf,0x55,0x5f,0x2b,0x1e,0xb6,0x0a,0x9e,0x44,
        0xca,0x53,0xaa,0x8f,0x7a,0x65,0x2e,0x6f,0x17,0xc7,0x9e,0x95,0x64,0x67,0xfe,0x69,0xac,0x60,0x6a,0x64,0xd5,0xfd,0xd3,0x18,
        0xfb,0x32,0xf1,0x94,0x51,0xd3,0x29,0x2d,0x68,0xca,0xa5,0x7d,0x09,0x8b,0xcf,0xd3,0x79,0xba,0xe9,0xee,0xb9,0x29,0xef,0x4f,
        0x62,0x9b,0xdd,0x0a,0xb2,0x36,0xfb,0x93,0xd8,0x74,0x73,0x2c,0xb5,0xe0,0x9f,0xcc,0xc7,0xd7,0xe3,0xcd,0x49,0x8b,0xf5,0xec,
        0xe0,0x57,0xc3,0x8d,0x78,0xde,0x3e,0xc1,0x2c,0xbe,0xf9,0x97,0x76,0xda,0xaf,0xf3,0xdb,0x3e,0xbf,0xa7,0xd0,0x4a,0xc9,0x33,
        0xda,0xb6,0x75,0xf7,0x4d,0xf9,0x88,0x98,0x44,0x5d,0x7a,0xe3,0x97,0x6a,0x08,0x19,0x67,0xa1,0x60,0x8c,0xbd,0x53,0xed,0x87,
        0x72,0x6b,0x76,0xac,0xdc,0x59,0xa5,0xb6,0x1b,0x5b,0x50,0x9b,0x59,0x35,0xcf,0xf9,0x9d,0x37,0x15,0x6c,0x7a,0xe1,0x73,0xb9,
        0xa0,0x9e,0xdb,0xa5,0x60,0x52,0xbc,0xcf,0x35,0x2d,0xb9,0x2a,0x4e,0x39,0xef,0x48,0x2c,0x68,0x1a,0x19,0xed,0xa5,0xc3,0xbf,
        0x1a,0x03,0xfb,0x6a,0x6c,0xeb,0xaf,0xc6,0xb6,0xce,0xb9,0x79,0xcf,0xf4,0xab,0x37,0x67,0xb6,0x67,0xe9,0x9d,0xd7,0x96,0xc7,
        0x52,0x62,0xdb,0x0a,0x56,0xd3,0xad,0xf8,0x74,0x6d,0xdb,0xa6,0x2b,0x3b,0x67,0xd6,0x55,0xf3,0x7d,0x34,0xa3,0xec,0xbf,0x1a,
        0x16,0x33,0xab,0xe9,0xe2,0xda,0x63,0xd5,0x62,0x9b,0x3e,0x28,0xdc,0x44,0x6d,0x41,0x2d,0x86,0x46,0x6d,0xbb,0x34,0x05,0x76,
        0x55,0x9c,0x72,0xde,0x91,0x6a,0x79,0xaf,0xc3,0xf8,0xab,0xf9,0x2f,0x94,0x44,0xe2,0xce,0x8b,0x26,0x5e,0xcf,0x6f,0xe5,0xef,
        0x79,0xdf,0xa4,0x2b,0xf3,0x9c,0xae,0xe8,0x95,0xd3,0xaf,0xc4,0xc0,0x22,0xe1,0x2b,0x95,0xf9,0xf7,0x95,0x98,0x7c,0x1d,0x6b,
        0x1a,0x79,0xe5,0x99,0xed,0x4b,0xd8,0x37,0xdb,0x56,0xb0,0x29,0xe8,0xf6,0x3b,0xab,0x9d,0x73,0x33,0xa0,0xbf,0x12,0x27,0x49,
        0x85,0xf9,0xfe,0x6b,0x9e,0x0f,0xab,0xcf,0x5c,0x1c,0x05,0xab,0x33,0x6b,0x51,0x50,0x8b,0x51,0xd0,0xd4,0x12,0x1b,0x67,0x7d,
        0x55,0xa4,0x3b,0xef,0x48,0x35,0xf6,0xda,0x7a,0x56,0x8e,0x4d,0xb7,0x5d,0x97,0x96,0xda,0xf3,0xcb,0xb1,0xd8,0x48,0xf8,0x72,
        0x65,0xfe,0x7d,0x39,0x96,0xbe,0x15,0x4d,0xb9,0xf4,0x53,0xc2,0x0e,0xd9,0xb6,0x82,0xb5,0xe9,0xda,0x67,0x56,0xdb,0x39,0xe5,
        0xb1,0xb2,0x73,0x16,0xa9,0xf6,0x5f,0x0d,0x2d,0x7a,0xba,0x36,0xcf,0xa5,0x09,0xb4,0x5c,0xc1,0x1d,0x52,0x6d,0x97,0xda,0x88,
        0x6e,0xe6,0xdf,0x97,0xe3,0x3c,0xea,0x42,0x2c,0x68,0x9a,0x5d,0xb9,0xbe,0x5c,0x7c,0x5a,0x6b,0xbb,0xea,0xb6,0xe8,0x8e,0xa5,
        0x5e,0xf9,0x52,0xec,0x8e,0x2f,0xc5,0x0e,0xf8,0x52,0x65,0xfe,0x7d,0x29,0x36,0xc1,0x66,0x2c,0xd5,0xac,0xe5,0xb1,0x94,0xd8,
        0xb6,0x82,0x4d,0x41,0x6f,0xdc,0x74,0x65,0xe7,0x94,0xc7,0x66,0x7d,0x54,0xed,0xb1,0x72,0xab,0x86,0xa5,0xfb,0xc1,0xe5,0x2e,
        0xae,0x3d,0x56,0x9b,0xbc,0xdd,0x5a,0x5e,0xa9,0x44,0x4d,0x41,0x2d,0x86,0x46,0x6d,0xbb,0x34,0x05,0x76,0x55,0x9c,0x72,0xde,
        0x91,0x58,0xd0,0x65,0xa5,0xd8,0x8b,0x16,0xf4,0x76,0x19,0x7c,0x11,0x7b,0xfa,0x45,0xa5,0x25,0x5e,0xc4,0xba,0xac,0x63,0x55,
        0xeb,0xae,0x3c,0xb3,0x63,0x66,0xd5,0x9d,0x4d,0xa7,0xb4,0xc8,0xa5,0x7d,0xad,0x5b,0x9c,0xd9,0x91,0xe6,0x3e,0xca,0x4f,0x39,
        0x9b,0x51,0xf4,0x58,0x75,0x67,0xab,0x74,0xf3,0x67,0xd6,0x1e,0x2b,0xb3,0xae,0x8d,0x76,0x93,0x21,0xb5,0xba,0x4a,0xbd,0x72,
        0x55,0x14,0x74,0xde,0x91,0x18,0xdf,0xc5,0x22,0x93,0x17,0xf1,0xad,0xbe,0x75,0xbc,0x9d,0xff,0x3c,0x5e,0xfe,0xc4,0x57,0x6c,
        0xb5,0x9b,0x71,0x5f,0x8c,0xfd,0xf7,0xc5,0xd8,0xf2,0x15,0x8a,0x63,0xa1,0xd8,0x1a,0x6f,0xcc,0x4d,0x8f,0x6d,0x76,0x2c,0x25,
        0x0e,0x54,0xcf,0xda,0x3e,0x2a,0x8f,0xcd,0x7a,0xac,0x4a,0x6d,0xff,0x15,0x2c,0xf6,0xf4,0x3c,0x4d,0xc7,0xaa,0xa7,0x6c,0xb4,
        0x88,0x2f,0x27,0x6f,0x11,0x66,0xd3,0xce,0xda,0xd8,0x97,0xe6,0xdf,0x17,0xe3,0x3c,0xea,0xc2,0xc6,0xd1,0x3e,0x2c,0xd2,0xe5,
        0x9f,0x72,0xfe,0x62,0x9c,0xa9,0xad,0xc9,0xd3,0xe5,0x17,0xda,0x2f,0xc4,0xc0,0xbe,0x10,0xbb,0xf1,0x0b,0xb1,0xe3,0x72,0x1e,
        0x16,0x4c,0xcb,0x53,0x36,0xe6,0xa6,0x05,0x37,0x3b,0x96,0x12,0xbd,0xd6,0xb3,0x7d,0x66,0x4b,0x03,0xfa,0x0b,0x71,0xfe,0xc5,
        0xae,0x9a,0xeb,0xa3,0x85,0x1e,0xab,0xe5,0xe6,0x42,0xb4,0x9c,0xa0,0x72,0xec,0x26,0x5d,0x39,0x18,0x0a,0xb6,0x98,0x7f,0x35,
        0x25,0xb4,0xa0,0x45,0x63,0x35,0xd5,0xec,0xaa,0x38,0xe5,0xbc,0x23,0x4d,0x75,0xd9,0x05,0x59,0x28,0x9f,0x8f,0x11,0x45,0xc2,
        0xe7,0x63,0xd3,0x7d,0x3e,0xbe,0xd7,0xf3,0xf9,0x58,0x89,0xad,0xb8,0x09,0x65,0xb3,0x63,0xb8,0xa6,0xda,0x39,0xb3,0x9d,0xf3,
        0x7d,0x34,0xa3,0xda,0x8d,0xf3,0xdc,0x4c,0x8b,0xe5,0x74,0x4d,0xc7,0xaa,0xc5,0x46,0x1a,0x6f,0xa9,0x56,0xc4,0x30,0x37,0x96,
        0x2e,0x36,0xa5,0xb1,0x41,0x96,0xe6,0xdf,0xe7,0xe3,0x3c,0xea,0xc2,0x96,0xf5,0xbc,0xe6,0x75,0x6c,0xa5,0x8d,0x27,0x49,0x16,
        0xd1,0x47,0xe3,0xfc,0xfb,0x68,0xec,0x80,0x8f,0x56,0x5e,0x7f,0xf9,0x68,0x4c,0x70,0x08,0x56,0xf5,0xf4,0x01,0xab,0xb4,0x71,
        0xe5,0x7b,0xcd,0xa5,0x63,0x4b,0xac,0x5a,0x85,0x3f,0xba,0xf6,0x7d,0x8b,0xf9,0xa1,0x31,0xa3,0x1c,0x36,0xf3,0xd4,0xe6,0xb9,
        0x22,0xf9,0x5a,0x36,0x7a,0xfe,0xfb,0x68,0x9c,0x47,0x5d,0x68,0x0a,0x3a,0xb6,0x60,0x4f,0xef,0x38,0x54,0xf9,0x70,0x08,0x7f,
        0x94,0x75,0x0e,0x7a,0xe3,0x34,0xe3,0xfe,0x8a,0x9d,0x6d,0x69,0xea,0xb1,0xec,0x49,0x3f,0xe3,0xd9,0xa6,0x35,0x1b,0x6d,0x5c,
        0x89,0x92,0x50,0x24,0xcf,0x3f,0x65,0xf2,0x47,0x71,0xb4,0x76,0x61,0x12,0x33,0x2b,0xd9,0xe1,0xed,0xdd,0x6d,0x21,0xeb,0x9c,
        0xcf,0xc5,0x3e,0xfa,0xdc,0xcd,0x1a,0x59,0xbb,0xd5,0x44,0xfb,0x33,0x93,0x60,0xdb,0x30,0x5b,0x71,0xfa,0xb9,0x38,0xff,0xea,
        0x76,0x6e,0xcb,0x7c,0x3d,0xb7,0xec,0x95,0xd1,0x6c,0xab,0x6d,0xb1,0x4b,0x57,0x99,0xcf,0xc5,0x79,0xd4,0x85,0xf6,0x31,0x24,
        0x31,0x3b,0x76,0x48,0xf6,0xe7,0x0f,0xb3,0x30,0x1f,0x16,0x5c,0xd6,0x31,0x8d,0xa7,0x4c,0x1f,0x56,0xb8,0x57,0xb7,0x73,0x52,
        0x39,0xf6,0x64,0x3d,0xf7,0xe6,0x4b,0xb8,0x7c,0x52,0xd9,0xaa,0x92,0xdd,0xcd,0xfc,0x61,0xbc,0xc5,0xf9,0xc3,0xb8,0x18,0x2f,
        0x9f,0x52,0xcb,0x55,0xc1,0x79,0x47,0x62,0x66,0x17,0x1d,0x99,0x54,0x42,0x09,0x0d,0x5b,0xaf,0xc2,0x24,0xe3,0xc9,0xa3,0x82,
        0xb7,0xaa,0x5b,0xc3,0xe2,0xcc,0xb3,0x92,0xb2,0xff,0x36,0xa6,0xda,0xfd,0x77,0x92,0xe6,0xbb,0xed,0xf9,0xe1,0xb6,0x0b,0x26,
        0xd7,0xdd,0xd8,0xf2,0xcc,0x69,0xdd,0x0c,0x68,0x8c,0xaf,0xd7,0xea,0xee,0xad,0x5d,0xd2,0xa5,0x6c,0x82,0x72,0x16,0x4f,0x5a,
        0x2f,0x41,0xb5,0x13,0xbd,0x05,0x95,0x29,0x7a,0xbd,0xb3,0x76,0xc1,0xad,0x2e,0xd4,0xa1,0x48,0x77,0x59,0xcb,0xbd,0xca,0xba,
        0x9b,0xc4,0x15,0xe8,0xe0,0x37,0x9e,0x3f,0x1b,0xaf,0xf6,0x68,0xc9,0xd9,0x8c,0xd3,0x9f,0x8d,0xb7,0x69,0x75,0xb4,0x38,0xa5,
        0x45,0x82,0xa6,0xf2,0x8a,0x2a,0x5d,0x15,0x9c,0x77,0x64,0x52,0x30,0x2d,0x09,0x0d,0xf4,0xd5,0x58,0x4d,0xe9,0x56,0x36,0xcf,
        0x52,0x9e,0x73,0xac,0x2a,0xb6,0x25,0xed,0x2a,0xd1,0xa2,0xf2,0x1d,0x69,0xd9,0xef,0xed,0x02,0xab,0x69,0xba,0x7e,0xc6,0xcb,
        0xfa,0xf2,0x2e,0x4b,0xb2,0x61,0xf3,0x6b,0xf1,0xc7,0xf0,0xee,0x3a,0x37,0x97,0xb3,0xb9,0xad,0x35,0x8f,0x26,0xcb,0x67,0xc6,
        0x9d,0x27,0x68,0x49,0x88,0x4b,0xd7,0xaf,0xc5,0x2f,0x33,0x2d,0x9f,0x72,0x39,0x0a,0x1f,0x1f,0x8c,0xc7,0x40,0x92,0x7c,0x28,
        0xe0,0x2e,0xf2,0x41,0x60,0x0f,0x64,0x97,0xc1,0xfb,0x01,0x77,0x95,0x9b,0xbb,0xca,0x75,0xa7,0xb4,0x38,0xb3,0x45,0x09,0xbb,
        0x2f,0xef,0x6e,0xa1,0x3d,0x8d,0xeb,0xce,0x2d,0x81,0x23,0x1c,0x9f,0x48,0x9f,0xa5,0xf7,0xdc,0xee,0xc7,0x77,0x5b,0xe6,0x88,
        0x67,0xe6,0xaf,0xcc,0xde,0x8f,0xaf,0x64,0x6c,0xc3,0x45,0x12,0xac,0x7a,0x91,0x6b,0xcf,0x2d,0x3f,0xb9,0x1f,0xbf,0x2e,0xd5,
        0x85,0x8d,0x3e,0x5f,0xf0,0xd3,0xf1,0x35,0x40,0xa0,0x4f,0xea,0xc6,0xd9,0x49,0x0b,0xaa,0xe9,0x26,0x61,0xf2,0x4e,0x18,0x3f,
        0xce,0xfe,0xff,0x9d,0xf8,0x34,0x08,0x00,0x00,0x00,0x00,0x00,0x70,0x6c,0x5c,0x46,0xa6,0x07,0xaa,0xc4,0x24,0x5c,0xfc,0x41,
        0x18,0xe7,0x3f,0xe1,0xfd,0x07,0xf1,0x45,0xca,0xdb,0xc4,0xab,0x30,0xf9,0x83,0xf8,0xc5,0x9c,0x8c,0xb7,0x56,0xbd,0xfe,0xf9,
        0x9f,0x62,0x02,0xdc,0x79,0xaa,0x3f,0x58,0x5f,0x12,0x8f,0x3d,0xa8,0x63,0xf3,0x12,0xb6,0xf9,0xfe,0xdf,0xef,0xc7,0xe4,0x87,
        0x60,0x18,0x4e,0x7f,0x3f,0xbe,0xd7,0x53,0x21,0x3f,0xd6,0xf5,0x17,0xeb,0x6e,0x0a,0xfa,0xbd,0xc3,0x91,0x45,0xf4,0x7b,0x31,
        0xb0,0x48,0xbe,0xf3,0xac,0xda,0xfd,0xdd,0xc3,0xfc,0xdd,0x58,0xde,0x21,0xc8,0x02,0xfb,0xdd,0x18,0xdf,0xef,0x2e,0x7d,0x53,
        0xb7,0x1f,0xb2,0x3f,0xbf,0x13,0xcb,0xeb,0x9d,0xa6,0x12,0x6e,0xde,0x92,0xfc,0x9d,0x18,0xdf,0xef,0xec,0x28,0xbe,0x0f,0x87,
        0xf0,0xd9,0x50,0x7c,0x3b,0x01,0xbb,0xe7,0xf4,0xb3,0xf1,0x8b,0x24,0x91,0x7c,0xe7,0xd9,0xe2,0xce,0x16,0x94,0xe9,0x36,0xa6,
        0x36,0xf9,0xc6,0xa5,0xaf,0x66,0x52,0xe4,0x39,0x2d,0x79,0x19,0x89,0x05,0x8d,0xca,0x9d,0xa1,0x72,0xca,0xdb,0x61,0xfc,0xd9,
        0xf8,0x1b,0x1a,0x73,0xcd,0x53,0x1d,0x9f,0x57,0xc5,0xb1,0xf3,0x8e,0x6c,0x32,0xdf,0x27,0x9f,0x8d,0x3f,0x12,0xd8,0x85,0x8d,
        0xca,0xfb,0x48,0xfc,0xf9,0xc2,0x8f,0xc4,0x7f,0x3e,0x30,0xdc,0x30,0x89,0x3b,0x27,0x71,0xeb,0x61,0xb1,0xf5,0xf8,0xe6,0xcc,
        0x39,0xca,0x9d,0xd9,0x3a,0xf1,0x91,0xb8,0x5c,0xcc,0x71,0x2f,0x9c,0x7d,0x24,0xfe,0x9b,0x51,0xf9,0x56,0xd6,0x20,0x1f,0x89,
        0xed,0xd2,0x85,0x8d,0xd6,0xcf,0x8f,0xc4,0x65,0xed,0xce,0x31,0x4a,0xa1,0x12,0x77,0x8c,0x6d,0xae,0xef,0xeb,0xf2,0xac,0x4e,
        0xa7,0x24,0xc2,0xdc,0x1b,0xd5,0xa0,0xdb,0xb5,0xe7,0xc1,0x5f,0x52,0xc0,0x3e,0xc8,0x3b,0x7c,0x10,0xfb,0xbd,0x03,0xd9,0xdf,
        0xdf,0x8e,0xc3,0xed,0xc8,0xb8,0xb9,0x83,0xff,0xed,0xb8,0x78,0x54,0xc8,0x4f,0x39,0x5b,0xc1,0xda,0x33,0x9b,0xe6,0xdf,0x8a,
        0xcc,0x9a,0xca,0x6b,0x2a,0xbd,0x60,0xcd,0xfd,0xcb,0x6f,0xaf,0xfa,0x40,0xde,0x93,0xe2,0xd8,0xb4,0xe4,0xaa,0x28,0xe1,0xbc,
        0x23,0xd9,0xbd,0xe2,0x7f,0x0c,0xc5,0x1d,0xf5,0xb1,0xb0,0xf4,0x3c,0xb6,0x59,0x82,0xee,0xcf,0x7f,0x7f,0x1e,0xb3,0x46,0x5b,
        0x8e,0xaf,0x3d,0xdb,0x8f,0x97,0xab,0x22,0xc1,0x79,0x47,0x62,0xb1,0xf5,0x8f,0x49,0x1f,0x0e,0x93,0x3f,0x2f,0x7e,0xbf,0xa0,
        0x20,0x3e,0x50,0x1d,0x33,0xe1,0xb7,0xe2,0x45,0x23,0x51,0xb2,0x6b,0xc0,0x6f,0xc5,0x4b,0x41,0x24,0xdf,0x79,0x36,0xa3,0xd8,
        0xb9,0x74,0x07,0xf8,0x5b,0xf1,0x49,0xff,0xb7,0xe2,0xe3,0xff,0x6f,0xc6,0x71,0x86,0xbd,0x92,0xbf,0xb4,0xf3,0x9b,0xf1,0xf5,
        0x9e,0xca,0x56,0x5b,0x66,0xb3,0xb1,0x36,0x97,0x72,0x6b,0xbb,0xac,0x57,0xe4,0xd2,0xc4,0xb6,0x25,0xd4,0x44,0x74,0x73,0x63,
        0xf6,0x9b,0xf1,0xf5,0x90,0xdf,0x8c,0xeb,0x52,0x17,0x36,0x5a,0x3f,0xf7,0x5a,0x5e,0x6c,0xc1,0xe9,0xa6,0x8c,0xda,0x27,0x58,
        0x6a,0xd6,0x41,0x5c,0x2e,0x76,0xcb,0xe9,0xbe,0x0a,0xba,0x66,0x17,0x2f,0xbe,0x23,0x1f,0x2f,0x9f,0x89,0x8d,0x5c,0x47,0xf8,
        0x4c,0xfc,0x89,0xc9,0xcf,0xc4,0xeb,0x11,0x3a,0x51,0x6d,0xd6,0xe6,0x96,0xaf,0x3d,0xb3,0xdc,0x6a,0xa2,0xe6,0xcc,0x6d,0xab,
        0xbb,0x6a,0xfd,0xfc,0x4c,0x5c,0x07,0xbb,0x30,0x29,0x98,0x96,0x84,0xca,0x56,0x2d,0xf1,0x94,0xcb,0x8e,0xb4,0x6a,0xf2,0xd5,
        0x74,0x2c,0xbd,0x1a,0xd1,0xcd,0xfc,0xfb,0xf5,0x58,0x42,0x24,0xfc,0x7a,0xec,0xb1,0x5f,0x8f,0xfd,0x70,0x2c,0x94,0x81,0x2d,
        0x46,0x5b,0x13,0xfb,0x8a,0x33,0x57,0x27,0xaf,0x16,0x74,0x55,0x70,0xde,0x91,0x8d,0xe2,0xfb,0x34,0x3e,0x1d,0x57,0xb7,0x4f,
        0xc7,0xae,0xfa,0x74,0xec,0xaa,0xb8,0x55,0xdd,0x59,0xa5,0x29,0x41,0x4d,0xba,0x6d,0xaf,0x71,0xb1,0x82,0xd3,0x55,0xfc,0x46,
        0x8c,0xe1,0xee,0x31,0x2c,0x38,0xdb,0x82,0xd3,0xdf,0x88,0x7d,0xb4,0x29,0xab,0x73,0xa9,0xed,0xb8,0x49,0x71,0xe6,0xb4,0x89,
        0xab,0x82,0xf3,0x8e,0xac,0x1a,0x36,0xcb,0x6d,0x96,0x25,0xf8,0x0f,0x31,0x5d,0x17,0x42,0xc1,0xf8,0xe8,0x18,0xb5,0x60,0x52,
        0x30,0x2d,0xc9,0x76,0xbe,0x1b,0xbf,0x9b,0xbb,0x15,0xdb,0xac,0x05,0xbf,0x76,0xf0,0x9f,0xc2,0xbc,0x2d,0xb4,0x6b,0xba,0x5f,
        0x8f,0xdd,0xb8,0x19,0x1d,0x9e,0x57,0x96,0x9f,0x0d,0x43,0x71,0x6c,0xba,0xd3,0x87,0xd0,0xbc,0x84,0xb3,0x8d,0xd3,0xc5,0x9a,
        0x8d,0x6b,0x69,0x71,0x4a,0x8b,0x04,0xb5,0x35,0x2b,0x8f,0xb5,0x9b,0x0f,0x17,0x71,0x3a,0x21,0x3c,0x5c,0xcf,0xbb,0xb3,0x04,
        0x23,0xf4,0xce,0xc3,0xf5,0x74,0x4f,0xbe,0xab,0xd1,0xf3,0xee,0xce,0x4b,0x48,0x93,0xc3,0x0f,0x9b,0x03,0x8e,0xd6,0x7d,0xbf,
        0x0c,0x8a,0x7e,0x89,0xbd,0x39,0xac,0x0c,0x68,0x5b,0xb6,0x6c,0xd9,0xea,0x42,0x4a,0x31,0xd8,0xb2,0xd5,0xb0,0xd5,0xe2,0xa9,
        0x67,0x15,0xc3,0x2d,0x69,0x53,0xc2,0xd2,0x73,0x6a,0xcb,0x12,0xea,0xd2,0xa5,0x18,0xdf,0xdd,0x6a,0xcf,0x6a,0xb1,0xed,0xb7,
        0xd2,0x6d,0xcf,0x8e,0x2d,0xd1,0xcb,0xf8,0xbc,0xb7,0x25,0x77,0xba,0xbc,0x86,0x96,0x6f,0x1a,0x13,0x59,0xd6,0x93,0x8c,0xe7,
        0xb3,0x04,0xe5,0x56,0x95,0x5d,0x94,0xb7,0xb6,0x84,0xd6,0x23,0xab,0x71,0x86,0x6f,0x5c,0x5e,0xe7,0xf9,0x57,0x53,0x5e,0x8b,
        0xd9,0xb1,0x6d,0x7b,0x3e,0xac,0x2b,0x68,0x92,0x5d,0x05,0xc3,0x78,0xd2,0x9e,0x7b,0x0d,0xc7,0xce,0xc2,0x59,0x28,0xbe,0xea,
        0x77,0x96,0x3d,0x5a,0x3e,0x3a,0xcf,0xb9,0xf7,0x68,0x58,0x14,0x94,0x31,0xc8,0x99,0x0c,0xf3,0x2f,0x13,0x3e,0xbe,0x57,0x7c,
        0xa7,0xf0,0xf1,0xa4,0x24,0xee,0xbc,0x88,0x3c,0xb8,0xa1,0xf8,0x5d,0x8e,0xe2,0xdb,0x87,0xef,0x14,0x3b,0x5f,0x44,0x3e,0xf8,
        0xaa,0xd8,0xf9,0xe4,0x9d,0x30,0xb7,0xb3,0xfe,0x58,0x99,0xcb,0x75,0x66,0xc5,0x29,0xd7,0x79,0x3e,0x1b,0xd6,0x10,0xeb,0x32,
        0x1d,0x14,0x11,0x3d,0x1c,0x84,0xf3,0x8c,0xf7,0x0c,0xb3,0x06,0x2c,0x62,0x78,0x6f,0xe4,0x7d,0x91,0xf7,0x44,0x1e,0xe5,0xdc,
        0x0b,0xf7,0xf3,0x26,0x78,0x3d,0x1c,0xe7,0x05,0xbd,0x0e,0x91,0x6c,0x2b,0xdc,0x10,0xca,0xad,0x49,0xdc,0x9a,0xe7,0x3a,0xcc,
        0x32,0xa2,0x82,0x0f,0x96,0xb5,0x7e,0x99,0x6d,0x0d,0x46,0x05,0x61,0x21,0x86,0x0f,0x54,0x28,0x76,0xde,0xb4,0xd9,0xf3,0xa2,
        0x4a,0x4f,0xcb,0x06,0x79,0x54,0xf0,0xd6,0xcb,0x70,0x93,0xe7,0x2c,0xeb,0x50,0xa9,0x44,0x4b,0x66,0x05,0x3d,0xad,0x1c,0xbb,
        0xae,0xcb,0xa4,0x92,0x60,0xd2,0xc0,0x8a,0x53,0x8a,0x21,0x75,0x1a,0x87,0x54,0x68,0xe0,0x4d,0xc6,0x20,0x7f,0x67,0xe4,0xfd,
        0xb1,0x3b,0x1e,0x47,0x9e,0x44,0xde,0x2a,0x47,0x64,0x4b,0x42,0xc8,0xa6,0x52,0xf1,0x93,0xd5,0x17,0x37,0xff,0x58,0x77,0xc9,
        0x83,0x61,0x79,0xac,0x05,0xa3,0xd8,0x2e,0xa3,0x9b,0xe6,0x59,0xe8,0x87,0x07,0xc3,0xa2,0x84,0x32,0x41,0x99,0x75,0x2d,0xfd,
        0x94,0x57,0x79,0xbf,0x63,0x27,0xaf,0x08,0xd6,0xbe,0xbf,0xf2,0x76,0x31,0xe6,0x9f,0xb5,0xe7,0x75,0x75,0xe7,0xbd,0xba,0x33,
        0x87,0x1b,0xe6,0xd9,0xa6,0xa0,0xd6,0xa7,0xbc,0x5e,0xbd,0x35,0xd8,0x3d,0x3b,0x7e,0x45,0x70,0x10,0x3f,0xd8,0x5a,0xcb,0xc2,
        0x29,0xf7,0x6a,0x8e,0xf5,0xc6,0xde,0x3e,0xff,0x19,0x17,0xb2,0xcb,0x19,0xf9,0xce,0xe1,0xeb,0xe1,0xfc,0xce,0x07,0xc3,0xc5,
        0x53,0x06,0xc5,0xc5,0x3b,0xdf,0x9a,0x9d,0x99,0xa7,0x3b,0x29,0x93,0xd7,0x71,0x12,0xd3,0xd5,0x9f,0x92,0xad,0x05,0xf1,0xd8,
        0x7c,0x25,0xde,0x5e,0x28,0xb6,0x5c,0x69,0x57,0x96,0x70,0xc3,0x62,0xba,0x9b,0x18,0x0a,0x46,0x0b,0xd1,0x5e,0x6f,0x0d,0xaf,
        0xef,0x4a,0x6a,0x8e,0xc5,0x67,0xe6,0xb8,0x66,0x2d,0x95,0x37,0x6e,0x6a,0x82,0xa5,0x04,0x65,0x4b,0x8c,0x57,0x33,0x3a,0x59,
        0xee,0x95,0x41,0xa5,0x05,0x07,0x31,0xb3,0xb2,0x1f,0x6e,0x96,0xbc,0x15,0xcc,0x25,0x38,0x59,0x41,0xb5,0x91,0x17,0x4a,0xbf,
        0x98,0xd5,0xba,0xdc,0x79,0xb1,0x40,0xd1,0xba,0x17,0x0f,0x4a,0x6a,0x4f,0xa9,0xa3,0x4c,0x10,0x86,0x35,0xb9,0xcc,0x4e,0x09,
        0x17,0x83,0xf1,0x83,0x5e,0xc9,0x23,0x5a,0xc5,0xee,0x4b,0x68,0xcd,0x30,0x72,0xd2,0x5c,0xdd,0x59,0x63,0x85,0xeb,0xfe,0x2b,
        0xbe,0x0c,0xba,0xea,0xfb,0xbe,0xf1,0xe6,0xf2,0xe9,0x6a,0x46,0x25,0x97,0x4d,0xd4,0x8d,0xb3,0xcb,0x06,0x6a,0x4f,0x69,0xbd,
        0x73,0x54,0x61,0x83,0x12,0x36,0x61,0x58,0xa1,0x63,0x66,0xfd,0x53,0xdf,0xd6,0x65,0xc7,0xd5,0x6e,0x1d,0x8a,0x5d,0xb7,0xe0,
        0x9a,0xb5,0x6e,0x43,0x2a,0x6d,0xb6,0x30,0x1f,0x9a,0x12,0xd4,0x9e,0xd2,0x22,0x5d,0x75,0xeb,0xed,0x78,0xc3,0x1a,0xef,0x6d,
        0xda,0xe4,0xd2,0x74,0xca,0x5c,0xcd,0x06,0x9b,0xd4,0xb3,0xa9,0x09,0x9a,0x68,0x5f,0xdd,0xb9,0x33,0x07,0xad,0x2a,0xd8,0xbe,
        0xc9,0xfb,0xe9,0xf7,0xf6,0xc7,0xb6,0x3d,0xb3,0x63,0x3d,0xd7,0x75,0xf8,0x46,0x67,0xee,0xbe,0x75,0x77,0x1f,0xed,0xee,0x2b,
        0xb1,0xb7,0x0e,0x4f,0xa2,0xe5,0xef,0x48,0xad,0xd3,0xe5,0xd9,0xe1,0x4a,0x5f,0x7a,0x5a,0x6a,0x41,0xf5,0xc5,0x95,0xea,0x56,
        0xfb,0x5c,0x16,0x1f,0x2d,0xdb,0xf2,0x62,0x54,0x3c,0xc7,0x95,0xac,0x4b,0xb0,0x18,0xf4,0xfc,0xfd,0x44,0x63,0x41,0xa3,0xcb,
        0x57,0x59,0x09,0xc3,0x2c,0xb0,0x82,0xb7,0x2a,0x79,0x76,0x27,0xe0,0xd8,0x99,0xeb,0xf0,0xad,0xa7,0x5a,0xd3,0xf8,0xec,0x3f,
        0xcf,0x58,0xf9,0xf6,0x09,0x0e,0xbe,0x70,0x6e,0x4a,0x1f,0xd3,0xb7,0xff,0x26,0x5f,0xd7,0xc8,0xf3,0x1d,0xbe,0x8b,0x4a,0xec,
        0xea,0x6a,0x51,0xae,0xd0,0xb5,0x84,0x1a,0xae,0x57,0xf6,0x50,0xa1,0xdd,0x3a,0x5f,0x9f,0xe7,0x61,0x7a,0x2c,0x79,0x56,0x0d,
        0xa2,0xa6,0x17,0x8f,0x12,0xa4,0xd7,0x76,0x49,0x22,0xa2,0xe5,0xd7,0xeb,0xe6,0x5e,0x5a,0xc5,0x8c,0x93,0x84,0x28,0x2e,0x98,
        0x88,0x8c,0x52,0xa8,0x44,0x62,0x84,0x24,0xa6,0xcc,0x3c,0x73,0xef,0xb6,0xcc,0x5e,0x3a,0x06,0xd6,0xdc,0xfd,0x26,0x51,0xc1,
        0x43,0xd1,0xdb,0xf3,0xc3,0xe1,0xda,0xb3,0x7d,0xe9,0xfd,0xc5,0x37,0x98,0x51,0xde,0x67,0x55,0x77,0xb6,0x60,0xdb,0xdb,0xfa,
        0xda,0xd2,0xe7,0x8f,0x8d,0xca,0x63,0x8d,0x4f,0x36,0xbb,0xc8,0x65,0x75,0x66,0xad,0xd2,0xd5,0x9e,0xb9,0x7b,0x76,0x58,0x97,
        0x5e,0x72,0x79,0x5a,0xc3,0x68,0xc6,0xa8,0x42,0x3c,0x65,0xe1,0x41,0xb3,0x42,0x39,0xdc,0x2a,0x5d,0xbc,0x30,0x14,0xcb,0x63,
        0x35,0xe3,0xb3,0x36,0x79,0xd3,0x99,0x8b,0xaf,0x6a,0x3e,0x08,0xf3,0x54,0xab,0x94,0x6f,0x85,0x1b,0x86,0x05,0xef,0xaf,0x6c,
        0x95,0xc7,0x9a,0x6a,0x96,0x20,0x4d,0xfd,0xd0,0xd4,0xba,0xed,0xbb,0xe3,0x9a,0xa6,0x17,0x26,0xb6,0x65,0x07,0x0d,0x52,0x3b,
        0x7a,0xaa,0x3b,0x6b,0x8f,0xbd,0x58,0x7f,0x4a,0xdd,0xce,0xa7,0xfb,0xeb,0xe9,0xad,0xda,0xac,0x9f,0x85,0x6c,0xd5,0x60,0xd8,
        0x5d,0x8f,0x75,0x2d,0x61,0x61,0x9c,0x85,0x83,0xcc,0xcd,0xca,0xf2,0x54,0x0e,0xb0,0xa6,0xf5,0xb3,0x3e,0x41,0xc3,0xf8,0xac,
        0x52,0x9e,0x12,0xd6,0x27,0xbf,0xde,0x39,0x5a,0x9b,0x59,0x6d,0xb1,0x4d,0x3b,0xab,0xb4,0x5f,0x66,0x16,0x32,0x6b,0xa8,0x59,
        0x63,0x75,0x6b,0x5b,0x70,0xb4,0xba,0x9e,0xef,0x6f,0x28,0xaf,0xbe,0x1f,0x9a,0xfa,0x6f,0x79,0xec,0xce,0xe6,0x51,0xf5,0x94,
        0xf7,0x8f,0x2a,0x84,0x79,0x36,0x9c,0xda,0x4d,0x5b,0xed,0xa7,0xf6,0xc6,0xc7,0x96,0x68,0xf9,0x92,0xf3,0xb6,0x8b,0xce,0x36,
        0xb9,0x6c,0x75,0x87,0x54,0x5b,0xde,0x9a,0x4a,0x54,0x63,0x6f,0xd1,0x12,0x2b,0xaf,0x86,0xd9,0x7d,0x5d,0x43,0xf2,0x77,0x5a,
        0x53,0x26,0x78,0x32,0x47,0xf3,0xed,0x41,0xd3,0xfb,0xd3,0xfd,0x6c,0xc5,0xfb,0xc1,0xce,0xaf,0xc6,0x84,0x74,0x08,0x21,0x3e,
        0xb9,0x8d,0xe2,0x9d,0x63,0x95,0xd0,0xfd,0x69,0xb0,0x9f,0x5c,0xd6,0xd0,0x5f,0x75,0x37,0x28,0xa8,0xf1,0x4d,0xb3,0xe5,0x0f,
        0x47,0x1c,0xfa,0x2d,0xd0,0x9d,0xf0,0x6a,0x10,0xc3,0x1c,0xc4,0xb9,0xd2,0x7e,0xeb,0xc0,0xcd,0x13,0xfe,0x7d,0xd1,0xa9,0x00,
        0x00,0xdc,0x29,0x12,0xb8,0x75,0xb8,0x4b,0x8c,0x2a,0x24,0x51,0x25,0x24,0xc6,0xc2,0x63,0x67,0xc3,0xed,0xf2,0xbe,0xb7,0x92,
        0x68,0x1e,0x74,0x1e,0x52,0x77,0xaa,0x53,0xef,0x56,0xb4,0x00,0x00,0xb4,0xe5,0x72,0x2d,0xa3,0x3a,0x5a,0xa4,0x5b,0x4f,0x97,
        0xf7,0xd5,0xea,0xde,0x70,0x3a,0x3c,0x87,0xee,0xcd,0x16,0x5d,0xbc,0xfd,0xd0,0x58,0xdf,0xd6,0x0f,0xe6,0xde,0x7c,0x6e,0xf8,
        0xac,0x58,0x3f,0xa3,0x67,0x9f,0x2c,0xff,0x72,0xd2,0xc1,0xab,0xb4,0xfb,0x68,0x2b,0xe3,0xfa,0xb8,0x49,0x62,0xf1,0xd8,0x62,
        0xb5,0x49,0xa0,0x12,0xdb,0xd6,0x3a,0x81,0x7e,0x08,0x0f,0xe2,0x4f,0x73,0xa1,0x3f,0x4e,0xd0,0x1f,0xab,0x7e,0xbb,0x7d,0x58,
        0x77,0xac,0x29,0x41,0xfb,0xcc,0x9a,0x12,0x34,0xed,0x6c,0xa2,0xfa,0x6b,0x99,0x73,0x05,0x7d,0x6a,0xc6,0x89,0x2d,0x5b,0xb7,
        0x6d,0x2b,0xff,0xf1,0xc4,0x4f,0xc5,0x2b,0xd7,0x7a,0x56,0x67,0x36,0x0c,0xa7,0x9f,0x8a,0xbf,0xff,0x39,0x47,0xf5,0x94,0x16,
        0x3f,0xe7,0xf9,0xa4,0x28,0x28,0xff,0x0d,0xe1,0x65,0x62,0x25,0x2e,0x9e,0x2c,0xec,0xcc,0xff,0x19,0xc7,0x13,0xcc,0xfd,0xac,
        0xf8,0xdc,0xd6,0xfa,0x7f,0x4a,0x74,0xf9,0x57,0xd7,0x7f,0x35,0xfe,0xe3,0x8c,0x38,0x24,0x5b,0xfc,0x10,0x6e,0x9e,0x2e,0xff,
        0x50,0xdf,0xaf,0x76,0xf8,0xe7,0x49,0xdb,0x93,0xcd,0xf0,0x5f,0x8d,0x33,0xfc,0x7e,0xac,0x75,0xa2,0x74,0xfe,0x7d,0xe1,0xed,
        0xca,0x93,0x2e,0x8d,0x74,0xd9,0x30,0x7d,0x37,0x5e,0x8f,0x8e,0x93,0xf8,0x2f,0x0d,0x4c,0x2b,0x5c,0xff,0xf3,0x03,0x75,0x5b,
        0xd3,0xf5,0x09,0x16,0x76,0x36,0xfd,0xf2,0x7f,0xeb,0x7f,0x15,0xe0,0xf1,0xab,0x9a,0x63,0x8f,0x43,0x5d,0x79,0xa1,0x92,0xbc,
        0x7a,0x6c,0xdb,0xb7,0xfd,0x9b,0xea,0xf9,0x64,0xbe,0xbc,0x51,0x8b,0x7f,0xd9,0x20,0xac,0xe7,0x22,0xc6,0x57,0xd2,0xa2,0x79,
        0xca,0x9d,0x65,0xad,0x7b,0x69,0xf2,0xd5,0x95,0x5f,0x7c,0x25,0xe3,0xba,0x09,0x9e,0x54,0x2a,0xb1,0x65,0x23,0x3f,0xde,0xdb,
        0x4f,0xbf,0x27,0xc1,0x8b,0xd1,0x64,0x38,0x28,0xbe,0xfa,0x7a,0x9b,0x28,0x7f,0x9c,0x27,0x89,0xba,0xe0,0xae,0xb0,0x8b,0xdf,
        0x97,0x4a,0x22,0x30,0x00,0x40,0x8f,0x24,0xf1,0xf3,0x85,0xdb,0x5c,0x8f,0x42,0x8c,0xa1,0x3f,0x9a,0xca,0x9b,0x9c,0x5c,0x64,
        0x4c,0xf2,0x1f,0x78,0x3e,0x19,0x4f,0x4b,0x62,0xba,0x92,0xcb,0x97,0xa3,0x90,0xb5,0xe7,0xe8,0xc5,0x28,0x4f,0x30,0x7e,0x51,
        0x6c,0x3d,0xa8,0xfe,0x30,0xf4,0xb6,0x5b,0xd5,0x7a,0x6e,0x9b,0xcb,0xcb,0x82,0xc7,0xd5,0x9d,0x8b,0x25,0x14,0xb5,0xde,0x96,
        0xe1,0x28,0xff,0x97,0xae,0xee,0xaf,0x20,0x7c,0x7c,0x30,0x3e,0x2b,0xc9,0xcb,0x1b,0x8e,0x1e,0xc7,0xc6,0x1a,0xd5,0x32,0xbc,
        0xde,0xaa,0x76,0x55,0x4d,0xb1,0x65,0xcb,0xbf,0xbc,0x6e,0xf9,0xfc,0x73,0x25,0xd7,0xbd,0xf2,0x22,0xcc,0xf7,0xed,0x7b,0x47,
        0xe1,0xde,0x49,0xd1,0x39,0xf7,0xb2,0x33,0xb7,0xdd,0xba,0x2d,0xdc,0xce,0x5a,0x6f,0x1b,0x5f,0xc7,0x1e,0xdb,0x36,0x41,0x49,
        0x75,0xb9,0xa8,0x0e,0xfd,0x97,0xc5,0x72,0xf1,0x7c,0x46,0x9e,0x7c,0x54,0x52,0x66,0x56,0xce,0xcd,0x2c,0xb3,0x47,0xb1,0x12,
        0x8f,0x62,0x09,0x5b,0x6d,0xdd,0x16,0x6e,0x67,0xad,0xb7,0x8d,0xaf,0x63,0x8f,0x6d,0x9b,0xa0,0x45,0xba,0x97,0xf1,0x3a,0x96,
        0x9d,0x99,0x8d,0xc1,0x7c,0x60,0x3e,0x1a,0xcc,0x28,0x4f,0x69,0x75,0x49,0x0e,0xbf,0x12,0xf3,0xbc,0x2b,0x9c,0x6c,0x9f,0xfc,
        0xa6,0xcd,0x3e,0x1e,0x77,0x02,0x29,0x70,0x33,0x30,0xcb,0xdb,0xad,0x1b,0x2e,0x1a,0xb8,0xec,0x4e,0x1a,0x73,0x7a,0x67,0xf4,
        0xff,0xbc,0x72,0x5b,0xd8,0xc5,0xf3,0x5f,0xf5,0xe2,0xb6,0x7b,0xe2,0xec,0xb8,0x3c,0x0e,0x26,0x73,0x54,0xe6,0x7b,0xd3,0x0c,
        0x6f,0x4f,0x0f,0x6b,0x41,0x32,0xc4,0x1f,0x66,0xbf,0xd8,0x98,0xca,0x0c,0xb8,0xd8,0x07,0x49,0x8c,0xac,0x1e,0x07,0xe6,0x8c,
        0x96,0x57,0xa0,0x14,0xc6,0xe7,0xa8,0x61,0x2b,0x51,0xf6,0x37,0x3e,0x77,0x54,0xf9,0x55,0xe4,0x8f,0x2d,0xe8,0x83,0x70,0x12,
        0xb7,0x80,0x3d,0x90,0xc2,0xaa,0xb8,0xf7,0xc5,0xaa,0x9c,0x63,0x8d,0x24,0x10,0x03,0xb0,0x62,0x40,0x27,0x51,0xa5,0x1d,0x86,
        0x99,0xbf,0xc5,0x34,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8e,
        0x92,0x10,0x26,0x61,0x30,0xbe,0xc4,0x2d,0x25,0x89,0x41,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x79,0xfe,0x5d,0x62,0xbc,0x2c,0x18,0x25,0x51,0x17,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x0e,0xcd,0x9b,0x10,0xc2,0x60,0x3c,0xa8,0xf2,0x62,0x34,0xc9,0x8e,0x5d,0xe2,0x50,0xd4,0xf6,
        0x4a,0x7b,0xb2,0xbf,0x1f,0x0c,0xe3,0x31,0x6a,0x19,0xb5,0x60,0x52,0x30,0x2d,0xc9,0x76,0xfe,0xdb,0x98,0x1c,0x00,0x00,0x74,
        0x27,0xbb,0xd0,0x4e,0x32,0xa6,0x4f,0x22,0xe5,0xd6,0xbe,0xeb,0x72,0x55,0x70,0xde,0x91,0x6a,0x9e,0xf9,0xad,0x58,0x88,0x77,
        0x64,0x07,0x27,0xfb,0xf3,0x6e,0x18,0x0f,0x23,0x27,0xb6,0xb6,0xdd,0xca,0x36,0x4f,0x33,0xee,0x47,0x4e,0xab,0x84,0x30,0xc9,
        0x98,0x56,0x4e,0xd9,0x9a,0x32,0xb3,0xc8,0x28,0x85,0xd8,0xfb,0xd9,0x2a,0x43,0x19,0x2d,0x46,0x1b,0x9e,0xc5,0x99,0x83,0xbb,
        0xc0,0x83,0x8e,0x64,0xc3,0xe6,0x59,0x1c,0x3d,0xcf,0x66,0xa3,0x27,0x7f,0x56,0x5b,0xc7,0xf6,0xeb,0x67,0x96,0xc1,0xf8,0x79,
        0xea,0x8c,0xba,0x71,0x93,0x59,0xb8,0x88,0xf3,0xf6,0x22,0xce,0xdb,0x1d,0x6d,0x9d,0xec,0xbc,0x04,0x5b,0xbb,0xdc,0x1a,0x35,
        0xf4,0x66,0x0b,0x5a,0x14,0xd4,0xbd,0x9e,0xe5,0xf0,0x0e,0xe5,0xb8,0x6e,0x37,0xbc,0xc7,0x47,0x30,0x76,0x9b,0xd8,0xf6,0xcc,
        0x03,0xd5,0x7a,0x21,0x41,0xbb,0xfb,0xdd,0xe5,0x04,0xb5,0x5b,0x1d,0x0b,0x4a,0x82,0x30,0x3f,0x1b,0xab,0xd4,0x8e,0xeb,0xea,
        0x29,0xd5,0x63,0x2b,0xf9,0xe5,0x1b,0xb2,0xcb,0xee,0x2f,0xc7,0xcb,0xee,0x2f,0x6f,0x74,0xac,0xfd,0x29,0x49,0x30,0x2c,0x38,
        0x9b,0x51,0xd4,0x7a,0x46,0xb1,0x33,0x5b,0x5f,0x32,0x2e,0x6b,0x29,0xcf,0x5c,0xd5,0x63,0x19,0x17,0xad,0xb9,0x2c,0x79,0x1d,
        0xb2,0xca,0x8d,0x9f,0x85,0x82,0xcb,0x8d,0x46,0xc8,0xbf,0x89,0x81,0x61,0xd7,0x3c,0x0c,0x93,0x8c,0x69,0xc9,0x93,0x0a,0xdb,
        0xce,0xf0,0x6c,0xb8,0x7d,0x32,0x0e,0xc5,0x4f,0x16,0xa3,0x2e,0x67,0x8c,0x9d,0x50,0x6d,0xeb,0x7d,0x70,0xfa,0xc9,0xb8,0xb0,
        0x7c,0x72,0xd5,0x72,0xd1,0x82,0x49,0x91,0xd9,0xb4,0xe4,0xaa,0xe0,0xbc,0x23,0x9b,0x8c,0xcf,0xd3,0x5f,0x8a,0x31,0x1c,0x82,
        0xac,0x0a,0xbf,0x14,0xfb,0xaf,0x1d,0xd3,0x3a,0xb2,0x79,0x3b,0xbf,0x55,0x52,0x96,0x90,0x35,0xc8,0x2f,0xc6,0x76,0xe9,0x42,
        0x28,0x18,0xdf,0x6a,0xb2,0x06,0xf9,0xc5,0xd8,0xf2,0x9b,0x31,0x29,0x06,0x4a,0x3e,0x3e,0x7f,0x31,0x36,0xf2,0x2f,0xc4,0x3c,
        0x13,0x25,0xab,0xf5,0x2f,0xc4,0xca,0xf7,0x42,0x9e,0xe7,0x59,0x49,0xdf,0xb7,0x7e,0x59,0x9e,0xd3,0xcb,0x51,0xf8,0xf8,0x60,
        0x3c,0x3e,0x06,0x6a,0x3f,0xcf,0x90,0x35,0xe4,0xcf,0xc7,0xf6,0xac,0x10,0x7e,0x3e,0x5e,0x0d,0x7f,0x3e,0x76,0x1c,0x3a,0x51,
        0x6d,0xd6,0x19,0xb5,0x2d,0xdf,0xe2,0xcc,0xea,0xb1,0xb9,0xae,0x9a,0xb6,0x60,0x52,0x47,0x53,0x82,0xca,0x8d,0xf5,0xe8,0xaa,
        0xd8,0x79,0xde,0x91,0x35,0x13,0x6f,0xae,0xf4,0x0f,0x87,0xf0,0xb3,0x59,0x0b,0x36,0x73,0x9a,0x71,0x3f,0x6e,0x8d,0x5a,0x25,
        0x58,0xcd,0x55,0xc1,0x79,0x47,0x5a,0x14,0x74,0x13,0xed,0xcf,0xc5,0x81,0x82,0x1a,0xb2,0x31,0xff,0x73,0x71,0xe8,0xaf,0x27,
        0x4f,0x70,0x36,0xa3,0xed,0xb1,0xee,0x97,0x89,0x95,0xb5,0xee,0x1c,0x74,0x51,0xdd,0x6a,0x79,0x57,0xc5,0xce,0xf3,0x8e,0xd4,
        0xc5,0x90,0xaf,0x05,0xcb,0x74,0x9c,0x4e,0x5b,0xb2,0xf1,0x2c,0xbe,0xac,0xa3,0x3c,0x76,0x56,0x59,0x20,0xce,0x96,0x92,0xff,
        0x0c,0xfa,0xe4,0xf4,0x67,0x62,0x5b,0x57,0xb6,0xaa,0xd4,0x26,0xa8,0x3d,0xd6,0x54,0x42,0x6d,0x41,0xc5,0x56,0x12,0xaf,0x1d,
        0xb6,0xa0,0x2e,0xbe,0x51,0x64,0x5a,0x92,0x9d,0xf7,0x53,0x71,0x65,0xe8,0x81,0x61,0x41,0xbe,0x00,0xfe,0x54,0x5c,0x00,0x23,
        0xe5,0xce,0x82,0x49,0xc1,0xb4,0xa4,0x92,0x7c,0x54,0x52,0xcd,0xa5,0xb7,0xbb,0xed,0xfb,0xb1,0xa0,0x75,0x1c,0x36,0xdd,0x0e,
        0x03,0xab,0x96,0xd7,0x21,0x5d,0x87,0x37,0x8e,0xee,0x1e,0x61,0xf6,0xb6,0x55,0xed,0xab,0xfc,0x2d,0xdf,0x88,0x7b,0xd8,0x7c,
        0xca,0xc2,0xbb,0xc0,0x6b,0xc6,0xcb,0x4f,0xc7,0x74,0x40,0x2b,0xfa,0xbb,0x3e,0xdc,0x16,0xaa,0x97,0xac,0xda,0xeb,0x58,0xd3,
        0xc5,0xad,0xe9,0x02,0x76,0x55,0x1c,0x3b,0xef,0x48,0xfb,0xeb,0xc3,0x5b,0xf1,0xb5,0x99,0x0f,0xdc,0x66,0xc4,0x90,0x06,0x62,
        0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,
        0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,
        0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,
        0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,
        0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,
        0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,
        0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,
        0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,
        0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,
        0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,
        0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,
        0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,
        0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,
        0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,
        0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,
        0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,
        0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,
        0x21,0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,
        0x0d,0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,
        0xc4,0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,
        0x90,0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,
        0x06,0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,
        0x62,0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0x88,0x21,0x0d,0xc4,0x90,0x06,0x62,
        0x48,0x03,0x31,0xa4,0x81,0x18,0xd2,0x40,0x0c,0x69,0x20,0x86,0x34,0x10,0x43,0x1a,0xbc,0x09,0x21,0x0c,0xc6,0x83,0x5e,0xf9,
        0x50,0x28,0xb2,0x4e,0x97,0xa6,0xca,0x7f,0x30,0x9e,0x82,0x7e,0x69,0x6a,0xf2,0xaf,0xcf,0x08,0xe3,0xf7,0x03,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x84,0x78,0x33,0xcc,0xbf,0x06,0xf8,
        0x3e,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0xe2,0xbc,0x18,0x85,0xe1,0x60,
        0x3c,0x06,0x00,0x00,0xc0,0x5d,0x22,0xbb,0x11,0x1c,0xdc,0x90,0x44,0x95,0x80,0x3d,0x8e,0xf9,0x3b,0x4b,0x08,0xe1,0x22,0x8c,
        0x4f,0xda,0xf3,0x26,0x33,0x8c,0x07,0xbd,0x92,0xfd,0xf9,0x44,0x2c,0x61,0x25,0x2d,0x72,0x99,0x86,0xc1,0x4f,0x86,0xf1,0x78,
        0x18,0x4e,0x33,0xee,0x97,0x5b,0x1d,0x09,0x61,0x94,0xf1,0xb4,0x9f,0xcc,0x6a,0x79,0x13,0x06,0x73,0x95,0x6f,0x7f,0x66,0x6d,
        0xba,0xf6,0x3b,0xb7,0x6f,0x90,0xe1,0x09,0x70,0x9b,0x78,0xbe,0x6f,0xde,0x84,0x61,0x5c,0x97,0xf6,0x44,0x0a,0x8d,0x0c,0x00,
        0x00,0x80,0x1d,0xf1,0x28,0x4c,0xb2,0x7b,0xbe,0xb7,0x5e,0x15,0x3c,0x09,0x61,0x92,0xed,0xbc,0xdc,0x23,0x09,0x34,0x01,0xb6,
        0xc6,0x83,0x43,0x1a,0x6c,0xf1,0x8c,0x57,0xcc,0xbf,0x37,0xe1,0x2c,0xee,0x3c,0x52,0x76,0xd1,0xd6,0xaf,0x0b,0x9e,0xd5,0xae,
        0x67,0xc3,0xe1,0xe9,0x20,0x8c,0xef,0xef,0x89,0x7b,0xf7,0x9f,0x8f,0x26,0x61,0x30,0x9e,0x3e,0xaa,0xf0,0xbc,0xc2,0x93,0x0a,
        0xaf,0x0a,0x9e,0xc4,0x33,0xdf,0x7a,0x16,0x77,0xe6,0xaf,0x78,0x0e,0xe2,0x0b,0x9f,0x6b,0x18,0x86,0x58,0xfa,0x4e,0x79,0xf7,
        0x86,0x17,0xb1,0x82,0x91,0xc7,0x15,0x3e,0x50,0x52,0x8d,0x68,0xf7,0x5b,0x95,0x2a,0x4d,0x97,0x8f,0x85,0x78,0x6c,0x05,0x1f,
        0xa8,0xf2,0x68,0x74,0x3f,0x9c,0xc4,0x63,0x27,0x79,0xff,0x15,0x5b,0x65,0x5b,0xbf,0x8c,0x25,0xbc,0xcc,0x76,0x2e,0x75,0x63,
        0x4c,0xfe,0x38,0x2f,0x7d,0x18,0x4b,0x1f,0xc6,0x7a,0xae,0xda,0x7a,0x31,0x1a,0xcf,0xd5,0x7a,0xc6,0x28,0xe3,0x69,0x4e,0x88,
        0xcd,0x1a,0xf2,0xad,0x79,0xca,0xba,0x94,0x09,0xde,0x2e,0x72,0x79,0xf6,0xba,0x92,0x59,0xd3,0x78,0x69,0xbf,0x74,0x25,0xb1,
        0x8c,0x02,0x68,0x47,0xf5,0x7e,0xe2,0xb4,0xb8,0xde,0x3e,0x2a,0xdf,0x3b,0x7b,0x72,0x52,0xac,0x67,0x2f,0x4e,0xe2,0x32,0x73,
        0x52,0x2c,0x17,0x6f,0x9f,0xc4,0xc5,0x23,0x6e,0xcd,0x8e,0xc5,0x65,0xf4,0x24,0x2e,0x56,0x27,0x71,0xb1,0xba,0x3e,0x56,0x2c,
        0x4f,0x8f,0x4e,0xe2,0x1a,0x79,0x52,0xac,0x52,0xb3,0xad,0xfb,0xf1,0x94,0x93,0x8c,0xf3,0x8e,0xbc,0xcc,0x32,0x1b,0x66,0x2b,
        0x6d,0x51,0x7a,0x51,0xa5,0x7c,0x01,0xac,0xb0,0x18,0x51,0xb8,0x18,0x8c,0x87,0xb7,0x8c,0xb2,0x91,0xcb,0x18,0x2a,0x6d,0xfd,
        0xfc,0x45,0xb1,0xf5,0x81,0xd9,0x56,0x9e,0xe0,0xe4,0x96,0x31,0x17,0xdf,0x30,0x1f,0x36,0xb3,0xf8,0xb2,0xad,0xea,0x00,0x7b,
        0x74,0xaf,0xb8,0x24,0xbf,0x2a,0x78,0x12,0xb7,0xae,0x77,0xbe,0x8c,0xbc,0xb8,0x17,0x1e,0x65,0xa3,0xfc,0xbd,0x9d,0xc9,0x32,
        0x1b,0x95,0x99,0x61,0x73,0x5e,0xc6,0x16,0x7c,0x39,0x6b,0xc8,0x8f,0xc7,0x63,0x47,0x46,0xed,0x47,0x10,0xb6,0x4d,0x9e,0x44,
        0xba,0xda,0x88,0xe2,0xe2,0xff,0xac,0xbc,0x14,0xc4,0xd5,0xe6,0x69,0x9c,0xa2,0xcf,0xe3,0xd6,0xe5,0xdb,0x95,0x53,0xae,0xef,
        0x23,0x8b,0x49,0x79,0x19,0x79,0xda,0x40,0x79,0xca,0xe5,0xac,0xd8,0xb3,0x78,0x37,0xba,0x1d,0xc3,0xe2,0x55,0xb1,0x47,0x55,
        0xde,0x13,0x8a,0x7a,0x3e,0x18,0x16,0x2c,0x3e,0x6f,0x86,0xff,0x52,0x5c,0x22,0x7b,0xe4,0xe6,0x53,0x2d,0xbd,0x67,0xbd,0x7b,
        0x52,0xaa,0x7c,0x6d,0x5d,0xf6,0x5d,0xc1,0xa5,0x8f,0x2a,0xed,0xb5,0xbc,0x63,0x27,0x89,0x71,0x06,0x1c,0x23,0x2d,0xe6,0xdf,
        0xa4,0x60,0x3a,0x29,0xae,0x94,0xd3,0x72,0x6b,0xcd,0x84,0xcd,0xb8,0xc0,0xe6,0x34,0xb4,0xe7,0xe5,0xf6,0x4c,0x1e,0x86,0xf1,
        0x03,0xec,0x9b,0x8b,0x06,0x2e,0x0f,0xc3,0x52,0x25,0xc2,0x1c,0xb3,0xe1,0x36,0x29,0x87,0x4d,0xef,0xac,0x18,0xde,0x93,0x87,
        0xab,0x8e,0x61,0xab,0xfb,0xa5,0x6e,0xcb,0xc5,0xdd,0xa5,0x7c,0xbf,0x2a,0x89,0x6e,0x04,0x10,0x8a,0xf5,0xec,0x5f,0xc7,0x1b,
        0xd6,0x5b,0xcf,0xaa,0xf8,0xfe,0x55,0x3c,0x05,0x3d,0xb2,0xb7,0xf1,0x59,0x5e,0x34,0x42,0x38,0x09,0xc5,0x9d,0xd5,0x6d,0x64,
        0xcd,0xfc,0xfb,0xdf,0xb1,0x59,0x8f,0x99,0x8b,0x5a,0xba,0x2f,0x5d,0xb5,0xe5,0xfd,0xcb,0x83,0xf1,0xa0,0x63,0xf2,0x76,0x41,
        0xff,0x65,0x4c,0x70,0xd7,0xc9,0x56,0x86,0xbf,0x8c,0x9f,0x7b,0x5a,0x49,0xbb,0xf6,0xfc,0x17,0x31,0xcf,0x63,0xa1,0x0c,0x6c,
        0x52,0x6c,0x4d,0xb3,0xff,0xff,0x5f,0xf1,0xd8,0xdd,0x63,0x4d,0xbf,0x6f,0x95,0xe7,0xa4,0xa0,0x6c,0xd6,0x07,0x4d,0x24,0x71,
        0x7b,0x97,0x3c,0xd9,0x9f,0x7f,0x1e,0xb2,0xb9,0x8c,0x3a,0x4e,0xb0,0xcc,0x3f,0x8b,0xb3,0x11,0x3d,0xf1,0x4f,0x8f,0x8b,0xea,
        0xfa,0x72,0x55,0xec,0x3c,0xef,0xc8,0x46,0xeb,0xd9,0x3f,0x89,0xd3,0xf7,0xee,0xb1,0xbe,0x5d,0x4e,0xee,0x24,0x6b,0xda,0xe5,
        0x1f,0xc7,0xb1,0x8b,0x79,0x3a,0xdc,0x4f,0xfc,0x44,0xcc,0x05,0xbb,0xa6,0xef,0x1b,0xc1,0x9f,0x88,0x37,0xd6,0xff,0x28,0xae,
        0x28,0x3d,0x70,0x93,0x75,0x7f,0x79,0xee,0xa2,0xbc,0xda,0x74,0x95,0x9d,0x27,0xfd,0xb0,0x8b,0x3c,0x77,0x51,0x5e,0x6d,0xba,
        0x24,0x9e,0x57,0xd0,0xfb,0xf3,0xdf,0x7f,0x8e,0x2b,0x0a,0x00,0x00,0xc0,0x9e,0x38,0xda,0xfb,0x97,0xc3,0xde,0x9f,0x25,0x71,
        0x73,0xd9,0x2e,0xbe,0x8b,0x30,0x1e,0xb5,0x60,0xe3,0x04,0x77,0x99,0x7e,0x1a,0x2b,0x89,0x11,0x82,0xdd,0xce,0xbf,0xc1,0x3f,
        0x2c,0x56,0x8d,0xb5,0x94,0x3f,0xa3,0x5c,0x52,0xdd,0xd9,0x3e,0x17,0x2c,0xb4,0x67,0x53,0x23,0xd7,0xb6,0xf5,0x52,0xba,0x7f,
        0x90,0x26,0x65,0x3d,0x27,0xc5,0xd6,0x34,0xee,0xbc,0xd8,0x86,0xbf,0x1f,0xf3,0x4c,0x9a,0xcb,0x0a,0x17,0x1b,0xf1,0xf7,0x62,
        0x2e,0x47,0x46,0xed,0x48,0x6e,0x41,0x4c,0x7e,0xb1,0x2d,0xdb,0x16,0x8b,0x95,0xdd,0xf1,0x77,0x63,0xa7,0x1e,0x90,0xda,0xba,
        0xb4,0xae,0xfc,0xc5,0x1a,0x7e,0x3c,0xe6,0x99,0x26,0x17,0x2d,0xa8,0x8d,0x7d,0x12,0x26,0x3f,0x1e,0xd7,0xdd,0x1f,0x8b,0x99,
        0xdd,0x26,0xda,0xf5,0xed,0x8f,0x15,0x5f,0xd5,0x4c,0x60,0x92,0xa0,0xcb,0xfa,0xf2,0xff,0x06,0x45,0xbf,0x1f,0x2d,0x3f,0x1a,
        0x6e,0x01,0x6b,0xfa,0xe8,0x47,0xe3,0x6a,0x73,0x43,0x35,0xf9,0xc5,0xde,0x58,0x5f,0x33,0x6c,0xc8,0x8f,0x84,0xe2,0xc3,0xdc,
        0x3f,0x12,0x3f,0xb7,0x3d,0x47,0xd9,0xc8,0xb5,0x3b,0x9b,0xa8,0xcd,0xac,0xa4,0x2c,0xaf,0x96,0x9b,0x4e,0xdd,0xec,0xd8,0xed,
        0xa4,0x45,0x9b,0xb5,0x68,0xe4,0x98,0xd9,0xc5,0xb6,0x6c,0xbc,0x5e,0x4f,0x8a,0x74,0xd3,0x8d,0x89,0xe5,0x6d,0x7e,0xb9,0xce,
        0xd2,0xfd,0x8f,0xd8,0x66,0x47,0xc2,0x45,0x85,0xbc,0x5d,0xfe,0x67,0x3c,0x76,0x24,0x5c,0x54,0x49,0xe2,0x26,0x63,0xb7,0xf7,
        0x2f,0x3f,0x1c,0x63,0xdf,0x86,0xcb,0x6e,0xc9,0xd3,0x20,0x5b,0x89,0x7e,0x38,0xae,0x59,0x3f,0xbc,0x66,0xcd,0x8a,0x09,0x2e,
        0x9a,0x98,0x14,0x4c,0x5b,0xf0,0x2c,0xd2,0xbd,0xff,0x7e,0x28,0x86,0x82,0xfd,0xb2,0xe2,0xf2,0x32,0xf9,0xa1,0xf8,0x73,0x3c,
        0x2d,0xc9,0x32,0xfb,0xeb,0x98,0x67,0x6a,0x2c,0x8d,0xb3,0xbf,0x8e,0xa3,0x7c,0x63,0xfe,0x2a,0xe6,0x79,0x40,0xda,0xcd,0xa3,
        0xbf,0x9a,0x55,0x77,0x33,0x7e,0x30,0x16,0x74,0x08,0xb6,0x59,0x2e,0x7e,0x30,0x8e,0xba,0x65,0x76,0x74,0x79,0xf9,0xc1,0x59,
        0x2b,0xb5,0x21,0x89,0x8b,0x22,0x6e,0x0d,0xd9,0xb0,0xf9,0x6f,0x71,0x3e,0xe0,0x0e,0x92,0xc4,0x18,0xbc,0xcb,0x64,0xfd,0xf0,
        0x5f,0x8b,0xef,0x1c,0x60,0x77,0xec,0xbc,0xff,0x4e,0xb0,0x4b,0xb6,0xec,0x95,0x8b,0x56,0x0c,0x7f,0xa0,0x58,0x0e,0xef,0x20,
        0xd9,0x13,0xfb,0x0f,0xc4,0x07,0xf7,0x65,0xb6,0xfd,0xa7,0x98,0xaf,0xc2,0xbd,0x1f,0x28,0xbe,0xb7,0xdb,0x89,0x4d,0x8a,0x4d,
        0x60,0x11,0x07,0x00,0x00,0xbd,0x12,0xc2,0xbd,0x93,0xe2,0x36,0xed,0xd8,0x98,0xc5,0x37,0xf8,0xfe,0xf8,0x38,0x8e,0xd5,0x94,
        0x8d,0x95,0xdd,0x9b,0x7e,0x7f,0xbc,0x45,0xcd,0x99,0x14,0xc7,0x1e,0x2f,0x8e,0x97,0x9a,0x74,0xc0,0xca,0xf5,0x65,0xf0,0xdf,
        0xe3,0x78,0x39,0x12,0x2e,0x7a,0xa1,0xc3,0xc7,0xb7,0x87,0xdf,0x17,0x9f,0x30,0x53,0x23,0x5b,0x35,0xbe,0x2f,0x2e,0x1e,0x3d,
        0x10,0xf3,0xbc,0x68,0x60,0xda,0xc4,0xb6,0x4f,0xd7,0xbb,0x27,0xeb,0xfe,0xbf,0x88,0x63,0xe9,0x10,0x64,0xad,0xfb,0x17,0xb1,
        0x91,0x97,0x89,0xa7,0x4c,0x6b,0x49,0x62,0x29,0x49,0x82,0xac,0x41,0xfe,0x4e,0x6c,0xcf,0xc4,0xb8,0x68,0x4f,0x12,0x0d,0x79,
        0xd3,0x9e,0x7f,0x3b,0xc6,0x90,0x28,0x1d,0xe3,0xcb,0x26,0xfd,0xf7,0x86,0xf1,0x59,0x36,0xc7,0xbe,0xb7,0x98,0x6a,0xc5,0x56,
        0x28,0x18,0x63,0x7f,0xcc,0x3a,0x60,0xd7,0x4c,0x8a,0xf2,0xa6,0x25,0x61,0x3d,0xcd,0xdf,0x1e,0x9d,0x7c,0x6f,0x31,0x6f,0x13,
        0xe0,0xaa,0xa8,0xee,0x79,0x47,0xca,0x56,0x6a,0xea,0x9c,0x72,0x67,0xc7,0x2f,0xde,0xde,0xe4,0xf9,0xb1,0x79,0xc2,0xc7,0xe2,
        0xfc,0xfb,0x58,0x2c,0x1d,0x1b,0x52,0xb6,0xe0,0x22,0xd5,0x46,0xae,0x69,0xf2,0xda,0x33,0x63,0x9e,0xd3,0x0a,0x8f,0x26,0x71,
        0xab,0x96,0x32,0x5d,0x6f,0x3f,0x8d,0xf0,0xb1,0x9b,0x7f,0xa1,0xfc,0x66,0xe7,0xdf,0x8a,0xd1,0xde,0x0a,0x36,0x0e,0xfa,0xaa,
        0x48,0x77,0xde,0x91,0x8d,0xa6,0xe1,0xf7,0xc4,0x0e,0x9f,0x23,0x3b,0xf2,0x3d,0x31,0x86,0xbb,0x47,0x6d,0x83,0x2c,0x92,0x9f,
        0x79,0xd6,0xea,0xcc,0x1a,0x6e,0x7a,0xfa,0x7b,0x62,0x8f,0x75,0x61,0x52,0x30,0xad,0x65,0x16,0x51,0x59,0xdd,0x82,0x72,0x67,
        0x87,0x49,0xf9,0xdd,0xb1,0xb1,0xee,0x10,0x59,0xc7,0x7d,0x77,0x9c,0x16,0x19,0xd3,0x87,0x91,0xf2,0x58,0xf7,0xcb,0xe0,0x77,
        0xc5,0xa1,0x11,0x09,0xdf,0x15,0xfb,0xe8,0xbb,0x62,0xe9,0xe8,0x83,0xb2,0x59,0x17,0xa9,0xb6,0xfc,0xc6,0xcc,0x67,0xd6,0x54,
        0xfa,0xa4,0x60,0x5a,0x12,0x2a,0x5b,0x4d,0x25,0x84,0xeb,0x53,0xe6,0xab,0x7b,0x55,0x6c,0x9d,0x77,0xa4,0xac,0xd9,0xc3,0xf9,
        0x2a,0x2d,0xd4,0xb3,0xe4,0x66,0xda,0x7f,0xb4,0xb8,0xd5,0x4c,0x80,0x9e,0x7f,0xff,0x25,0x7c,0x67,0xec,0xaa,0x43,0x90,0x44,
        0x25,0x36,0x66,0x58,0x70,0xb6,0xc8,0xe9,0x77,0xc6,0xb1,0x5b,0xa1,0xf6,0x94,0x9a,0x04,0x93,0x62,0xe7,0xb4,0x24,0x54,0xb6,
        0xaa,0x3b,0xcb,0x04,0x49,0xfc,0x7a,0x10,0x7a,0x98,0x7f,0x7f,0x33,0x8e,0xac,0x96,0x54,0xd3,0xb5,0xc8,0x65,0xdb,0x82,0xb6,
        0xa5,0xd7,0xf2,0x26,0x05,0x8d,0xaf,0x41,0x2c,0xb5,0xe7,0x77,0xc4,0xe4,0xe8,0x44,0xb6,0xcc,0x7c,0x47,0x5c,0xac,0xbe,0x23,
        0xae,0x60,0x15,0xca,0x63,0x1b,0xd2,0xee,0xf9,0xef,0x3b,0xe2,0x75,0xba,0x0b,0x1b,0xdd,0x7f,0x7e,0x7b,0xac,0x60,0x24,0x7c,
        0x7b,0x0c,0x33,0xe7,0x66,0x48,0x7d,0x7b,0x6c,0x97,0xcd,0x58,0x1a,0x98,0x2d,0x8f,0xa5,0xc4,0xb6,0x15,0x6c,0x0a,0xba,0x7d,
        0xf2,0xf2,0x1f,0x9f,0x2c,0x3b,0xa7,0x3c,0x56,0xed,0xa3,0xda,0x8e,0xab,0xed,0xc6,0x59,0x9e,0x6d,0x8f,0xd5,0x9e,0x52,0xdb,
        0x7f,0xcd,0x03,0xac,0x26,0x97,0x55,0x4b,0xd7,0xba,0x76,0x69,0xae,0x7c,0x75,0x24,0xaf,0xa5,0xd5,0x7c,0x68,0x9b,0x0e,0xc0,
        0xed,0x23,0xfb,0xf3,0x6d,0x71,0x7d,0xc1,0x9e,0x18,0x16,0x9c,0x2d,0x72,0xfa,0x6d,0x71,0x89,0xdd,0x94,0x16,0xb9,0xcc,0x1f,
        0x6b,0xaa,0xd9,0xa4,0x60,0x5a,0x4b,0x3c,0x65,0x7a,0x55,0x70,0xde,0x91,0x6a,0xd6,0x95,0xa1,0x38,0x6d,0xaa,0x67,0x12,0x53,
        0xa6,0xef,0xf9,0xf7,0xad,0x31,0x3e,0xf4,0xc1,0xb0,0xe0,0x6c,0x91,0xd3,0x6f,0x8d,0xf3,0x61,0x3d,0xd5,0x04,0xb5,0x99,0xd5,
        0xe6,0x59,0xb3,0xb3,0xae,0x82,0xd3,0x72,0x6b,0xdb,0xf1,0x72,0x55,0x24,0x3f,0xef,0x48,0x12,0x43,0x1f,0xc0,0x1d,0x27,0xfb,
        0xf3,0x37,0x8a,0x9f,0xc9,0x44,0x37,0x86,0x05,0x67,0x8b,0x9c,0x66,0xdc,0xaf,0x25,0xef,0x80,0x41,0xec,0x87,0x41,0xf1,0x32,
        0x45,0x5d,0xf2,0xda,0x5c,0x6a,0x76,0x3e,0x2c,0x76,0xe6,0x37,0x74,0x73,0x3c,0xa9,0x10,0xea,0x98,0xec,0x8b,0xe6,0xa6,0x5b,
        0xdd,0x2e,0xc7,0x4e,0xf6,0xf7,0x5b,0xe2,0xcd,0x09,0xfa,0x60,0x58,0x70,0xb6,0xc8,0xe9,0xb7,0xc4,0xdb,0xc2,0x4d,0xa9,0xcd,
        0xa5,0x26,0xcf,0x49,0xb1,0x73,0x5a,0x12,0xe6,0xe9,0xf7,0xa1,0xe9,0xaa,0xc8,0xf3,0xbc,0x23,0xed,0xcb,0x7b,0x18,0x26,0x1f,
        0x29,0x3e,0x73,0xd2,0x3f,0xa9,0x5c,0xff,0xbe,0x39,0x8e,0x1e,0xac,0x25,0x1b,0xec,0xdf,0x1c,0x67,0xc0,0x37,0xc7,0x19,0x50,
        0xd9,0xaa,0xee,0x6c,0xc1,0x7c,0x82,0x4a,0x41,0x1b,0x0c,0x8d,0xab,0x22,0xc1,0x79,0x47,0x92,0x18,0x8a,0x77,0x92,0x0f,0x87,
        0xf0,0x4d,0x59,0xbf,0xdf,0x4e,0x16,0x43,0x39,0xcd,0x76,0xde,0x5f,0x24,0x3f,0xf3,0x6c,0x46,0xed,0x29,0xd5,0x33,0x9b,0x4e,
        0x29,0x58,0x33,0x1f,0xbe,0x29,0x8e,0xeb,0x2e,0xc4,0xba,0x5c,0x94,0x4c,0x0a,0xa6,0x55,0xe2,0xb1,0xcb,0x8e,0x6d,0x06,0x00,
        0x77,0x96,0xec,0xcf,0x37,0xc6,0xfb,0x1e,0xb4,0x65,0x58,0x70,0xb6,0xc8,0xe9,0x37,0xc6,0x9b,0xcb,0x2d,0xa9,0xc9,0x25,0x96,
        0x97,0xc6,0x83,0x0a,0xb0,0xdd,0xfa,0xf2,0xa6,0xf8,0x21,0xce,0xa3,0xe6,0xe4,0x68,0xb9,0x2a,0x38,0xef,0x48,0xcc,0xec,0x62,
        0x5b,0x76,0x37,0x30,0xf3,0x25,0xfd,0x1b,0x36,0x62,0x29,0x97,0x95,0x3b,0xb7,0xcd,0xba,0xc5,0xce,0x74,0x19,0x16,0x9c,0xb5,
        0xa0,0x0c,0xac,0x76,0xe7,0xec,0xd8,0xe9,0x37,0xc4,0xab,0x61,0xa4,0x3c,0xb3,0xd8,0xaa,0x96,0xb7,0x58,0x7a,0x71,0xca,0xa4,
        0xd8,0x9a,0x56,0xe9,0x38,0x6c,0xae,0x8a,0x5c,0xce,0x3b,0x92,0xc4,0xd2,0x8c,0xfd,0x5c,0xff,0xbe,0x3e,0xce,0x8e,0xcd,0xe8,
        0x27,0x97,0xa3,0x65,0x58,0x70,0xb6,0xc8,0xe9,0xd7,0xc7,0xe5,0x62,0x99,0xa5,0x85,0xa5,0x72,0x4a,0x6d,0x66,0xf3,0x79,0x96,
        0xc5,0x3e,0x2c,0x98,0x56,0x2b,0x91,0xc4,0x38,0xeb,0x30,0x3e,0x3f,0x1c,0xdf,0x02,0xfd,0x70,0x7c,0x0b,0x74,0x3d,0xa1,0xc2,
        0xfc,0xb1,0x32,0x33,0x2c,0xb0,0x69,0xeb,0x9e,0xb5,0x48,0xf7,0xb0,0x38,0x65,0x1a,0xcf,0xcc,0x2f,0x6e,0x2b,0xa9,0x9e,0x59,
        0x26,0xcf,0xbb,0x7f,0xcb,0xf7,0x8b,0xbf,0x2e,0xd6,0xec,0xeb,0x62,0x5d,0xd6,0x93,0x27,0x38,0x9b,0x51,0x73,0x4a,0x99,0x27,
        0xb6,0x66,0x93,0xee,0xd8,0x98,0x9a,0x8e,0x9b,0x14,0x3b,0xa7,0x55,0xca,0xba,0x3c,0x8c,0x5b,0x55,0xaa,0x67,0x6e,0x3b,0xf8,
        0xda,0x8d,0xcf,0x77,0x62,0xbb,0xec,0x8d,0xb7,0x03,0x52,0x21,0x89,0x6b,0x2a,0x00,0x00,0xa9,0x93,0xfd,0x79,0x1d,0x2f,0x9f,
        0x5b,0xd1,0x4f,0x2e,0xc7,0xc0,0xb0,0x20,0xbf,0x4f,0x7e,0x1d,0x5f,0x59,0x68,0x49,0xd9,0x82,0x9b,0xa4,0x2b,0x0b,0xca,0xb9,
        0x5f,0x96,0x3e,0x29,0x98,0x36,0xd1,0xcf,0x78,0xb9,0x2a,0x32,0x3b,0xef,0x48,0xac,0xf5,0xc5,0x61,0xa9,0x8d,0x2f,0x7b,0x5e,
        0x79,0x15,0x8a,0xcf,0xac,0xbf,0x8a,0x2f,0xf4,0xbc,0x8a,0x5d,0x7c,0x2c,0x54,0xa3,0xcd,0x86,0xd4,0xab,0x38,0xb2,0x52,0x63,
        0x56,0xc1,0xbc,0xd6,0x67,0x5b,0xe4,0x52,0x9b,0xae,0x66,0x67,0x12,0xab,0x30,0x0e,0x78,0xfd,0xbb,0x8a,0xb3,0xa3,0x25,0xb5,
        0xe9,0xb6,0xcd,0xec,0x68,0xc9,0x26,0xd7,0x55,0x9c,0x63,0xeb,0xa8,0xed,0x95,0xea,0xb1,0xda,0x04,0xed,0x4b,0x28,0xa9,0xd6,
        0xac,0xc3,0xf5,0xef,0x2a,0x5e,0xc7,0xba,0xb0,0xd1,0xf8,0x7c,0x19,0x9b,0x75,0x1d,0xb5,0x09,0x36,0xce,0xe5,0x76,0x32,0x2c,
        0x38,0x6b,0xe2,0x66,0xd8,0xbc,0x8c,0x63,0xe2,0x65,0xe5,0x2a,0x53,0x49,0x50,0x9e,0x32,0xbf,0x55,0x6d,0xcf,0x8d,0xc6,0xcb,
        0xcb,0xd8,0xef,0x5d,0x48,0x62,0xa9,0x04,0x00,0x00,0x77,0x84,0xec,0xcf,0x87,0xe2,0x7d,0xd6,0x01,0xc9,0xee,0xc1,0x3e,0x14,
        0x6f,0xda,0x22,0xf9,0xce,0xb3,0x92,0x78,0xca,0xb4,0x24,0x89,0x36,0x03,0x80,0x43,0x90,0xfd,0xf9,0xe0,0x36,0xcf,0xa9,0x1d,
        0xca,0x7b,0x11,0x5f,0x52,0x4f,0x88,0x49,0xc6,0xe5,0xc6,0xbc,0x0e,0xe3,0x8c,0x67,0xb5,0x6c,0x9b,0x67,0x47,0x76,0xd8,0x71,
        0x93,0xa2,0xb1,0xf2,0xeb,0xe6,0x65,0xbc,0xd0,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xdc,0x65,0x92,0x78,0xab,0x0f,
        0xe8,0xe3,0xfd,0xdb,0x6c,0x40,0x3f,0x00,0xee,0x36,0x37,0x6f,0x8b,0x5f,0xc6,0xef,0x4c,0xae,0x64,0x07,0xf3,0x2f,0x8d,0xab,
        0x1a,0x00,0x1c,0x0b,0x49,0xdc,0x60,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xfc,0xf7,0xd6,0x9e,0xc7,0x5f,0x08,0x02,0x12,0x67,0x5a,0xb2,0xdb,0xf9,
        0xf0,0x7f,0x07,0x45,0x79,0x00,0x00,0x00,0x00,0x00,0x00,0xdd,0x78,0x55,0xc7,0xa3,0x3a,0x92,0xa8,0xee,0x81,0x68,0x6a,0x82,
        0x6a,0x9b,0xbd,0x5a,0x9f,0x00,0x00,0x80,0x7d,0x51,0x7b,0x61,0x4f,0xa2,0x66,0x00,0xb0,0x2b,0xf2,0xf7,0x53,0x07,0xe3,0xc1,
        0xbd,0x62,0xeb,0x51,0x7b,0xe2,0x8a,0xf9,0x28,0x2e,0x95,0x8f,0xca,0x5c,0x36,0xe3,0xe0,0x1f,0xaa,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x3b,0x40,0xf6,0xe7,0xff,0xc4,0x7f,0xc4,0x0e,0x00,0x00,0x00,
        0x00,0x0e,0xcd,0xab,0xc8,0xa3,0x14,0xea,0x02,0xdc,0x72,0xf2,0xe7,0xfe,0x8d,0xff,0x85,0x79,0xec,0x92,0x83,0xbf,0x08,0x04,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0x01,0xd9,0x9f,0x67,0x61,0x3c,0x06,0x00,0x00,
        0x00,0x00,0x00,0xfb,0x63,0x9b,0xe7,0xf7,0xa7,0x31,0xf9,0x91,0x51,0xc6,0x77,0x55,0x6c,0x9d,0x77,0xa4,0xa9,0x05,0x27,0xc5,
        0x29,0xd3,0x92,0x6c,0xd7,0x34,0x56,0x02,0x68,0xe0,0x66,0x7c,0x4e,0xe3,0x38,0xeb,0xc2,0x46,0xf3,0xfd,0x03,0xb1,0x12,0xc7,
        0x42,0x35,0xbe,0x61,0x38,0xcd,0x76,0xde,0x6f,0x62,0x52,0xa4,0x9b,0xf6,0xf6,0xfa,0xe7,0x5b,0xb1,0x2e,0xe8,0x83,0xac,0x8f,
        0xde,0x8a,0x5d,0xd5,0x92,0x49,0x91,0x6e,0x5a,0x25,0xd4,0x70,0xd9,0xc0,0xa8,0x96,0x4a,0x66,0x97,0xfd,0x8c,0x97,0xab,0x22,
        0xb3,0xf3,0x8e,0x54,0x06,0xdf,0x45,0x7b,0xd2,0x1d,0xb4,0xb7,0x65,0xaa,0x25,0xf1,0xa6,0x0b,0x00,0x00,0x00,0x00,0x00,0x00,
        0xd0,0x92,0xec,0xcf,0x93,0x30,0x1e,0x1e,0x39,0x27,0x47,0xce,0x45,0xba,0x4c,0x0a,0xa6,0x25,0xa1,0x42,0x12,0x33,0x00,0xe8,
        0xb4,0x7e,0x3e,0x8e,0xef,0x0e,0xe0,0xce,0xd3,0x34,0x50,0x86,0xe1,0x34,0x3b,0xe5,0x7e,0x79,0xe6,0xb0,0xe0,0x0c,0x87,0x62,
        0xe3,0x89,0x5e,0xf6,0x5f,0x2d,0xb1,0x53,0x47,0xbb,0x20,0x89,0x45,0x0e,0x5b,0xf3,0x72,0x14,0xde,0x37,0x18,0x8f,0x32,0xde,
        0x5b,0x30,0x19,0x0e,0xc6,0xd3,0x6c,0x2b,0x63,0x74,0x6f,0x74,0xfa,0xfe,0xc1,0xf8,0x7e,0x7e,0xe6,0x20,0x26,0xe8,0xce,0xb0,
        0xc8,0xfa,0xac,0xa4,0xd7,0xac,0x6b,0x79,0x51,0xc4,0x37,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0xe0,0x20,0xbc,0x09,0x21,0x0c,0xc6,0x83,0x8d,0x49,0xa2,0xf2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0x03,0xc2,0x28,
        0xbc,0x1f,0x00,0x00,0x00,0x00,0x6e,0x37,0xdb,0x7e,0x22,0xe4,0x6e,0xd1,0xf1,0xf1,0x31,0x89,0x18,0x80,0xdb,0x4c,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1b,0x11,0x46,0xe1,0xd1,0x60,0x3c,0xaa,
        0x25,0x3f,0x65,0xd0,0x7c,0x4a,0xfb,0x74,0xe5,0xce,0x17,0xc5,0xd6,0x38,0xe3,0x1e,0x00,0x00,0x38,0x1e,0x1e,0x2d,0x51,0x5e,
        0xfb,0xb1,0x39,0xef,0x8d,0xcd,0x5a,0xe5,0x70,0x75,0x79,0x14,0x2b,0xb1,0xcc,0xb6,0x79,0x66,0x23,0xe4,0xbd,0x71,0xa0,0x00,
        0x00,0x00,0x00,0xb7,0x93,0x75,0xb7,0xbd,0x07,0x7f,0xd1,0x17,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0xe7,0xc9,0xfe,0xbc,0x27,0x8c,0xc7,0xc0,0xf1,0x73,0xd8,0xf9,0xb0,
        0x8b,0xf2,0x0e,0x34,0xdf,0xeb,0xf2,0xbc,0x1c,0x86,0xd3,0x8c,0xfb,0x25,0x93,0x62,0xe7,0x74,0x18,0xce,0xb2,0x53,0x1e,0xd5,
        0x92,0xc4,0x02,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x06,0xc3,0x70,0x3a,0x18,0x8c,0xef,0x63,0x6b,0xf2,0x86,0x1c,0xc4,0xf6,0x1c,0xdc,0x82,0xf6,0xac,0x56,0x77,
        0x63,0x42,0xc1,0x25,0x80,0x35,0x74,0x9c,0x6a,0x2b,0x39,0xf8,0x45,0xe3,0x88,0xe8,0xa7,0xa7,0x01,0x00,0xd8,0x17,0xfd,0xdc,
        0x4f,0x24,0x11,0x4a,0x4a,0x68,0x9e,0xdb,0xc6,0xb0,0xe0,0x6c,0x46,0xcd,0x43,0x6f,0xab,0x74,0x4d,0x9c,0x65,0x3c,0xda,0x82,
        0xed,0x9e,0xd1,0x17,0xeb,0xd9,0x29,0xf9,0x31,0x70,0xb7,0xa2,0x3d,0x2a,0x26,0x73,0xec,0xea,0x81,0xb8,0xd3,0xf5,0x6f,0xf2,
        0x30,0x14,0x15,0xc4,0xed,0x27,0x5b,0x12,0x32,0xee,0x57,0x29,0x8f,0x25,0xf1,0x72,0x03,0x80,0x5b,0x44,0xfe,0x77,0x30,0xbe,
        0x48,0x82,0x37,0x61,0x27,0xd7,0xbf,0xe5,0xeb,0x34,0xd0,0x17,0x9d,0xc7,0x67,0x02,0x13,0x6f,0xc7,0x53,0xf4,0xe0,0x8b,0x1c,
        0x70,0x47,0x09,0x05,0xc3,0x5b,0xc9,0xc9,0xad,0x24,0x89,0xa6,0xdb,0x65,0x5b,0x57,0x46,0x96,0xb6,0xbe,0x33,0xe3,0xba,0x76,
        0x45,0xa9,0x5d,0x6d,0x9a,0x96,0xa0,0xf6,0xcb,0x53,0x2f,0x2b,0x58,0x9b,0x21,0xbc,0xed,0x80,0xde,0xdd,0x44,0x48,0x62,0x98,
        0xde,0x95,0x71,0x5d,0xd2,0x7e,0x44,0x76,0x1c,0x98,0xdb,0x26,0x4f,0x68,0xdd,0xdd,0x5f,0x5b,0x6f,0xda,0x2e,0xc3,0x7e,0xee,
        0x82,0x8e,0xa8,0xad,0x6f,0x17,0x49,0x34,0x9d,0xb6,0x4e,0x98,0x24,0x9a,0x4e,0x5b,0x27,0x4c,0x12,0x4d,0xa7,0xad,0x13,0x26,
        0x89,0xa6,0xbb,0x23,0x6d,0x7d,0xcd,0x11,0xbc,0x2e,0xf5,0x74,0x6f,0x8d,0xd5,0x17,0x93,0x6d,0x6a,0x9d,0x44,0x5b,0xeb,0x15,
        0xbd,0xa2,0x57,0xf4,0x8a,0x5e,0x49,0x91,0x2e,0xbd,0xb2,0x70,0x27,0x70,0x15,0xb2,0xc3,0xe3,0x73,0xec,0x84,0xac,0x89,0x27,
        0xf1,0xe3,0xaa,0x00,0x00,0x00,0x00,0x80,0x63,0x66,0x18,0x4e,0x27,0xf1,0xbb,0xa4,0x38,0x08,0x3a,0x27,0xc5,0x96,0x6f,0xd1,
        0x2b,0x7b,0x9b,0x9b,0xd5,0x9d,0x93,0x82,0x69,0x2d,0xf1,0x94,0x69,0xfe,0xc2,0xd9,0x5e,0xbe,0x3c,0xdc,0xa2,0x91,0xdb,0x9f,
        0xd9,0x0f,0x49,0x7c,0xa8,0x1e,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0x77,0xb2,0x3f,0x17,0xf1,0x0b,0x28,
        0xdd,0x19,0xf7,0x99,0x59,0x57,0x92,0x68,0x5d,0x00,0x40,0xaa,0x0c,0xc3,0xe9,0x83,0xf8,0x4f,0x05,0xb4,0x64,0xdb,0x74,0xfd,
        0xb0,0x8b,0xca,0xb7,0xcf,0x2c,0x89,0x1e,0x03,0x00,0xf4,0x41,0xf6,0xe7,0x3c,0x3e,0xb9,0x61,0x43,0x6a,0xdb,0x33,0xbb,0x52,
        0x9e,0xc7,0x0b,0x66,0x4b,0x62,0x66,0xa3,0xf6,0x54,0xcb,0xbb,0x2a,0x76,0x9e,0x77,0xa4,0xf5,0x08,0xf1,0xaf,0x28,0x01,0x00,
        0x80,0x5b,0x47,0xf6,0xe7,0x2c,0xde,0xcd,0x1c,0x0b,0xd9,0x7d,0xe4,0xd9,0xe1,0x5e,0x9b,0xc9,0xfe,0x9c,0xc6,0xba,0xa0,0x2d,
        0x59,0x8f,0x9d,0xc6,0x07,0x80,0x65,0xae,0x8a,0x53,0xce,0x3b,0xd2,0xa2,0xe3,0xde,0x0e,0xe3,0xec,0xcc,0x67,0x49,0x4c,0x4a,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xec,0x8c,0xec,0xcf,0xfd,0xf8,0xc9,0x95,0x96,0xb4,0x48,0x57,
        0x7b,0xca,0xb6,0xe5,0x6d,0xcb,0xac,0xbc,0xc1,0xb8,0xd8,0x09,0x00,0xe8,0x9b,0x37,0x61,0x10,0x2f,0x28,0x83,0x9b,0x05,0xf7,
        0x62,0x5b,0x86,0xe1,0x74,0x1c,0x3f,0x2d,0xd9,0x92,0x6a,0xe9,0x58,0x20,0x6b,0xd6,0x51,0xf1,0x2f,0x0f,0xdf,0x21,0x4e,0xee,
        0x08,0xd9,0x0c,0x18,0xc5,0x89,0xd0,0x92,0x98,0x6e,0xd4,0x34,0x5e,0xae,0x8a,0x53,0xce,0x3b,0xd2,0x54,0xc2,0xab,0xe2,0x94,
        0x69,0x49,0x12,0x93,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8e,0x82,0x61,0x4e,0x28,0xb8,0x40,0x8a,0xbc,0x2e,0xb8,0x6c,
        0x38,0xf6,0xac,0x29,0x79,0xd9,0xc5,0x7d,0x91,0xc4,0xa0,0x85,0xf9,0x7e,0x50,0xea,0x67,0xe3,0x2e,0xe8,0x79,0xfa,0x9a,0xef,
        0x00,0xee,0x28,0xd9,0xa2,0x7a,0x52,0x7c,0x54,0xa9,0x86,0x9b,0x6b,0xe3,0xca,0x53,0x92,0xa7,0x36,0x86,0xea,0xce,0xea,0x02,
        0xbf,0x8b,0xd8,0x5b,0x64,0xd6,0x54,0x7a,0xfb,0xab,0x53,0x8b,0x04,0xed,0xcb,0xdb,0x61,0x7b,0x76,0xac,0x4b,0x8b,0xf2,0x7a,
        0x1d,0xd0,0xfd,0x64,0x7d,0xd8,0x01,0xb6,0xfb,0x39,0xd6,0x62,0xd4,0x6d,0x5b,0xeb,0xe4,0xe3,0x43,0x4f,0x84,0x30,0xc9,0x9a,
        0xfc,0x69,0x89,0xfb,0x79,0x00,0x00,0x76,0xc1,0x8b,0x30,0x09,0x83,0xd1,0x65,0x3d,0xa3,0x86,0x63,0x77,0x9a,0xfc,0x8b,0x02,
        0x83,0x15,0x24,0x51,0x41,0xa0,0xe7,0x71,0xbd,0x2d,0x07,0xaa,0xfc,0xbe,0x8b,0xdd,0x79,0x41,0xa3,0x3d,0x97,0x97,0x3e,0x7d,
        0x8c,0xcf,0x07,0xd9,0x13,0xfe,0x60,0x7c,0xd1,0x13,0x79,0xcd,0x66,0xd4,0xd6,0xfa,0xc1,0x30,0xff,0x22,0xe7,0x31,0x91,0xff,
        0x8e,0x40,0xc1,0x91,0xc6,0x77,0x0c,0x14,0x9d,0x53,0x65,0xc5,0xac,0x2a,0x12,0x54,0xd9,0x45,0x95,0x76,0x5f,0xc2,0x0d,0xa3,
        0xba,0x82,0xca,0x9d,0x1d,0x39,0x68,0xa7,0xee,0xf4,0xaa,0x8d,0x4d,0xf9,0xba,0x9c,0x7b,0x97,0xb7,0x8c,0x17,0xa3,0x70,0xef,
        0x64,0x9c,0x13,0x4e,0x46,0x05,0x3b,0x1e,0x52,0xa3,0x38,0x76,0x7b,0x60,0xd4,0x4b,0x2e,0xbd,0x52,0x5b,0xa5,0x51,0x37,0xc6,
        0xcd,0xdd,0xd1,0xb1,0xd6,0x5b,0xf5,0xdf,0x5e,0x06,0xca,0x75,0x79,0x95,0xbb,0xdf,0x26,0x52,0x58,0x04,0x00,0xdc,0x26,0xda,
        0xaf,0x2f,0xb7,0x9a,0x1d,0xbe,0x22,0xd1,0xb1,0x4a,0x49,0x34,0x0f,0xf6,0x37,0x06,0x37,0x4d,0xf7,0xfe,0xd4,0x27,0x49,0xa8,
        0xd0,0xef,0x24,0xd9,0x01,0xa1,0x42,0xfb,0x04,0xed,0xbb,0xaa,0x4a,0x8b,0xde,0xbc,0x4e,0x10,0xe6,0xd9,0xbc,0x21,0x13,0x5c,
        0xe4,0x7a,0x1d,0x8a,0x2d,0x8e,0x55,0x78,0x9c,0x73,0xef,0x79,0x13,0x49,0x8c,0x48,0x6c,0x3f,0x42,0x2a,0x5b,0x2d,0x26,0x5e,
        0x3f,0xa4,0xd1,0x04,0xbb,0x8b,0xaf,0xd7,0x76,0x69,0xd1,0x66,0x4d,0x99,0xf5,0xd3,0x01,0xb5,0x79,0xee,0xbe,0x57,0x42,0xeb,
        0xad,0xf6,0x95,0x6f,0x5f,0x50,0xfb,0x12,0x70,0xfc,0xd4,0x5d,0x30,0xc3,0xa6,0xec,0xbb,0xd6,0x1b,0x57,0x30,0xa5,0x88,0x76,
        0x52,0xeb,0xe5,0xf7,0xb2,0xda,0xb3,0x78,0xb3,0x1e,0xda,0xdf,0xec,0x25,0x7b,0x3f,0xdf,0x66,0xb0,0xf7,0x42,0xe3,0xed,0x79,
        0xf5,0xcc,0xdd,0x55,0xa2,0x9f,0x18,0x5a,0x54,0x7e,0x0f,0x31,0x24,0xdb,0x4a,0xb7,0x93,0xa6,0x19,0xde,0x66,0xf6,0xa7,0xce,
        0xe6,0x11,0x25,0xb1,0x12,0xe1,0xb0,0x6c,0xfc,0xa0,0xd2,0x44,0xa7,0xab,0xef,0x51,0xf1,0x4e,0x1d,0x2f,0xeb,0x68,0xca,0x25,
        0x89,0x11,0x72,0xa0,0x11,0x99,0xee,0x44,0xd8,0x49,0x28,0xad,0xd7,0xf9,0xc6,0x3b,0x8f,0x16,0x59,0xd7,0x9f,0xb2,0xe5,0x5b,
        0x61,0xbd,0x5e,0xc7,0xda,0x54,0x69,0xd3,0xd8,0x37,0xe8,0x80,0x9a,0xf2,0x92,0x58,0x4a,0x80,0x5b,0xc0,0x2e,0xde,0x7f,0x4f,
        0x22,0x30,0x1c,0x13,0xcd,0x17,0xa2,0x14,0x3e,0x1f,0xd2,0xa2,0xbc,0x7d,0x7d,0xfe,0x25,0x6f,0xb3,0x10,0x9b,0xae,0x37,0xf2,
        0xac,0x43,0x2c,0x61,0x89,0x5d,0x94,0x87,0x7e,0xfb,0x68,0xf7,0xfd,0xb7,0xef,0xf2,0x8e,0x8f,0xc3,0xf6,0x5f,0x4a,0x2c,0x86,
        0x99,0xed,0x8c,0xcb,0x7d,0x18,0x6d,0xdd,0x4a,0xb8,0x2b,0xe4,0x83,0xa8,0xef,0x1b,0x90,0xf6,0x9f,0x7b,0xea,0xf8,0xda,0x45,
        0xaf,0x1f,0xd4,0xe8,0xf5,0xf5,0x90,0x9e,0x3e,0x46,0x51,0x69,0xcf,0xae,0xaf,0xdb,0xc4,0xcc,0x92,0x78,0x01,0xec,0x88,0x48,
        0xf7,0x03,0x41,0x68,0x33,0x45,0x5b,0xec,0x4c,0xa2,0x9e,0x2d,0x3e,0x13,0x75,0xb4,0x1f,0x6d,0xdb,0x21,0xc5,0x25,0x0b,0xe8,
        0x81,0xdd,0xbc,0x4c,0x51,0x66,0xbd,0x43,0x36,0x7d,0xbd,0xa7,0xb6,0x9e,0x3b,0x4c,0xd7,0x22,0x97,0xf6,0xa4,0xb0,0xe8,0xf4,
        0xc4,0xee,0xe3,0xdb,0xb2,0x3d,0xeb,0xaf,0x2b,0xfd,0xcc,0x87,0xbd,0x05,0xd6,0xb1,0xbc,0x2d,0xe3,0xdb,0xfc,0x72,0xdd,0xad,
        0x3d,0x93,0x78,0x09,0x03,0xc0,0xe1,0x78,0x93,0xbf,0x16,0x34,0xb6,0x16,0x1c,0x9a,0xb2,0x1f,0xee,0x32,0x57,0xe1,0x24,0xe3,
        0xbc,0x23,0xed,0xcb,0x4b,0xe2,0x45,0x58,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5b,0xf2,0x26,0x9c,0xc5,0xf7,
        0xff,0x0a,0x5e,0x8c,0xc2,0x70,0x30,0x1e,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x2a,0x61,0x30,0x1e,0x1c,0x33,0x49,0x34,0x32,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x7b,0xfc,0xfb,0x8d,0x00,0x80,0x2d,0x71,0x7d,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0x0f,0xc2,0x60,0x3c,0xd8,0x2d,0x49,0x84,0x09,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x74,0x23,
        0x0c,0xc6,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd8,0x88,0x24,0x7e,0x5d,0x15,0x00,0x00,0x00,0x00,0x80,0x8d,0x49,0xe2,
        0xb1,0x1a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x2f,0xde,0x84,0xb3,0x10,0x66,0xbc,
        0x38,0xf4,0x67,0x35,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x49,0x10,0x06,0xe3,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x44,0xde,0x84,0xb3,0x10,0xe6,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x40,0x85,0xb3,0x0a,0x49,0x54,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x09,0x71,0x56,0x21,0x89,0x2a,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,0xb3,0x77,0xc7,0x38,0x08,0x03,0x31,0x14,0x05,0x37,0xab,0x28,0xe0,0x32,
        0xdc,0xff,0xb0,0x48,0xa9,0xb0,0x48,0x43,0x43,0x3e,0x62,0xaa,0x29,0xdc,0x3c,0x17,0xd6,0x96,0x0b,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0x35,0x22,0x92,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x44,0xbd,0x33,0x0f,0xf6,0xd3,0xd9,0x4f,0x73,0x2c,0xb6,0xee,0x40,0x24,0x11,
        0x47,0x02,0xfc,0x29,0x73,0xde,0x97,0xb1,0xdd,0x00,0x00,0xb8,0x94,0x88,0x47,0x11,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x1f,0x10,0xf1,0x25,0x21,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xf0,0x35,0xea,0x95,0x39,0xc6,0xb2,0x3d,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xcf,0x76,0xe7,0xd0,0x06,
        0x00,0x00,0x84,0x01,0xd8,0x30,0x24,0xfc,0xff,0x30,0x8a,0x1f,0x10,0x55,0xb5,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1c,0x99,0xa4,0xeb,0x78,0x51,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0x23,0x32,0x49,0xd7,0x02,0x07,0x5b,0x8e,0x31,
    };
};

class BreakPairTable
{
public:
    BreakPairTable() = delete;

    static BreakOpportunity getLineBreakOpportunity (LineBreakType a, LineBreakType b)
    {
        return (BreakOpportunity) table[(int) a][(int) b];
    }

private:
    static constexpr auto numTypes = 44;
    static constexpr uint8_t table[numTypes][numTypes]
    {
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x2,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x2,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x0,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x1,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x0,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x1,0x1,0x0,0x1,0x1,0x2,0x1,0x0,0x0,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x1,0x2,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x1,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x2,0x0,0x4,0x0,0x2,0x0,0x0,0x0,0x2,0x2,0x2,0x0,0x2,0x2,0x2,0x0,0x0,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x2,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x0,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x1,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x1,0x1,0x1,0x0,0x0,0x2,0x2,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x2,0x1,0x1,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x0,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x3,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x1,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x2,0x0,0x0,0x2,0x0,0x0,0x1,0x1,0x0,0x2,0x0,0x0,0x0,0x0,0x1,0x1,0x0,0x0,0x0,0x1,0x0,0x0,0x0,0x2,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
        { 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,},
    };
};
}
