/*
 *  $Id: volume_linestat.c 24975 2022-08-31 12:05:41Z yeti-dn $
 *  Copyright (C) 2015-2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwynlfit.h>
#include <libprocess/brick.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/gwyprocess.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/correlation.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-volume.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)


enum {
    PREVIEW_SIZE = 500,
    RESPONSE_RECALC = 101,
};

enum {
    PARAM_XOFFSETS,
    PARAM_YOFFSETS,
    PARAM_XOFFSETS_CURVE,
    PARAM_YOFFSETS_CURVE,
    PARAM_XOFFSETS_FLIP,
    PARAM_YOFFSETS_FLIP,
    PARAM_SHIFT,
    PARAM_MAXSHIFT,
    PARAM_ROTATE,
    PARAM_MAXROTATE
};


typedef struct {
    GwyParams *params;
    GwyBrick *brick;
    GwySurface *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table_options;
    GwyContainer *data;
} ModuleGUI;


static gboolean              module_register          (void);
static GwyParamDef*          define_module_params     (void);
static void                  xystitch                 (GwyContainer *data,
                                                       GwyRunType runtype);
static void                  execute                  (ModuleArgs *args);
static GwyDialogOutcome      run_gui                  (ModuleArgs *args,
                                                       GwyContainer *data,
                                                       gint id);
static void                  param_changed            (ModuleGUI *gui,
                                                       gint id);
static void                  dialog_response          (GwyDialog *dialog,
                                                       gint response,
                                                       ModuleGUI *gui);
static void                  recalc                   (gpointer user_data);
static void                  sanitise_params          (ModuleArgs *args);


static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Merges all XY planes into XYZ data"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek & David Nečas (Yeti)",
    "2023",
};

GWY_MODULE_QUERY2(module_info, volume_xystitch)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_xystitch",
                             (GwyVolumeFunc)&xystitch,
                             N_("/SPM M_odes/_XY Stitch..."),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME | GWY_MENU_FLAG_GRAPH_CURVE,
                             N_("Merge all the XY planes to XYZ data"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_volume_func_current());
    gwy_param_def_add_graph_id(paramdef, PARAM_XOFFSETS, "xoffsets", _("_X offsets graph"));
    gwy_param_def_add_graph_curve(paramdef, PARAM_XOFFSETS_CURVE, "xcurve", _("X _offsets curve"));
    gwy_param_def_add_graph_id(paramdef, PARAM_YOFFSETS, "yoffsets", _("_Y offsets graph"));
    gwy_param_def_add_graph_curve(paramdef, PARAM_YOFFSETS_CURVE, "ycurve", _("Y o_ffsets curve"));
    gwy_param_def_add_boolean(paramdef, PARAM_XOFFSETS_FLIP, "xflip", _("_Flip X axis"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_YOFFSETS_FLIP, "yflip", _("_Flip Y axis"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_SHIFT, "shift", _("_Adjust shifts"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_ROTATE, "rotate", _("A_djust rotation"), FALSE);
    gwy_param_def_add_double(paramdef, PARAM_MAXSHIFT, "max_shift", _("_Shift range"), 0.0, 50, 20);
    gwy_param_def_add_double(paramdef, PARAM_MAXROTATE, "max_rotate", _("_Rotation range"), 0.0, 10, 1);

    return paramdef;
}

static void
xystitch(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyBrick *brick = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint oldid, newid;
    GwyAppDataId graph_id;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_app_data_browser_get_current(GWY_APP_BRICK, &brick,
                                     GWY_APP_BRICK_ID, &oldid,
                                     GWY_APP_CONTAINER_ID, &graph_id.datano,
                                     GWY_APP_GRAPH_MODEL_ID, &graph_id.id,
                                     0);
    g_return_if_fail(GWY_IS_BRICK(brick));
    args.result = NULL;
    args.brick = brick;
    args.params = gwy_params_new_from_settings(define_module_params());
    gwy_params_set_graph_id(args.params, PARAM_XOFFSETS, graph_id);
    gwy_params_set_graph_id(args.params, PARAM_YOFFSETS, graph_id);
    sanitise_params(&args);

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args);

    if (args.result) {
        newid = gwy_app_data_browser_add_surface(args.result, data, TRUE);
        gwy_app_set_surface_title(data, newid, _("Stitched"));
    }

end:
    g_object_unref(args.params);
    if (args.result)
        g_object_unref(args.result);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyContainer *data, gint id)
{
    GtkWidget *hbox, *dataview;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDialogOutcome outcome;
    GwyBrick *brick = args->brick;
    GwyDataField *field = gwy_data_field_new(gwy_brick_get_xres(brick),
                                             gwy_brick_get_yres(brick),
                                             gwy_brick_get_xreal(brick),
                                             gwy_brick_get_yreal(brick),
                                             TRUE);
    const guchar *gradient;

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();

    gwy_container_set_object(gui.data, gwy_app_get_data_key_for_id(0), field);
    if (gwy_container_gis_string(data, gwy_app_get_brick_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(gui.data, gwy_app_get_data_palette_key_for_id(0), gradient);

    gui.dialog = gwy_dialog_new(_("XY Stitch"));
    dialog = GWY_DIALOG(gui.dialog);
    gtk_dialog_add_button(GTK_DIALOG(dialog), gwy_sgettext("verb|_Stitch"), RESPONSE_RECALC);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(gui.data, 0, PREVIEW_SIZE, FALSE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table_options = gwy_param_table_new(args->params);
    gwy_param_table_append_graph_id(table, PARAM_XOFFSETS);
    gwy_param_table_append_graph_curve(table, PARAM_XOFFSETS_CURVE, gwy_params_get_graph(args->params,
                                                                                         PARAM_XOFFSETS));
    gwy_param_table_append_checkbox(table, PARAM_XOFFSETS_FLIP);

    gwy_param_table_append_graph_id(table, PARAM_YOFFSETS);
    gwy_param_table_append_graph_curve(table, PARAM_YOFFSETS_CURVE, gwy_params_get_graph(args->params,
                                                                                         PARAM_YOFFSETS));
    gwy_param_table_append_checkbox(table, PARAM_YOFFSETS_FLIP);

    gwy_param_table_append_checkbox(table, PARAM_SHIFT);
    gwy_param_table_append_slider(table, PARAM_MAXSHIFT);
    gwy_param_table_set_unitstr(table, PARAM_MAXSHIFT, "px");
    gwy_param_table_append_checkbox(table, PARAM_ROTATE);
    gwy_param_table_append_slider(table, PARAM_MAXROTATE);
    gwy_param_table_set_unitstr(table, PARAM_MAXROTATE, "deg");

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_after(dialog, "response", G_CALLBACK(dialog_response), &gui);
    param_changed(&gui, PARAM_XOFFSETS);

    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.data);

    return outcome;
}


static void
param_changed(ModuleGUI *gui, gint id)
{
    if (id == PARAM_XOFFSETS || id == PARAM_YOFFSETS) {
        gwy_param_table_graph_curve_set_model(gui->table_options, PARAM_XOFFSETS_CURVE,
                                              gwy_params_get_graph(gui->args->params, PARAM_XOFFSETS));
        gwy_param_table_graph_curve_set_model(gui->table_options, PARAM_YOFFSETS_CURVE,
                                              gwy_params_get_graph(gui->args->params, PARAM_YOFFSETS));
    }

    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
dialog_response(G_GNUC_UNUSED GwyDialog *dialog, gint response, ModuleGUI *gui)
{
    if (response == RESPONSE_RECALC) {
        recalc(gui);
    }
}

static void
recalc(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    GwyDataField *dfield = gwy_container_get_object(gui->data, gwy_app_get_data_key_for_id(0));
    gdouble xmin, xmax, ymin, ymax, xrange, yrange;
    gint newxres, newyres;

    execute(gui->args);

    gwy_surface_get_xrange(gui->args->result, &xmin, &xmax);
    gwy_surface_get_yrange(gui->args->result, &ymin, &ymax);
    xrange = xmax - xmin;
    yrange = ymax - ymin;

    if (xrange>=yrange) {
       newxres = PREVIEW_SIZE;
       newyres = yrange*newxres/xrange;
    } else {
       newyres = PREVIEW_SIZE;
       newxres = xrange*newyres/yrange;
    }

    gwy_data_field_resample(dfield, newxres, newyres, GWY_INTERPOLATION_NONE);
    gwy_data_field_set_xoffset(dfield, xmin);
    gwy_data_field_set_yoffset(dfield, ymin);
    gwy_data_field_set_xreal(dfield, xrange);
    gwy_data_field_set_yreal(dfield, yrange);

    gwy_data_field_average_xyz(dfield, NULL, 
                               gwy_surface_get_data_const(gui->args->result),
                               gwy_surface_get_npoints(gui->args->result));
    gwy_data_field_data_changed(dfield);

    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
}

static gdouble
get_curve_value_at(GwyGraphCurveModel *gc, gint level)
{
    const gdouble *ys = gwy_graph_curve_model_get_ydata(gc);
    gint n = gwy_graph_curve_model_get_ndata(gc);

    if (level < 0 || level > (n-1)) return 0;
    else return ys[level];
}

/*
   z optimisation:
   create a list of mutual average shifts and their projected areas
   1-2
   1-3
   1-5
   4-5

   create function that sets the offsets to minimize the area normalized mean shift
   apply the offsets

   */
static void
get_overlap(double moff, double noff, gint res, gint *start1, gint *start2, gint *size)
{
    int imoff = (int)moff;
    int inoff = (int)noff;

    if (inoff == imoff) {
        *start1 = 0;
        *start2 = 0;
        *size = res;
    }
    else if (inoff > imoff && (inoff-imoff)<res) {
        *start1 = inoff - imoff;
        *start2 = 0;
        *size = res - (inoff-imoff);
    }
    else if (inoff < imoff && (imoff-inoff)<res) {
        *start1 = 0;
        *start2 = imoff - inoff;
        *size = res - (imoff-inoff);
    }
    else *size = 0;
}
    
static void 
optimize_zshifts(gdouble **zs, gint **rzs, gdouble *zshift, gint zres)
{
    gint k;

    for (k = 0; k < zres; k++) zshift[k] = 0;

    //every next stitches to one with which it has biggest overlap?
    zshift[1] = zs[0][1];
    zshift[2] = zs[1][2] + zshift[1];
    zshift[3] = zs[0][3];
    zshift[4] = zs[3][4] + zshift[3];
    zshift[5] = zs[4][5] + zshift[4];
  

/*    for (m = 0; m < 1; m++) {
       for (n = m + 1; n < MIN(m+2, zres); n++) {
           if (rzs[m][n]>0) zshift[m] -= zs[m][n];
       }
    }
    */

}


static void
get_z_shifts(GwyBrick *brick, gdouble *xoff, gdouble *yoff, gdouble *zshift)
{
    gint k, m, n, col1, row1, col2, row2, width, height;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);
    GwyDataField *mfield, *nfield;
    gdouble **zs, mh, nh;
    gint **rzs;

    zs = g_new(gdouble*, zres);
    for (k = 0; k < zres; k++) zs[k] = g_new(gdouble, zres);

    rzs = g_new(gint*, zres);
    for (k = 0; k < zres; k++) rzs[k] = g_new(gint, zres);

    mfield = gwy_data_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);
    nfield = gwy_data_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);

    //find z shift between mth and nth plane
    for (m = 0; m < zres; m++) {
       gwy_brick_extract_xy_plane(brick, mfield, m);
       for (n = m + 1; n < zres; n++) {
          gwy_brick_extract_xy_plane(brick, nfield, n);

          get_overlap(gwy_brick_rtoi(brick, xoff[m]), gwy_brick_rtoi(brick, xoff[n]), xres, &col1, &col2, &width);
          get_overlap(gwy_brick_rtoj(brick, yoff[m]), gwy_brick_rtoj(brick, yoff[n]), yres, &row1, &row2, &height);

          /*printf("%d %d shifts %d %d   %d %d  overlap start: %d %d   %d %d   size %d %d\n", m, n, 
                 (gint)gwy_brick_rtoi(brick, xoff[m]), (gint)gwy_brick_rtoj(brick, yoff[m]),
                 (gint)gwy_brick_rtoi(brick, xoff[n]), (gint)gwy_brick_rtoj(brick, yoff[n]),
                 col1, row1, col2, row2, width, height);*/

          if (width>0 && height>0) 
          {
             mh = gwy_data_field_area_get_avg_mask(mfield, NULL, GWY_MASK_IGNORE,
                                                   col1, row1, width, height);
             nh = gwy_data_field_area_get_avg_mask(nfield, NULL, GWY_MASK_IGNORE,
                                                   col2, row2, width, height);

             zs[m][n] = mh - nh;
             rzs[m][n] = width*height;
          }
          else {
             zs[m][n] = 0;
             rzs[m][n] = 0;
          }
       }
    }

    for (m = 0; m < zres; m++) {
       for (n = m + 1; n < zres; n++) {
           if (rzs[m][n]>0) printf("(%d %d : %d %g) ", m, n, rzs[m][n], zs[m][n]);
       }
       printf("\n");
    }

    //calculate optimum z shifts
    optimize_zshifts(zs, rzs, zshift, zres);
    for (k = 0; k < zres; k++) printf("zshift %d %g\n", k, zshift[k]);
  
    for (k = 0; k < zres; k++) {
        g_free(zs[k]);
        g_free(rzs[k]);
    }
    g_free(zs);
    g_free(rzs);
    g_object_unref(mfield);
    g_object_unref(nfield);


}


static void
execute(ModuleArgs *args)
{
    GwyParams *params = args->params;
    gint k, m, n;
    GwyBrick *brick = args->brick;
    GwySurface *frame;
    GwyXYZ *xyz, *sxyz;
    GwyDataField *dfield;
    gdouble *xoffset, *yoffset, *zshift;
    gboolean xflip = gwy_params_get_boolean(params, PARAM_XOFFSETS_FLIP);
    gboolean yflip = gwy_params_get_boolean(params, PARAM_YOFFSETS_FLIP);
    GwyGraphCurveModel* gcmx = gwy_graph_model_get_curve(gwy_params_get_graph(params, PARAM_XOFFSETS),
                                                         gwy_params_get_int(params, PARAM_XOFFSETS_CURVE));
    GwyGraphCurveModel* gcmy = gwy_graph_model_get_curve(gwy_params_get_graph(params, PARAM_YOFFSETS),
                                                         gwy_params_get_int(params, PARAM_YOFFSETS_CURVE));

    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);

    xoffset = g_new(gdouble, zres);
    yoffset = g_new(gdouble, zres);
    zshift = g_new(gdouble, zres);

    if (args->result) g_object_unref(args->result);
    args->result = gwy_surface_new_sized(xres*yres*zres);
    xyz = gwy_surface_get_data(args->result);

    dfield = gwy_data_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);


    n = 0;
    for (k = 0; k < zres; k++) {
        if (xflip) {
            xoffset[k] = get_curve_value_at(gcmx, zres-k-1) - get_curve_value_at(gcmx, zres-1);
        } else {
            xoffset[k] = get_curve_value_at(gcmx, k) - get_curve_value_at(gcmx, 0);
        }
        if (yflip) {
            yoffset[k] = get_curve_value_at(gcmy, zres-k-1) - get_curve_value_at(gcmy, zres-1);
        } else {
            yoffset[k] = get_curve_value_at(gcmy, k) - get_curve_value_at(gcmy, 0);
        }
    }

    get_z_shifts(brick, xoffset, yoffset, zshift);

    for (k = 0; k < zres; k++) {
        gwy_brick_extract_xy_plane(brick, dfield, k);

        frame = gwy_surface_new();
        gwy_surface_set_from_data_field(frame, dfield);
        sxyz = gwy_surface_get_data(frame);
        for (m=0; m<(xres*yres); m++) {
            xyz[n].x = sxyz[m].x + xoffset[k];
            xyz[n].y = sxyz[m].y + yoffset[k];
            xyz[n].z = sxyz[m].z + zshift[k];
            n++;
        }
        g_object_unref(frame);
    }

    g_object_unref(dfield);
    g_free(xoffset);
    g_free(yoffset);
    g_free(zshift);
}

static void
sanitise_one_param(GwyParams *params, gint id, gint min, gint max, gint defval)
{
    gint v;

    v = gwy_params_get_int(params, id);
    if (v >= min && v <= max) {
        gwy_debug("param #%d is %d, i.e. within range [%d..%d]", id, v, min, max);
        return;
    }
    gwy_debug("param #%d is %d, setting it to the default %d", id, v, defval);
    gwy_params_set_int(params, id, defval);
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    //sanitise_one_param(params, PARAM_MAX_DEGREE, 0, MAX_DEGREE, 2);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
