/*
 *  $Id: volume_linestat.c 24975 2022-08-31 12:05:41Z yeti-dn $
 *  Copyright (C) 2015-2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwynlfit.h>
#include <libprocess/brick.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/gwyprocess.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/correlation.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-volume.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_SIZE = 360,
    /* 16 is good for current processors; increasing it to 32 might not hurt in the future. */
    BLOCK_SIZE = 16,
    MAX_PARAMS = 3,
    MAX_PLOT_DATA = 16384,
    PLOT_FUNC_SAMPLES = 241,
};

enum {
    PARAM_OUTPUT_TYPE,
    PARAM_DIR,
    PARAM_XDO,
    PARAM_YDO,
    PARAM_ZDO,
    PARAM_XRAW,
    PARAM_YRAW,
    PARAM_ZRAW,
    PARAM_XORDER,
    PARAM_YORDER,
    PARAM_ZORDER,
};

typedef enum {
    OUTPUT_DRIFT = 0,
    OUTPUT_CROP  = 1,
    NOUTPUTS
} DriftOutput;

typedef enum {
    DIR_X = 0,
    DIR_Y = 1,
    DIR_Z = 2,
    NDIRS
} DriftDir;

typedef struct {
    GwyParams *params;
    GwyBrick *brick;
    GwyBrick *result;
    gboolean xydata_done;
    gboolean zdata_done;
    gint xdrift_n;
    gdouble *xdrift_z;
    gdouble *xdrift_drift;
    gdouble *xdrift_fit;
    gint ydrift_n;
    gdouble *ydrift_z;
    gdouble *ydrift_drift;
    gdouble *ydrift_fit;
    gint zdrift_n;
    gdouble *zdrift_z;
    gdouble *zdrift_drift;
    gdouble *zdrift_fit;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table_quantity;
    GwyParamTable *table_options;
    GwyParamTable *table_x;
    GwyParamTable *table_y;
    GwyParamTable *table_z;
    GwyContainer *data;
    GwyGraphModel *gmodel;
    GwySelection *graph_selection;
} ModuleGUI;


static gboolean                    module_register        (void);
static GwyParamDef*                define_module_params   (void);
static void                        drift                  (GwyContainer *data,
                                                           GwyRunType runtype);
static void                        execute                (ModuleArgs *args);
static GwyDialogOutcome            run_gui                (ModuleArgs *args,
                                                           GwyContainer *data,
                                                           gint id);
static void                        param_changed          (ModuleGUI *gui,
                                                           gint id);
static void                        dialog_response        (GwyDialog *dialog,
                                                           gint response,
                                                           ModuleGUI *gui);
static void                        preview                (gpointer user_data);
static void                        update_image           (ModuleGUI *gui,
                                                           gint z);
static void                        calculate_xydrift_data (ModuleArgs *args,
                                                           GtkWindow *wait_window);
static void                        calculate_zdrift_data  (ModuleArgs *args,
                                                           GtkWindow *wait_window);
static void                        graph_selection_changed(ModuleGUI *gui,
                                                           gint id,
                                                           GwySelection *selection);
static void                        update_graph_curve     (ModuleGUI *gui);
static void                        sanitise_params        (ModuleArgs *args);


static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Estimates drift across stacked high-speed SPM frames."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek",
    "2023",
};

GWY_MODULE_QUERY2(module_info, volume_drift)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_drift",
                             (GwyVolumeFunc)&drift,
                             N_("/_Correct Data/Estimate _Drift..."),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Estimates drift"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum output_types[] = {
        { N_("_Extract drift"), OUTPUT_DRIFT,  },
        { N_("_Crop data"),     OUTPUT_CROP,   },
    };
    static const GwyEnum dirs[] = {
        { N_("_X"), DIR_X,   },
        { N_("_Y"), DIR_Y,   },
        { N_("_Z"), DIR_Z,   },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;


    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_volume_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_DIR, "dir", _("Show direction"),
                              dirs, G_N_ELEMENTS(dirs), DIR_Z);
    gwy_param_def_add_gwyenum(paramdef, PARAM_OUTPUT_TYPE, "output_type", _("Output type"),
                              output_types, G_N_ELEMENTS(output_types), OUTPUT_CROP);
    gwy_param_def_add_boolean(paramdef, PARAM_XDO, "xdo", "apply", FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_YDO, "ydo", "apply", FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_ZDO, "zdo", "apply", FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_XRAW, "xraw", "use raw data", FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_YRAW, "yraw", "use raw data", FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_ZRAW, "zraw", "use raw data", FALSE);
    gwy_param_def_add_int(paramdef, PARAM_XORDER, "xorder", "polynom order", 0, 10, 1);
    gwy_param_def_add_int(paramdef, PARAM_YORDER, "yorder", "polynom order", 0, 10, 1);
    gwy_param_def_add_int(paramdef, PARAM_ZORDER, "zorder", "polynom order", 0, 10, 1);
    return paramdef;
}

static void
drift(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyBrick *brick = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    DriftOutput output_type;
    gint oldid, newid;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_app_data_browser_get_current(GWY_APP_BRICK, &brick,
                                     GWY_APP_BRICK_ID, &oldid,
                                     0);
    g_return_if_fail(GWY_IS_BRICK(brick));
    args.result = NULL;
    args.brick = brick;
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitise_params(&args);

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args);

    output_type = gwy_params_get_enum(args.params, PARAM_OUTPUT_TYPE);

    if (output_type == OUTPUT_DRIFT) {
        GwyGraphModel *gmodel;
        GwyGraphCurveModel *gcmodel;

        if (!args.xydata_done)
            calculate_xydrift_data(&args, gwy_app_find_window_for_volume(data, oldid));
        if (!args.zdata_done)
            calculate_zdrift_data(&args, gwy_app_find_window_for_volume(data, oldid));

        gmodel = gwy_graph_model_new();
        g_object_set(gmodel,
                     "title", _("Lateral drift"),
                     "axis-label-left", _("Lateral drift"),
                     "axis-label-bottom", "slice level",
                     NULL);

        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel, "description", _("x-axis drift"), NULL);
        gwy_graph_curve_model_set_data(gcmodel, args.xdrift_z, args.xdrift_drift, args.xdrift_n);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel, "description", _("y-axis drift"), NULL);
        gwy_graph_curve_model_set_data(gcmodel, args.ydrift_z, args.ydrift_drift, args.ydrift_n);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
        gwy_app_data_browser_add_graph_model(gmodel, data, TRUE);
        g_object_unref(gmodel);

        gmodel = gwy_graph_model_new();
        g_object_set(gmodel,
                     "title", _("Z drift"),
                     "axis-label-left", _("Z drift"),
                     "axis-label-bottom", "slice level",
                     NULL);

        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel, "description", _("z-axis drift"), NULL);
        gwy_graph_curve_model_set_data(gcmodel, args.zdrift_z, args.zdrift_drift, args.zdrift_n);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
        gwy_app_data_browser_add_graph_model(gmodel, data, TRUE);
        g_object_unref(gmodel);
    } else {
        newid = gwy_app_data_browser_add_brick(args.result, NULL, data, TRUE);

        gwy_app_set_brick_title(data, newid, _("Drift corrected"));
        gwy_app_sync_volume_items(data, data, oldid, newid, FALSE,
                                  GWY_DATA_ITEM_GRADIENT,
                                  0);

        gwy_app_volume_log_add_volume(data, -1, newid);
    }

end:
    g_object_unref(args.params);
    g_object_unref(args.result);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyContainer *data, gint id)
{
    GtkWidget *hbox, *graph, *area, *dataview;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyGraphCurveModel *gcmodel;
    GwyDialogOutcome outcome;
    GwyBrick *brick = args->brick;
    GwyDataField *field = gwy_data_field_new(gwy_brick_get_xres(brick),
                                             gwy_brick_get_yres(brick),
                                             gwy_brick_get_xreal(brick),
                                             gwy_brick_get_yreal(brick),
                                             TRUE);
    const guchar *gradient;

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();

    args->xydata_done = FALSE;
    args->zdata_done = FALSE;
    args->xdrift_z = NULL;
    args->xdrift_drift = NULL;
    args->xdrift_fit = NULL;
    args->ydrift_z = NULL;
    args->ydrift_drift = NULL;
    args->ydrift_fit = NULL;
    args->zdrift_z = NULL;
    args->zdrift_drift = NULL;
    args->zdrift_fit = NULL;
    args->result = gwy_brick_duplicate(brick);

    gwy_container_set_object(gui.data, gwy_app_get_data_key_for_id(0), field);
    if (gwy_container_gis_string(data, gwy_app_get_brick_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(gui.data, gwy_app_get_data_palette_key_for_id(0), gradient);

    gui.dialog = gwy_dialog_new(_("Estimate drift"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(gui.data, 0, PREVIEW_SIZE, FALSE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);
    
    gui.gmodel = gwy_graph_model_new();
    g_object_set(gui.gmodel,
                 "label-visible", TRUE,
                 NULL);

    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel, "mode", GWY_GRAPH_CURVE_POINTS, "description", "drift", NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);
    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel, "mode", GWY_GRAPH_CURVE_LINE, "description", "fit", NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);


    graph = gwy_graph_new(gui.gmodel);
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gtk_widget_set_size_request(graph, PREVIEW_SIZE, PREVIEW_SIZE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    area = gwy_graph_get_area(GWY_GRAPH(graph));
    gwy_graph_area_set_status(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XLINES);
    gui.graph_selection = gwy_graph_area_get_selection(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XLINES);
    gwy_selection_set_max_objects(gui.graph_selection, 1);

    hbox = gwy_hbox_new(20);
    gwy_dialog_add_content(dialog, hbox, TRUE, TRUE, 4);

    table = gui.table_quantity = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_DIR);

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_x = gwy_param_table_new(args->params);
    gwy_param_table_append_header(table, -1, _("X correction"));
    gwy_param_table_append_checkbox(table, PARAM_XDO);
    gwy_param_table_append_checkbox(table, PARAM_XRAW);
    gwy_param_table_append_slider(table, PARAM_XORDER);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_y = gwy_param_table_new(args->params);
    gwy_param_table_append_header(table, -1, _("Y correction"));
    gwy_param_table_append_checkbox(table, PARAM_YDO);
    gwy_param_table_append_checkbox(table, PARAM_YRAW);
    gwy_param_table_append_slider(table, PARAM_YORDER);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_z = gwy_param_table_new(args->params);
    gwy_param_table_append_header(table, -1, _("Z correction"));
    gwy_param_table_append_checkbox(table, PARAM_ZDO);
    gwy_param_table_append_checkbox(table, PARAM_ZRAW);
    gwy_param_table_append_slider(table, PARAM_ZORDER);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);


    table = gui.table_options = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_OUTPUT_TYPE);
    gwy_param_table_append_separator(table);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    g_signal_connect_swapped(gui.table_quantity, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_x, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_y, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_z, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.graph_selection, "changed", G_CALLBACK(graph_selection_changed), &gui);
    g_signal_connect_after(dialog, "response", G_CALLBACK(dialog_response), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_UPON_REQUEST, preview, &gui, NULL);

    update_graph_curve(&gui);
    update_image(&gui, 0);
    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.data);
    g_object_unref(gui.gmodel);

    return outcome;
}


static void
param_changed(ModuleGUI *gui, gint id)
{
    if (id != PARAM_OUTPUT_TYPE)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));

    update_graph_curve(gui);
}

static void
dialog_response(G_GNUC_UNUSED GwyDialog *dialog, gint response, ModuleGUI *gui)
{
    if (response == GWY_RESPONSE_RESET)
        gwy_selection_clear(gui->graph_selection);
}

static void
calculate_zdrift_data(ModuleArgs *args, GtkWindow *wait_window)
{
    gint k;
    GwyBrick *brick = args->brick;
    gint xres = brick->xres;
    gint yres = brick->yres;
    gint zres = brick->zres;
    GwyDataField *dfield;
    gdouble zerodrift;

    gwy_app_wait_start(wait_window, _("Updating z drift data..."));

    if (args->zdrift_z)
        g_free(args->zdrift_z);
    if (args->zdrift_drift)
        g_free(args->zdrift_drift);
    if (args->zdrift_fit)
        g_free(args->zdrift_fit);

    args->zdrift_n = zres;
    args->zdrift_z = g_new(gdouble, zres);
    args->zdrift_drift = g_new(gdouble, zres);
    args->zdrift_fit = g_new(gdouble, zres);

    dfield = gwy_data_field_new(xres, yres, brick->xreal, brick->yreal, FALSE);

    for (k = 0; k < zres; k++) {
        gwy_brick_extract_plane(brick, dfield,
                                0, 0, k,
                                xres, yres, -1,
                                FALSE);
        args->zdrift_z[k] = k;
        if (k == 0) {
            args->zdrift_drift[k] = 0;
            zerodrift = gwy_data_field_get_avg(dfield);
        }
        else
            args->zdrift_drift[k] = gwy_data_field_get_avg(dfield) - zerodrift;

        if (!gwy_app_wait_set_fraction((gdouble)k/zres))
            break;
     }
    gwy_app_wait_finish();

    args->zdata_done = TRUE;

    g_object_unref(dfield);
}

static gdouble
polynom(gint order, gdouble *coeffs, gdouble x)
{
    int i;
    double sum, xpow;

    sum = coeffs[0];
    xpow = 1;
    for (i = 1; i <= order; i++) {
        xpow *= x;
        sum += coeffs[i]*xpow;
    }
    return sum;
}

static void
calculate_xydrift_data(ModuleArgs *args, GtkWindow *wait_window)
{
    gint i, j, k, n, xborder, yborder;
    gdouble xoffset, yoffset, xoff, yoff, maxscore, xsum, ysum, zvals[9];
    GwyBrick *brick = args->brick;
    gint xres = brick->xres;
    gint yres = brick->yres;
    gint zres = brick->zres;
    GwyDataField *dfield, *dfieldp, *score, *kernel;
    gwy_app_wait_start(wait_window, _("Updating lateral drift data..."));

    args->xdrift_n = zres;
    args->ydrift_n = zres;
    if (args->xdrift_z)
        g_free(args->xdrift_z);
    if (args->ydrift_z)
        g_free(args->ydrift_z);
    if (args->xdrift_drift)
        g_free(args->xdrift_drift);
    if (args->ydrift_drift)
        g_free(args->ydrift_drift);
    if (args->xdrift_fit)
        g_free(args->xdrift_fit);
    if (args->ydrift_fit)
        g_free(args->ydrift_fit);

    args->xdrift_z = g_new0(gdouble, zres);
    args->ydrift_z = g_new0(gdouble, zres);
    args->xdrift_drift = g_new0(gdouble, zres);
    args->ydrift_drift = g_new0(gdouble, zres);
    args->xdrift_fit = g_new0(gdouble, zres);
    args->ydrift_fit = g_new0(gdouble, zres);

    dfield = gwy_data_field_new(xres, yres, brick->xreal, brick->yreal, FALSE);
    dfieldp = gwy_data_field_new(xres, yres, brick->xreal, brick->yreal, FALSE);
    score = gwy_data_field_new_alike(dfieldp, FALSE);
    xborder = xres/10;
    yborder = yres/10;
    kernel = gwy_data_field_area_extract(dfield, xborder, yborder, xres - 2*xborder, yres - 2*yborder);

    xsum = 0;
    ysum = 0;
    args->xdrift_z[0] = 0;
    args->ydrift_z[0] = 0;

    args->xdrift_drift[0] = 0;
    args->ydrift_drift[0] = 0;

    for (k = 1; k < zres; k++) {
        gwy_brick_extract_plane(brick, dfieldp,
                                0, 0, k-1,
                                xres, yres, -1,
                                FALSE);
        gwy_brick_extract_plane(brick, dfield,
                                0, 0, k,
                                xres, yres, -1,
                                FALSE);

        gwy_data_field_area_copy(dfield, kernel, xborder, yborder, xres - 2*xborder, yres - 2*yborder, 0, 0);
        gwy_data_field_correlation_search (dfieldp, kernel, NULL, score, GWY_CORRELATION_POC,
                                   0.1, GWY_EXTERIOR_MIRROR_EXTEND, 0);

        //remove all the far values
        //gwy_data_field_area_clear(score, 0, 0, xres/2 - xborder + 1, yres);
        //gwy_data_field_area_clear(score, 0, 0, xres, yres/2 - yborder + 1);
        //gwy_data_field_area_clear(score, xres/2 + xborder - 1, 0, xres/2 - xborder + 1, yres);
        //gwy_data_field_area_clear(score, 0, yres/2 + yborder - 1, xres, yres/2 - yborder + 1);

//        for (si=0; si<xres; si++) {
//            for (sj=0; sj<yres; sj++) {
//                printf("%g ", gwy_data_field_get_val(score, si, sj));
//            }
//            printf("\n");
//        }

        if (gwy_data_field_get_local_maxima_list(score, &xoff, &yoff, &maxscore, 1, 0, 0.0, FALSE)) {
            //printf("%g %g  %d %d  x %g  y %g score %g\n", xoff, yoff, xres/2, yres/2, 
            //       xoff-xres/2, yoff-yres/2, maxscore);
            if (xoff > 0 && xoff < (xres - 1) && yoff > 0 && yoff < (yres - 1)) {
                n = 0;
                for (i = -1; i <= 1; i++) {
                    for (j = -1; j <= 1; j++) {
                        zvals[n++] = gwy_data_field_get_val(score, xoff+i, yoff+j);
                    }
                }
                gwy_math_refine_maximum_2d(zvals, &xoffset, &yoffset);
                xoffset += xoff - xres/2;
                yoffset += yoff - yres/2;
            } else {
                xoffset = GWY_ROUND(xoff) - xres/2;
                yoffset = GWY_ROUND(yoff) - yres/2;
            }
        }
        else
            xoffset = yoffset = 0;

        xsum += xoffset;
        ysum += yoffset;
        args->xdrift_z[k] = k;
        args->ydrift_z[k] = k;
        args->xdrift_drift[k] = xsum;
        args->ydrift_drift[k] = ysum;

        if (!gwy_app_wait_set_fraction((gdouble)k/zres))
            break;
    }
    gwy_app_wait_finish();
    args->xydata_done = TRUE;

    g_object_unref(dfield);
    g_object_unref(dfieldp);
    g_object_unref(score);
    g_object_unref(kernel);
}

static void
fit_polynom(gint order, gboolean raw, gdouble *z, gdouble *drift, gdouble *fit, gint n)
{
    gint k;
    gdouble *coeffs;

    if (raw) {
       for (k = 0;  k < n; k++)
           fit[k] = drift[k];
    } else {
       coeffs = gwy_math_fit_polynom(n, z, drift, order, NULL);
       for (k = 0; k < n; k++)
           fit[k] = polynom(order, coeffs, k);

       g_free(coeffs);
   }
}

static void
update_graph_curve(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    gint dir = gwy_params_get_enum(params, PARAM_DIR);
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    GwyGraphCurveModel *fitmodel = gwy_graph_model_get_curve(gui->gmodel, 1);
    gboolean xraw = gwy_params_get_boolean(args->params, PARAM_XRAW);
    gboolean yraw = gwy_params_get_boolean(args->params, PARAM_YRAW);
    gboolean zraw = gwy_params_get_boolean(args->params, PARAM_ZRAW);
    gint xorder = gwy_params_get_int(args->params, PARAM_XORDER);
    gint yorder = gwy_params_get_int(args->params, PARAM_YORDER);
    gint zorder = gwy_params_get_int(args->params, PARAM_ZORDER);
    gint zres = gwy_brick_get_zres(args->brick);

    if (dir == DIR_X || dir == DIR_Y) {
        if (!args->xydata_done)
            calculate_xydrift_data(args, GTK_WINDOW(gui->dialog));

        fit_polynom(xorder, xraw, args->xdrift_z, args->xdrift_drift, args->xdrift_fit, zres);
        fit_polynom(yorder, yraw, args->ydrift_z, args->ydrift_drift, args->ydrift_fit, zres);

        if (dir == DIR_X) {
            gwy_graph_curve_model_set_data(gcmodel, args->xdrift_z, args->xdrift_drift, args->xdrift_n);
            gwy_graph_curve_model_set_data(fitmodel, args->xdrift_z, args->xdrift_fit, args->xdrift_n);
        }
        else {
            gwy_graph_curve_model_set_data(gcmodel, args->ydrift_z, args->ydrift_drift, args->ydrift_n);
            gwy_graph_curve_model_set_data(fitmodel, args->ydrift_z, args->ydrift_fit, args->ydrift_n);
        }
    }
    else
    {
        if (!args->zdata_done)
            calculate_zdrift_data(args, GTK_WINDOW(gui->dialog));

        fit_polynom(zorder, zraw, args->zdrift_z, args->zdrift_drift, args->zdrift_fit, zres);

        gwy_graph_curve_model_set_data(gcmodel, args->zdrift_z, args->zdrift_drift, args->zdrift_n);
        gwy_graph_curve_model_set_data(fitmodel, args->zdrift_z, args->zdrift_fit, args->zdrift_n);
    }
}

static void
update_image(ModuleGUI *gui, gint z)
{
    GwyDataField *dfield;
    GwyBrick *brick = gui->args->result;
    dfield = gwy_container_get_object(gui->data, gwy_app_get_data_key_for_id(0));

    gwy_brick_extract_xy_plane(brick, dfield, CLAMP(z, 0, brick->zres-1));

    gwy_data_field_data_changed(dfield);
}

static void
graph_selection_changed(ModuleGUI *gui, G_GNUC_UNUSED gint id, GwySelection *selection)
{
    gdouble z;

    if (!gwy_selection_get_object(selection, 0, &z))
        z =  0;
    update_image(gui, z);
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;

    if (!gui->args->xydata_done)
        calculate_xydrift_data(gui->args, GTK_WINDOW(gui->dialog));
    if (!gui->args->zdata_done)
        calculate_zdrift_data(gui->args, GTK_WINDOW(gui->dialog));

    execute(gui->args);
    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
}

static void
execute(ModuleArgs *args)
{
    GwyParams *params = args->params;
    gint k;
    gboolean xdo = gwy_params_get_boolean(params, PARAM_XDO);
    gboolean ydo = gwy_params_get_boolean(params, PARAM_YDO);
    gboolean zdo = gwy_params_get_boolean(params, PARAM_ZDO);
    gint xc, yc, xs, ys;
    GwyBrick *original = args->brick;
    GwyBrick *brick = args->result;
    GwyDataField *dfield;
    GwyDataField *cropfield;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);

    xc = args->xdrift_fit[zres/2];
    yc = args->ydrift_fit[zres/2];

    dfield = gwy_data_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);
    cropfield = gwy_data_field_new_alike(dfield, FALSE);

    for (k = 0; k < zres; k++) {
        gwy_brick_extract_xy_plane(original, dfield, k);
        gwy_data_field_fill(cropfield, gwy_data_field_get_avg(dfield));

        if (xdo)
            xs = args->xdrift_fit[k] - xc;
        else
            xs = 0;
        if (ydo)
            ys = args->ydrift_fit[k] - yc;
        else
            ys = 0;

        gwy_data_field_area_copy(dfield, cropfield, 0, 0, xres, yres, xs, ys);

        if (zdo)
            gwy_data_field_add(cropfield, -args->zdrift_fit[k]);

        gwy_brick_set_xy_plane(brick, cropfield, k);
    }

    g_object_unref(dfield);
    g_object_unref(cropfield);
}

static void
sanitise_one_param(GwyParams *params, gint id, gint min, gint max, gint defval)
{
    gint v;

    v = gwy_params_get_int(params, id);
    if (v >= min && v <= max) {
        gwy_debug("param #%d is %d, i.e. within range [%d..%d]", id, v, min, max);
        return;
    }
    gwy_debug("param #%d is %d, setting it to the default %d", id, v, defval);
    gwy_params_set_int(params, id, defval);
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;

    sanitise_one_param(params, PARAM_XORDER, 0, 10, 0);
    sanitise_one_param(params, PARAM_YORDER, 0, 10, 0);
    sanitise_one_param(params, PARAM_ZORDER, 0, 10, 0);
}



/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
