/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2020-04-11
 * Description : A tab to display information about the current selection.
 *
 * SPDX-FileCopyrightText: 2020 by Kartik Ramesh <kartikx2000 at gmail dot com>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

#include "itemselectionpropertiestab.h"

// Qt includes

#include <QApplication>
#include <QGridLayout>
#include <QStyle>
#include <QIcon>

// KDE includes

#include <klocalizedstring.h>

// Local includes

#include "itempropertiestab.h"
#include "itempropertiestxtlabel.h"

namespace Digikam
{

class Q_DECL_HIDDEN ItemSelectionPropertiesTab::Private
{
public:

    enum Section
    {
        SelectionProperties = 0
    };

public:

    Private() = default;

    DTextLabelValue* labelSelectionCount = nullptr;
    DTextLabelValue* labelSelectionSize  = nullptr;
    DTextLabelValue* labelTotalCount     = nullptr;
    DTextLabelValue* labelTotalSize      = nullptr;
};

ItemSelectionPropertiesTab::ItemSelectionPropertiesTab(QWidget* const parent)
    : DExpanderBox(parent),
      d           (new Private)
{
    setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    setLineWidth(style()->pixelMetric(QStyle::PM_DefaultFrameWidth));

    // --------------------------------------------------

    const int spacing = qMin(QApplication::style()->pixelMetric(QStyle::PM_LayoutHorizontalSpacing),
                             QApplication::style()->pixelMetric(QStyle::PM_LayoutVerticalSpacing));

    QWidget* const w1                    = new QWidget(this);
    QGridLayout* const glay1             = new QGridLayout(w1);

    DTextLabelName* const selectionCount = new DTextLabelName(i18n("Count: "),       w1);
    DTextLabelName* const selectionSize  = new DTextLabelName(i18n("Size: "),        w1);
    DTextLabelName* const totalCount     = new DTextLabelName(i18n("Total Count: "), w1);
    DTextLabelName* const totalSize      = new DTextLabelName(i18n("Total Size: "),  w1);


    d->labelSelectionCount               = new DTextLabelValue(QString(), w1);
    d->labelSelectionSize                = new DTextLabelValue(QString(), w1);
    d->labelTotalCount                   = new DTextLabelValue(QString(), w1);
    d->labelTotalSize                    = new DTextLabelValue(QString(), w1);

    glay1->addWidget(selectionCount,         0, 0, 1, 1);
    glay1->addWidget(d->labelSelectionCount, 0, 1, 1, 1);
    glay1->addWidget(selectionSize,          1, 0, 1, 1);
    glay1->addWidget(d->labelSelectionSize,  1, 1, 1, 1);
    glay1->addWidget(totalCount,             2, 0, 1, 1);
    glay1->addWidget(d->labelTotalCount,     2, 1, 1, 1);
    glay1->addWidget(totalSize,              3, 0, 1, 1);
    glay1->addWidget(d->labelTotalSize,      3, 1, 1, 1);


    glay1->setContentsMargins(spacing, spacing, spacing, spacing);
    glay1->setColumnStretch(0, 10);
    glay1->setColumnStretch(1, 25);
    glay1->setSpacing(0);

    insertItem(ItemSelectionPropertiesTab::Private::SelectionProperties, w1,
               QIcon::fromTheme(QLatin1String("dialog-information")),
               i18n("Selection Properties"), QLatin1String("Selection Properties"), true);

    // --------------------------------------------------

    addStretch();
}

ItemSelectionPropertiesTab::~ItemSelectionPropertiesTab()
{
    delete d;
}

void ItemSelectionPropertiesTab::setCurrentURL(const QUrl& url)
{
    if (url.isEmpty())
    {
         d->labelSelectionCount->setAdjustedText(QString());
         d->labelSelectionSize->setAdjustedText(QString());
         d->labelTotalCount->setAdjustedText(QString());
         d->labelTotalSize->setAdjustedText(QString());
         setEnabled(false);
         return;
    }

    setEnabled(true);
}

void ItemSelectionPropertiesTab::setSelectionCount(const QString& str)
{
    d->labelSelectionCount->setAdjustedText(str);
}

void ItemSelectionPropertiesTab::setSelectionSize(const QString& str)
{
    d->labelSelectionSize->setAdjustedText(str);
}

void ItemSelectionPropertiesTab::setTotalCount(const QString& str)
{
    d->labelTotalCount->setAdjustedText(str);
}

void ItemSelectionPropertiesTab::setTotalSize(const QString& str)
{
    d->labelTotalSize->setAdjustedText(str);
}

} // namespace Digikam

#include "moc_itemselectionpropertiestab.cpp"
