/* bender-tags: editor,clipboard,widget */
/* bender-ckeditor-plugins: uploadwidget,uploadimage,toolbar,image2 */
/* bender-include: %BASE_PATH%/plugins/clipboard/_helpers/pasting.js */
/* bender-include: %BASE_PATH%/plugins/uploadfile/_helpers/waitForImage.js */
/* global pasteFiles, waitForImage */

'use strict';

( function() {
	var uploadCount, loadAndUploadCount, lastUploadUrl, resumeAfter,
		IMG_URL = '%BASE_PATH%_assets/logo.png',
		LOADING_IMG = 'data:image/gif;base64,R0lGODlhDgAOAIAAAAAAAP///yH5BAAAA',
		LOADED_IMG = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABC';

	bender.editors = {
		classic: {
			name: 'classic',
			creator: 'replace',
			config: {
				extraPlugins: 'uploadimage,image',
				removePlugins: 'image2',
				imageUploadUrl: 'http://foo/upload',
				// Disable pasteFilter on Webkits (pasteFilter defaults semantic-text on Webkits).
				pasteFilter: null
			}
		},
		classicSupportingWebp: {
			name: 'classicSupportingWebp',
			creator: 'replace',
			config: {
				extraPlugins: 'uploadimage,image',
				removePlugins: 'image2',
				imageUploadUrl: 'http://foo/upload',
				uploadImage_supportedTypes: /image\/(jpeg|png|gif|bmp|webp)/,
				// Disable pasteFilter on Webkits (pasteFilter defaults semantic-text on Webkits).
				pasteFilter: null
			}
		},
		inline: {
			name: 'inline',
			creator: 'inline',
			config: {
				extraPlugins: 'uploadimage,image2',
				filebrowserImageUploadUrl: 'http://foo/upload?type=Images',
				// Disable pasteFilter on Webkits (pasteFilter defaults semantic-text on Webkits).
				pasteFilter: null
			}
		}
	};

	function assertUploadingWidgets( editor, expectedSrc ) {
		var widgets = editor.editable().find( 'img[data-widget="uploadimage"]' ),
			widget, i;

		assert.areSame( 1, widgets.count(), 'Expected widgets count should be 1' );

		for ( i = 0; i < widgets.count(); i++ ) {
			widget = widgets.getItem( i );
			assert.areSame( '0', widget.getAttribute( 'data-cke-upload-id' ) );
			assert.areSame( expectedSrc, widget.getAttribute( 'src' ).substring( 0, 55 ) );
		}
	}

	bender.test( {
		init: function() {
			resumeAfter = bender.tools.resumeAfter;

			CKEDITOR.fileTools.fileLoader.prototype.loadAndUpload = function( url ) {
				loadAndUploadCount++;
				lastUploadUrl = url;

				this.responseData = {};
			};

			CKEDITOR.fileTools.fileLoader.prototype.load = function() {};

			CKEDITOR.fileTools.fileLoader.prototype.upload = function( url ) {
				uploadCount++;
				lastUploadUrl = url;

				this.responseData = {};
			};
		},

		setUp: function() {
			bender.tools.ignoreUnsupportedEnvironment( 'uploadimage' );

			var editorName;

			uploadCount = 0;
			loadAndUploadCount = 0;

			for ( editorName in this.editors ) {
				// Clear upload repository.
				this.editors[ editorName ].uploadRepository.loaders = [];
			}

			if ( CKEDITOR.fileTools.bindNotifications.reset ) {
				CKEDITOR.fileTools.bindNotifications.reset();
			}
		},

		'test classic with image1 (integration test)': function() {
			var editor = this.editors.classic;

			pasteFiles( editor, [ bender.tools.getTestPngFile() ] );

			assertUploadingWidgets( editor, LOADING_IMG );
			assert.areSame( '', editor.getData(), 'getData on loading.' );

			var loader = editor.uploadRepository.loaders[ 0 ];

			loader.data = bender.tools.pngBase64;
			loader.uploadTotal = 10;
			loader.changeStatus( 'uploading' );

			assertUploadingWidgets( editor, LOADED_IMG );
			assert.areSame( '', editor.getData(), 'getData on uploading.' );

			var image = editor.editable().find( 'img[data-widget="uploadimage"]' ).getItem( 0 );

			waitForImage( image, function() {
				loader.url = IMG_URL;
				loader.changeStatus( 'uploaded' );

				assert.sameData( '<p><img src="' + IMG_URL + '" style="height:1px; width:1px" /></p>', editor.getData() );
				assert.areSame( 0, editor.editable().find( 'img[data-widget="image"]' ).count() );

				assert.areSame( 1, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
				assert.areSame( 'http://foo/upload', lastUploadUrl );
			} );
		},

		'test finish upload notification marked as important and is visible (https://dev.ckeditor.com/ticket/13032).': function() {
			var editor = this.editors.classic;

			pasteFiles( editor, [ bender.tools.getTestPngFile() ] );

			var loader = editor.uploadRepository.loaders[ 0 ];

			loader.data = bender.tools.pngBase64;
			loader.uploadTotal = 10;
			loader.changeStatus( 'uploading' );

			var area = editor._.notificationArea;

			// Closing notification.
			area.notifications[ 0 ].hide();

			assertUploadingWidgets( editor, LOADED_IMG );

			var image = editor.editable().find( 'img[data-widget="uploadimage"]' ).getItem( 0 );

			waitForImage( image, function() {
				loader.url = IMG_URL;
				loader.changeStatus( 'uploaded' );

				assert.areSame( 1, area.notifications.length, 'Successs notification is present because it\'s important one.' );
				assert.areSame( 'success', area.notifications[ 0 ].type );
			} );
		},

		'test inline with image2 (integration test)': function() {
			var editor = this.editors.inline;

			pasteFiles( editor, [ bender.tools.getTestPngFile() ] );

			assertUploadingWidgets( editor, LOADING_IMG );
			assert.areSame( '', editor.getData(), 'getData on loading.' );

			var loader = editor.uploadRepository.loaders[ 0 ];

			loader.data = bender.tools.pngBase64;
			loader.changeStatus( 'uploading' );

			assertUploadingWidgets( editor, LOADED_IMG );
			assert.areSame( '', editor.getData(), 'getData on uploading.' );

			var image = editor.editable().find( 'img[data-widget="uploadimage"]' ).getItem( 0 );

			waitForImage( image, function() {
				loader.url = IMG_URL;
				loader.changeStatus( 'uploaded' );

				assert.sameData( '<p><img alt="" height="1" src="' + IMG_URL + '" width="1" /></p>', editor.getData() );
				assert.areSame( 1, editor.editable().find( 'img[data-widget="image"]' ).count() );

				assert.areSame( 1, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
				assert.areSame( 'http://foo/upload?type=Images&responseType=json', lastUploadUrl );
			} );
		},

		'test paste img as html (integration test)': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				pasteFiles( editor, [], '<p>x<img src="' + bender.tools.pngBase64 + '">x</p>' );

				assertUploadingWidgets( editor, LOADED_IMG );
				assert.areSame( '<p>xx</p>', editor.getData(), 'getData on loading.' );

				var loader = editor.uploadRepository.loaders[ 0 ];

				loader.data = bender.tools.pngBase64;
				loader.changeStatus( 'uploading' );

				assertUploadingWidgets( editor, LOADED_IMG );
				assert.areSame( '<p>xx</p>', editor.getData(), 'getData on uploading.' );

				var image = editor.editable().find( 'img[data-widget="uploadimage"]' ).getItem( 0 );

				waitForImage( image, function() {
					loader.url = IMG_URL;
					loader.changeStatus( 'uploaded' );

					assert.sameData( '<p>x<img src="' + IMG_URL + '" style="height:1px; width:1px" />x</p>', editor.getData() );
					assert.areSame( 0, editor.editable().find( 'img[data-widget="image"]' ).count() );

					assert.areSame( 0, loadAndUploadCount );
					assert.areSame( 1, uploadCount );
					assert.areSame( 'http://foo/upload', lastUploadUrl );
				} );
			} );
		},

		'test setting image dimensions via response (integration test) (https://dev.ckeditor.com/ticket/13794)': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				pasteFiles( editor, [ bender.tools.getTestPngFile() ] );

				var loader = editor.uploadRepository.loaders[ 0 ];

				loader.data = bender.tools.pngBase64;
				loader.uploadTotal = 10;
				loader.changeStatus( 'uploading' );

				loader.responseData.width = 555;
				loader.responseData.height = 444;

				resumeAfter( loader, 'uploaded', function() {
					assert.sameData( '<p><img src="' + IMG_URL + '" style="height:444px; width:555px" /></p>', editor.getData() );
					assert.areSame( 0, editor.editable().find( 'img[data-widget="image"]' ).count() );
				} );

				loader.url = IMG_URL;
				loader.changeStatus( 'uploaded' );

				wait();
			} );
		},

		'test supportedTypes png': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assertUploadingWidgets( editor, LOADING_IMG );
				} );

				pasteFiles( editor, [ { name: 'test.png', type: 'image/png' } ] );

				wait();
			} );
		},

		'test supportedTypes jpg': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assertUploadingWidgets( editor, LOADING_IMG );
				} );

				pasteFiles( editor, [ { name: 'test.jpg', type: 'image/jpeg' } ] );

				wait();
			} );
		},

		'test supportedTypes gif': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assertUploadingWidgets( editor, LOADING_IMG );
				} );

				pasteFiles( editor, [ { name: 'test.gif', type: 'image/gif' } ] );

				wait();
			} );
		},

		'test supportedTypes bmp': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assertUploadingWidgets( editor, LOADING_IMG );
				} );

				pasteFiles( editor, [ { name: 'test.bmp', type: 'image/bmp' } ] );

				wait();
			} );
		},

		// (#4400)
		'test not supportedTypes webp': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assert.areSame( 0, editor.editable().find( 'img[data-widget="uploadimage"]' ).count() );
				} );

				pasteFiles( editor, [ { name: 'test.webp', type: 'image/webp' } ] );

				wait();
			} );
		},

		'test not supportedTypes tiff': function() {
			var bot = this.editorBots.classic,
				editor = this.editors.classic;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assert.areSame( 0, editor.editable().find( 'img[data-widget="uploadimage"]' ).count() );
				} );

				pasteFiles( editor, [ { name: 'test.tiff', type: 'image/tiff' } ] );

				wait();
			} );
		},

		// (#4400)
		'test setting config.uploadImage_supportedTypes allows uploading webp images': function() {
			var bot = this.editorBots.classicSupportingWebp,
				editor = this.editors.classicSupportingWebp;

			bot.setData( '', function() {
				resumeAfter( editor, 'paste', function() {
					assertUploadingWidgets( editor, LOADING_IMG );
				} );

				pasteFiles( editor, [ { name: 'test.webp', type: 'image/webp' } ] );

				wait();
			} );
		},

		'test paste single image': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue );

				assert.areSame( '0', img.getAttribute( 'data-cke-upload-id' ) );
				assert.areSame( 'uploadimage', img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 1, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue: '<img src="' + bender.tools.pngBase64 + '">'
			} );

			wait();
		},

		'test paste nested image': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var imgs = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue ).find( 'img[data-widget="uploadimage"]' ),
					img, i;

				assert.areSame( 2, imgs.count(), 'Expected imgs count should be 2' );

				for ( i = 0; i < imgs.count(); i++ ) {
					img = imgs.getItem( i );
					assert.areSame( i + '', img.getAttribute( 'data-cke-upload-id' ) );
				}

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 2, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue: '<div>x<img src="' + bender.tools.pngBase64 + '">x' +
							'<p>x<img src="' + bender.tools.pngBase64 + '">x</p></div>'
			} );

			wait();
		},

		'test paste no image': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				assert.areSame( 'foo', evt.data.dataValue );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue: 'foo'
			} );

			wait();
		},

		'test paste no data in image': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue );

				assert.isNull( img.getAttribute( 'data-cke-upload-id' ) );
				assert.isNull( img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue: '<img src="' + IMG_URL + '">'
			} );

			wait();
		},

		'test paste image already marked': function() {
			var editor = this.editors.classic,
				uploads = editor.uploadRepository;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue );

				assert.areSame( '0', img.getAttribute( 'data-cke-upload-id' ) );
				assert.areSame( 'uploadimage', img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
			} );

			// Fill upload repository.
			uploads.create( bender.tools.getTestPngFile() );

			editor.fire( 'paste', {
				dataValue: '<img src="' + bender.tools.pngBase64 + '" data-widget="uploadimage" data-cke-upload-id="0">'
			} );

			wait();
		},

		'test omit images in non contentEditable': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue ).findOne( 'img' );

				assert.isNull( img.getAttribute( 'data-cke-upload-id' ) );
				assert.isNull( img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 0, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue:
					'<div contentEditable="false">' +
						'<img src="' + bender.tools.pngBase64 + '">' +
					'</div>'
			} );

			wait();
		},

		'test handle images in nested editable': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue ).findOne( 'img' );

				assert.areSame( '0', img.getAttribute( 'data-cke-upload-id' ) );
				assert.areSame( 'uploadimage', img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 1, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue:
					'<div contentEditable="false">' +
						'<div contentEditable="true">' +
							'<img src="' + bender.tools.pngBase64 + '">' +
						'</div>' +
					'</div>'
			} );

			wait();
		},

		'test handle images in nested editable using cke-editable': function() {
			var editor = this.editors.classic;

			resumeAfter( editor, 'paste', function( evt ) {
				var img = CKEDITOR.dom.element.createFromHtml( evt.data.dataValue ).findOne( 'img' );

				assert.areSame( '0', img.getAttribute( 'data-cke-upload-id' ) );
				assert.areSame( 'uploadimage', img.getAttribute( 'data-widget' ) );

				assert.areSame( 0, loadAndUploadCount );
				assert.areSame( 1, uploadCount );
			} );

			editor.fire( 'paste', {
				dataValue:
					'<div contentEditable="false">' +
						'<div data-cke-editable="1">' +
							'<img src="' + bender.tools.pngBase64 + '">' +
						'</div>' +
					'</div>'
			} );

			wait();
		},

		'test bindNotifications when paste image': function() {
			var editor = this.editors.classic;

			CKEDITOR.fileTools.bindNotifications = sinon.spy();

			resumeAfter( editor, 'paste', function() {
				var spy = CKEDITOR.fileTools.bindNotifications;
				assert.areSame( 1, spy.callCount );
				assert.isTrue( spy.calledWith( editor ) );
				assert.areSame( bender.tools.pngBase64, spy.firstCall.args[ 1 ].data );
			} );

			editor.fire( 'paste', {
				dataValue: '<img src="' + bender.tools.pngBase64 + '">'
			} );

			wait();
		},

		'test XSS attack': function() {
			var editor = this.editors.inline;

			window.attacked = sinon.spy();

			editor.fire( 'paste', {
				dataValue: '<img src="x" onerror="window.attacked();">' + bender.tools.pngBase64
			} );

			editor.once( 'afterPaste', function() {
				resume( function() {
					assert.areSame( 0, window.attacked.callCount );
				} );
			} );

			wait();
		},

		'test prevent upload fake elements (https://dev.ckeditor.com/ticket/13003)': function() {
			var editor = this.editors.inline,
				createSpy = sinon.spy( editor.uploadRepository, 'create' );

			editor.fire( 'paste', {
				dataValue: '<img src="data:image/gif;base64,aw==" alt="nothing" data-cke-realelement="some" />'
			} );

			editor.once( 'afterPaste', function() {
				resume( function() {
					createSpy.restore();
					assert.isTrue( createSpy.notCalled );
				} );
			} );

			wait();
		},

		'test uploads generate unique names (#1213)': function() {
			var editor = this.editors.inline,
				createSpy = sinon.spy( editor.uploadRepository, 'create' );

			editor.fire( 'paste', {
				dataValue: '<img src="data:image/gif;base64,aw==" alt="gif" />' +
					'<img src="data:image/gif;base64,aw==" alt="gif" />' +
					'<img src="data:image/png;base64,aw==" alt="png" />'
			} );

			editor.once( 'afterPaste', function() {
				resume( function() {
					createSpy.restore();
					assert.areSame( 3, createSpy.callCount, 'create call count' );

					assert.isMatching( /image-\d+-\d+\.gif/, createSpy.args[ 0 ][ 1 ], 'file name passed to first call' );
					assert.isMatching( /image-\d+-\d+\.gif/, createSpy.args[ 1 ][ 1 ], 'file name passed to second call' );
					assert.areNotSame( createSpy.args[ 0 ][ 1 ], createSpy.args[ 1 ][ 1 ], 'first and second call names are different' );
					assert.isMatching( /image-\d+-\d+\.png/, createSpy.args[ 2 ][ 1 ], 'png type is recognized' );
				} );
			} );

			wait();
		},

		'test no error if missing configuration': function() {
			var spy = sinon.spy( CKEDITOR, 'error' );

			bender.editorBot.create( {
				name: 'configerror_test',
				config: {
					extraPlugins: 'uploadimage'
				}
			}, function( bot ) {
				spy.restore();

				assert.areSame( 0, spy.callCount, 'CKEDITOR.error call count' );
				assert.isFalse( !!bot.editor.widgets.registered.uploadimage, 'uploadimage widget' );
			} );
		},

		// (#5333)
		'test warning about disabling the clipboard image handling logic (default config value)': createHandlingImageWarnTest(
			'clipboard-warning-default' ),

		// (#5333)
		'test warning about disabling the clipboard image handling logic (config value of true)': createHandlingImageWarnTest(
			'clipboard-warning-true', true ),

		// (#5333)
		'test warning about disabling the clipboard image handling logic (config value of false)': createHandlingImageWarnTest(
			'clipboard-warning-false', false ),

		// (#5333)
		'test original file name is preserved after the upload': function() {
			bender.editorBot.create( {
				name: 'clipboard-integration-original-file-name',
				config: {
					uploadUrl: '%BASE_PATH',
					extraPlugins: 'uploadimage'
				}
			}, function( bot ) {
				var editor = bot.editor,
					imageName = 'test.png',
					image = {
						name: imageName,
						type: 'image/png'
					};

				bot.setData( '', function() {
					resumeAfter( editor, 'paste', function() {
						var widget = CKEDITOR.tools.object.values( editor.widgets.instances )[ 0 ],
							loader = widget._getLoader();

						assert.areSame( imageName, loader.fileName, 'The name of the uploaded file' );
					} );

					pasteFilesWithFilesMimeType( editor, [ image ] );

					wait();
				} );
			} );
		},

		// (#5414)
		'test change event is fired after upload finishes': function() {
			bender.editorBot.create( {
				name: 'undo-integration-change-after-upload',
				config: {
					uploadUrl: '%BASE_PATH',
					extraPlugins: 'uploadimage'
				}
			}, function( bot ) {
				var editor = bot.editor,
					imageName = 'test.png',
					image = {
						name: imageName,
						type: 'image/png'
					},
					loader;

				bot.setData( '', function() {
					editor.once( 'change', function( evt ) {
						resume( function() {
							var editorData = evt.editor.getData(),
								containsUploadedImageUrl = editorData.indexOf( 'src="' + IMG_URL ) !== -1;

							assert.isTrue( containsUploadedImageUrl );
						} );
					} );

					pasteFilesWithFilesMimeType( editor, [ image ] );

					loader = editor.uploadRepository.loaders[ 0 ];

					loader.url = IMG_URL;
					loader.changeStatus( 'uploaded' );

					wait();
				} );
			} );
		}
	} );

	function createHandlingImageWarnTest( editorName, configValue ) {
		return function() {
			var spy = sinon.spy( CKEDITOR, 'warn' ),
				editorConfig =  {
					uploadUrl: '%BASE_PATH',
					extraPlugins: 'uploadimage'
				};

			if ( configValue !== undefined ) {
				editorConfig.clipboard_handleImages = configValue;
			}

			bender.editorBot.create( {
				name: editorName,
				config: editorConfig
			}, function() {
				var warnCalls = spy.args,
					expectedWarnDetails = {
						editor: editorName,
						plugin: 'uploadimage'
					};

				spy.restore();

				var warningCalled = spy.calledWith( 'clipboard-image-handling-disabled', expectedWarnDetails );

				if ( configValue === false ) {
					assert.isFalse( warningCalled, 'CKEDITOR.warn should not be called' );
				} else {
					assert.isTrue( warningCalled, 'CKEDITOR.warn should be called' );

					var warningDetails = CKEDITOR.tools.array.find( warnCalls, function( item ) {
						return item[ 0 ] === 'clipboard-image-handling-disabled';
					} );

					objectAssert.areDeepEqual( expectedWarnDetails, warningDetails[ 1 ],
						'CKEDITOR.warn should include proper details' );
				}
			} );
		};
	}

	function pasteFilesWithFilesMimeType( editor, files, pasteMethod ) {
		var	nativeData = bender.tools.mockNativeDataTransfer(),
			dataTransfer;

		pasteMethod = pasteMethod || 'paste';
		nativeData.files = files;
		nativeData.types.push( 'Files' );
		dataTransfer = new CKEDITOR.plugins.clipboard.dataTransfer( nativeData );

		editor.fire( 'paste', {
			dataTransfer: dataTransfer,
			dataValue: ''
		} );
	}
} )();
