// Copyright © Tavian Barnes <tavianator@tavianator.com>
// SPDX-License-Identifier: 0BSD

/**
 * Date/time handling.
 */

#ifndef BFS_XTIME_H
#define BFS_XTIME_H

#include <time.h>

/**
 * localtime_r() wrapper that calls tzset() first.
 *
 * @param[in] timep
 *         The time_t to convert.
 * @param[out] result
 *         Buffer to hold the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xlocaltime(const time_t *timep, struct tm *result);

/**
 * gmtime_r() wrapper that calls tzset() first.
 *
 * @param[in] timep
 *         The time_t to convert.
 * @param[out] result
 *         Buffer to hold the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xgmtime(const time_t *timep, struct tm *result);

/**
 * mktime() wrapper that reports errors more reliably.
 *
 * @param[in,out] tm
 *         The struct tm to convert.
 * @param[out] timep
 *         A pointer to the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xmktime(struct tm *tm, time_t *timep);

/**
 * A portable timegm(), the inverse of gmtime().
 *
 * @param[in,out] tm
 *         The struct tm to convert.
 * @param[out] timep
 *         A pointer to the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xtimegm(struct tm *tm, time_t *timep);

/**
 * Parse an ISO 8601-style timestamp.
 *
 * @param[in] str
 *         The string to parse.
 * @param[out] result
 *         A pointer to the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xgetdate(const char *str, struct timespec *result);

/**
 * Get the current time.
 *
 * @param[out] result
 *         A pointer to the result.
 * @return
 *         0 on success, -1 on failure.
 */
int xgettime(struct timespec *result);

#endif // BFS_XTIME_H
