# Write launcher scripts and menu entries.
#
# This function can take several applications as its arguments,
# and default to handle all applications if none are explicitely given.
#
# USAGE: launchers_generation [$package [$application…]]
launchers_generation() {
	local package
	if [ $# -ge 1 ]; then
		package="$1"
		shift 1
	else
		package=$(current_package)
	fi

	# If no applications are explicitely listed,
	# write launchers and menu entries for all applications.
	if [ "$#" -eq 0 ]; then
		applications_list=$(applications_list)
		## If launchers_generation has been called with no argument, the applications list should not be empty.
		if [ -z "$applications_list" ]; then
			error_applications_list_empty
			return 1
		fi
		launchers_generation "$package" $applications_list
		return 0
	fi

	local application
	for application in "$@"; do
		launcher_write_script "$package" "$application"
		launcher_write_desktop "$package" "$application"
	done
}

# Write the launcher script for the given application.
# USAGE: launcher_write_script $package $application
launcher_write_script() {
	local package application
	package="$1"
	application="$2"

	# Check that the launcher target exists.
	if ! launcher_target_presence_check "$package" "$application"; then
		local application_exe
		application_exe=$(
			set_current_package "$package"
			application_exe "$application"
		)
		## Check that application binary has been found
		if [ -z "$application_exe" ]; then
			error_application_exe_empty "$application" 'launcher_write_script'
			return 1
		fi
		error_launcher_missing_binary "$application_exe"
		return 1
	fi

	# Get application type and prefix type.
	local application_type prefix_type
	application_type=$(
		set_current_package "$package"
		application_type "$application"
	)
	if [ -z "$application_type" ]; then
		error_no_application_type "$application"
		return 1
	fi
	prefix_type=$(
		set_current_package "$package"
		application_prefix_type "$application"
	)

	# For native applications, add execution permissions to the game binary file.
	case "$application_type" in
		('native')
			local application_exe application_exe_path
			application_exe=$(
				set_current_package "$package"
				application_exe "$application"
			)
			## The earlier call to launcher_target_presence_check already ensured that the game binary is found.
			application_exe_path=$(
				set_current_package "$package"
				application_exe_path "$application_exe"
			)
			chmod +x "$application_exe_path"
		;;
	esac

	# Write the launcher script.
	local launcher_path launcher_directory
	debug_write_launcher "$application_type" "$binary_file"
	launcher_path=$(launcher_path "$package" "$application")
	launcher_directory=$(dirname "$launcher_path")
	mkdir --parents "$launcher_directory"
	touch "$launcher_path"
	chmod 755 "$launcher_path"
	## The *_launcher functions are called on their own first, to avoid being spawned in a subshell. This prevents their return code from being lost.
	## The package context is always set to ensure context-sensitive values for runtime options are used.
	local launcher_content
	case "$application_type" in
		('custom')
			launcher_content=$(
				set_current_package "$package"
				custom_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
		;;
		('dosbox')
			launcher_content=$(
				set_current_package "$package"
				dosbox_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'dosbox'
		;;
		('java')
			launcher_content=$(
				set_current_package "$package"
				java_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'java'
		;;
		('mono')
			launcher_content=$(
				set_current_package "$package"
				mono_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'mono'
		;;
		('native')
			launcher_content=$(
				set_current_package "$package"
				native_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
		;;
		('renpy')
			launcher_content=$(
				set_current_package "$package"
				renpy_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'renpy'
		;;
		('scummvm')
			launcher_content=$(
				set_current_package "$package"
				scummvm_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'scummvm'
		;;
		('wine')
			launcher_content=$(
				set_current_package "$package"
				wine_launcher "$application"
			)
			printf '%s' "$launcher_content" | snippet_clean > "$launcher_path"
			dependencies_add_command "$package" 'wine'
			## Add a package dependency on winetricks if the current game relies on some winetricks verbs.
			local winetricks_verbs
			winetricks_verbs=$(wine_winetricks_verbs)
			if [ -n "$winetricks_verbs" ]; then
				dependencies_add_command "$package" 'winetricks'
			fi
			## Add package dependencies on winetricks and rendering libraries if a non-default Direct3D renderer is required.
			local direct3d_renderer
			direct3d_renderer=$(wine_renderer_name)
			case "$direct3d_renderer" in
				('wined3d/'*)
					dependencies_add_command "$package" 'winetricks'
					local wined3d_backend
					wined3d_backend=$(printf '%s' "$direct3d_renderer" | cut --delimiter='/' --fields=2)
					case "$wined3d_backend" in
						('gl')
							dependencies_add_native_libraries "$package" 'libGL.so.1'
						;;
						('vulkan')
							dependencies_add_native_libraries "$package" 'libvulkan.so.1'
						;;
					esac
				;;
				('dxvk')
					dependencies_add_command "$package" 'winetricks'
					dependencies_add_native_libraries "$package" 'libvulkan.so.1'
				;;
				('vkd3d')
					dependencies_add_command "$package" 'winetricks'
					dependencies_add_native_libraries "$package" 'libvulkan.so.1'
				;;
			esac
		;;
	esac
}

# Check that the launcher target exists.
# USAGE: launcher_target_presence_check $package $application
# RETURN: 0 if the game binary has been found,
#         0 if no game binary is expected,
#         1 if the game binary has not been found
launcher_target_presence_check() {
	local package application
	package="$1"
	application="$2"

	local application_type application_type_status
	application_type=$(
		set_current_package "$package"
		application_type "$application"
	)
	application_type_status=$?
	## Exit early if fetching the application type failed.
	if [ $application_type_status -ne 0 ]; then
		## Use exit instead of return to ensure the script execution ends here,
		## return 1 would not stop it because this function is called through test.
		exit 1
	fi
	case "$application_type" in
		('custom')
			## Custom launchers might not rely on a provided binary.
			return 0
		;;
		('renpy')
			## Ren'Py games do not rely on a provided binary.
			return 0
		;;
		('scummvm')
			## ScummVM games do not rely on a provided binary.
			return 0
		;;
	esac

	local application_exe application_exe_path
	application_exe=$(
		set_current_package "$package"
		application_exe "$application"
	)
	## Check that application binary has been found
	if [ -z "$application_exe" ]; then
		error_application_exe_empty "$application" 'launcher_target_presence_check'
		## Use exit instead of return to ensure the script execution ends here,
		## return 1 would not stop it because this function is called through test.
		exit 1
	fi
	application_exe_path=$(
		set_current_package "$package"
		application_exe_path "$application_exe"
	)
	test -f "$application_exe_path"
}

# Print the path to the launcher script for the given application.
# USAGE: launcher_path $package $application
# RETURN: The absolute path to the launcher
launcher_path() {
	local package application
	package="$1"
	application="$2"

	local package_path path_binaries application_id
	package_path=$(package_path "$package")
	path_binaries=$(path_binaries)
	application_id=$(
		set_current_package "$package"
		application_id "$application"
	)

	printf '%s%s/%s' "$package_path" "$path_binaries" "$application_id"
}

# Print the headers common to all launcher scripts
# USAGE: launcher_headers
launcher_headers() {
	cat <<- EOF
	#!/bin/sh
	# script generated by ./play.it $LIBRARY_VERSION - https://www.dotslashplay.it/
	set -o errexit

	EOF
}

# Print the exit actions common to all launcher scripts
# USAGE: launcher_exit
launcher_exit() {
	cat <<- 'EOF'
	# Return the game exit code

	if [ -n "$game_exit_status" ]; then
	    exit "$game_exit_status"
	else
	    exit 0
	fi
	EOF
}

# Print the line starting the game
# USAGE: game_exec_line $application
# RETURN: the command to execute, including its command line options
game_exec_line() {
	local application
	application="$1"

	local application_type
	application_type=$(application_type "$application")
	case "$application_type" in
		('dosbox')
			dosbox_exec_line "$application"
		;;
		('java')
			java_exec_line "$application"
		;;
		('mono')
			mono_exec_line "$application"
		;;
		('native')
			native_exec_line "$application"
		;;
		('renpy')
			renpy_exec_line "$application"
		;;
		('scummvm')
			scummvm_exec_line "$application"
		;;
		('wine')
			wine_exec_line "$application"
		;;
	esac
}

